/*
 * unity-webapps-indicator-manager.c
 * Copyright (C) Canonical LTD 2011
 *
 * Author: Robert Carr <racarr@canonical.com>
 * 
 unity-webapps is free software: you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published
 * by the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * unity-webapps is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.";
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <glib/gstdio.h>
#include <gio/gio.h>

#include "unity-webapps-indicator-model.h"
#include "unity-webapps-indicator-model-controller.h"
#include "unity-webapps-indicator-view-messaging-menu.h"

#include "unity-webapps-indicator-manager.h"

#include "../unity-webapps-debug.h"

struct _UnityWebappsIndicatorManagerPrivate {
  UnityWebappsIndicatorModel  *model;
  UnityWebappsIndicatorView *view;
  UnityWebappsIndicatorModelController *controller;
  
  UnityWebappsInterestManager *interest_manager;
  
  gchar *desktop_file;
};

enum {
  PROP_0,
  PROP_DESKTOP_FILE,
  PROP_INTEREST_MANAGER
};

enum {
  INDICATOR_ACTIVATED,
  APPLICATION_ACTIVATED,
  LAST_SIGNAL
};

static guint signals[LAST_SIGNAL] = { 0 };

G_DEFINE_TYPE(UnityWebappsIndicatorManager, unity_webapps_indicator_manager, G_TYPE_OBJECT)

#define UNITY_WEBAPPS_INDICATOR_MANAGER_GET_PRIVATE(object) (G_TYPE_INSTANCE_GET_PRIVATE ((object), UNITY_WEBAPPS_TYPE_INDICATOR_MANAGER, UnityWebappsIndicatorManagerPrivate))

static void
unity_webapps_indicator_manager_finalize (GObject *object)
{
  UnityWebappsIndicatorManager *manager;
  
  manager = UNITY_WEBAPPS_INDICATOR_MANAGER (object);
  
  g_free (manager->priv->desktop_file);
}

static void
unity_webapps_indicator_manager_dispose (GObject *object)
{
  UnityWebappsIndicatorManager *manager;
  
  manager = UNITY_WEBAPPS_INDICATOR_MANAGER (object);
  
  if (manager->priv->model)
    {
      g_object_unref (G_OBJECT (manager->priv->model));
      manager->priv->model = NULL;
    }

  if (manager->priv->view)
    {
      g_object_unref (G_OBJECT (manager->priv->view));
      manager->priv->view = NULL;
    }

  if (manager->priv->controller)
    {
      g_object_unref (G_OBJECT (manager->priv->controller));
      manager->priv->controller = NULL;
    }
  
  if (manager->priv->interest_manager)
    {
      g_object_unref (G_OBJECT (manager->priv->interest_manager));
      manager->priv->interest_manager = NULL;
    }
}

static void
unity_webapps_indicator_manager_server_raised (UnityWebappsIndicatorView *view,
					       gpointer user_data)
{
  UnityWebappsIndicatorManager *manager;
  
  manager = (UnityWebappsIndicatorManager *)user_data;
  
  g_signal_emit (manager, signals[APPLICATION_ACTIVATED], 0);
}

static void
unity_webapps_indicator_manager_indicator_raised (UnityWebappsIndicatorView *view,
						  const gchar *name,
						  gpointer user_data)
{
  UnityWebappsIndicatorManager *manager;
  
  manager = (UnityWebappsIndicatorManager *)user_data;
  
  g_signal_emit (manager, signals[INDICATOR_ACTIVATED], 0, name);
}

static void
unity_webapps_indicator_manager_create_view (UnityWebappsIndicatorManager *self)
{
  self->priv->view = unity_webapps_indicator_view_messaging_menu_new (self->priv->model,
                                                                      self->priv->desktop_file);
  
  g_signal_connect_object (self->priv->view,
			   "server-raised",
			   G_CALLBACK (unity_webapps_indicator_manager_server_raised),
			   self, 0);

  g_signal_connect_object (self->priv->view,
			   "indicator-raised",
			   G_CALLBACK (unity_webapps_indicator_manager_indicator_raised),
			   self, 0);

}

static void
unity_webapps_indicator_manager_get_property (GObject *object,
					      guint prop_id,
					      GValue *value,
					      GParamSpec *pspec)
{
  UnityWebappsIndicatorManager *manager;
  
  manager = UNITY_WEBAPPS_INDICATOR_MANAGER (object);
  
  switch (prop_id)
    {
    case PROP_DESKTOP_FILE:
      g_value_set_string (value, manager->priv->desktop_file);
      break;
    case PROP_INTEREST_MANAGER:
      g_value_set_object (value, manager->priv->interest_manager);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
    }
}

static void
unity_webapps_indicator_manager_create_controller (UnityWebappsIndicatorManager *manager)
{
  manager->priv->controller = unity_webapps_indicator_model_controller_new (manager->priv->model,
									    manager->priv->interest_manager);

}

static void
unity_webapps_indicator_manager_set_property (GObject *object,
					      guint prop_id,
					      const GValue *value,
					      GParamSpec *pspec)
{
  UnityWebappsIndicatorManager *manager;
  
  manager = UNITY_WEBAPPS_INDICATOR_MANAGER (object);
  
  switch (prop_id)
    {
    case PROP_DESKTOP_FILE:
      g_return_if_fail (manager->priv->desktop_file == NULL);
      manager->priv->desktop_file = g_value_dup_string (value);

      unity_webapps_indicator_manager_create_view (manager);
      break;
    case PROP_INTEREST_MANAGER:
      g_return_if_fail (manager->priv->interest_manager == NULL);
      manager->priv->interest_manager = UNITY_WEBAPPS_INTEREST_MANAGER (g_value_dup_object (value));
      
      unity_webapps_indicator_manager_create_controller (manager);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
    }
}

static void
unity_webapps_indicator_manager_class_init (UnityWebappsIndicatorManagerClass *klass)
{
  GObjectClass *object_class = G_OBJECT_CLASS (klass);
  
  object_class->finalize = unity_webapps_indicator_manager_finalize;
  object_class->dispose = unity_webapps_indicator_manager_dispose;
  object_class->get_property = unity_webapps_indicator_manager_get_property;
  object_class->set_property = unity_webapps_indicator_manager_set_property;
  
  g_type_class_add_private (object_class, sizeof(UnityWebappsIndicatorManagerPrivate));
  
  g_object_class_install_property (object_class, PROP_DESKTOP_FILE,
				   g_param_spec_string ("desktop-file",
							"Desktop File",
							"Full path to the desktop file to use for the indicator view",
							NULL,
							G_PARAM_READWRITE |
							G_PARAM_STATIC_STRINGS |
							G_PARAM_CONSTRUCT_ONLY));
  
  g_object_class_install_property (object_class, PROP_INTEREST_MANAGER,
				   g_param_spec_object ("interest-manager",
							"Interest Manager",
							"Interest manager to use to track indicator references",
							UNITY_WEBAPPS_TYPE_INTEREST_MANAGER,
							G_PARAM_READWRITE |
							G_PARAM_STATIC_STRINGS |
							G_PARAM_CONSTRUCT_ONLY));
  
  signals[INDICATOR_ACTIVATED] = g_signal_new ("indicator-activated",
					       UNITY_WEBAPPS_TYPE_INDICATOR_MANAGER,
					       G_SIGNAL_RUN_LAST,
					       0,
					       NULL,
					       NULL,
					       NULL,
					       G_TYPE_NONE,
					       1,
					       G_TYPE_STRING);
  signals[APPLICATION_ACTIVATED] = g_signal_new ("application-activated",
						 UNITY_WEBAPPS_TYPE_INDICATOR_MANAGER,
						 G_SIGNAL_RUN_LAST,
						 0,
						 NULL,
						 NULL,
						 NULL,
						 G_TYPE_NONE,
						 0);

}

static void
unity_webapps_indicator_manager_init (UnityWebappsIndicatorManager *manager)
{
  manager->priv = UNITY_WEBAPPS_INDICATOR_MANAGER_GET_PRIVATE (manager);
  
  manager->priv->model = unity_webapps_indicator_model_new ();
}

UnityWebappsIndicatorManager *
unity_webapps_indicator_manager_new (const gchar *desktop_file, UnityWebappsInterestManager *interest_manager)
{
  return g_object_new (UNITY_WEBAPPS_TYPE_INDICATOR_MANAGER, 
		       "desktop-file", desktop_file,
		       "interest-manager", interest_manager,
		       NULL);
}

UnityWebappsIndicatorModel *
unity_webapps_indicator_manager_get_model (UnityWebappsIndicatorManager *manager)
{
  g_return_val_if_fail (manager, NULL);
  
  return manager->priv->model;
}

UnityWebappsIndicatorView *
unity_webapps_indicator_manager_get_view (UnityWebappsIndicatorManager *manager)
{
  g_return_val_if_fail (manager, NULL);
  
  return manager->priv->view;
}
