/*
 * Isomorphic SmartClient
 * Version SC_SNAPSHOT-2010-12-31 (2010-12-31)
 * Copyright(c) 1998 and beyond Isomorphic Software, Inc. All rights reserved.
 * "SmartClient" is a trademark of Isomorphic Software, Inc.
 *
 * licensing@smartclient.com
 *
 * http://smartclient.com/license
 */
//> @class AutoTest
// Standalone class providing a general interface for integration with Automated Testing Tools
// <p>
// For automated testing tools we need a way to create string identifiers for DOM elements such that 
// when a page is reloaded, we can retrieve a functionally equivalent DOM element. We call these
// +link{AutoTestLocator,autoTestLocators}.
// <p>
// This allows automated testing tools to set up or record user generated events on DOM elements
// then play them back on page reload and have our components react correctly.
// <P>
// The primary APIs for the AutoTest subsystem are +link{AutoTest.getLocator()} and 
// +link{AutoTest.getElement()}.
// <P>
// 
// Implementation considerations:
// <ul>
// <li> Some components react to the structure of DOM elements embedded within them - for example
//   GridRenderer cells have meaning to the grid. So in some cases we need to identify elements
//   within a component, while in others we can just return a pointer to a handle (A simple
//   canvas click handler doesn't care about what native DOM element within the  handle received
//   the click).
//
// <li>When a DOM element is contained by a component, it is not sufficient to store the component
//   ID. Most SmartClient components are auto-generated by their parents, and rather than 
//   attempting to store a specific component identifier we should instead store the
//   "logical function" of the component.<br>
//   For example a listGrid header button may have a different auto-generated ID across page
//   reloads due to various timing-related issues (which can change the order of of widget
//   creation), loading a new skin, or otherwise trivial changes to an application.<br>
//   Rather than storing the header button ID therefore, we want to store this as
//   a string meaning "The header button representing field X within this list grid".
//
// <li>fallback strategies: In some cases a component or DOM element can be identified in 
//   several ways. For example a cell in a ListGrid can be identified by simple row and
//   column index, but also by fieldName and record primary key value. In these cases we
//   attempt to record information for multiple locator strategies and then when parsing
//   stored values we can provide APIs to govern which strategy is preferred. See the
//   +link{type:LocatorStrategy} documentation for more on this.
// </ul>
// 
// In order to address these concerns the AutoTest locator pattern is similar to an
// XPath type structure, containing a root component identifier, followed by 
// details of sub-components and then potentially details used to identify an element within
// the components handle in the DOM.
// <br>
// The actual implementation covers a large number of common cases, including (but not limited to)
// the following. Note that for cases where an element is being identified from a pool of
// possible candidates, such as the +link{canvas.children} array, we usually will use
// +link{LocatorStrategy,fallback locator paths} rather than simply relying on index:
// <ul><li>Root level components identified by explicit ID</li>
//     <li>Any +link{autoChild,autoChildren}</li>
//     <li>Standard component parts such as scrollbars, edges, shadows, etc</li>
//     <li>Section stack items and headers</li>
//     <li>Window items</li>
//     <li>ListGrid headers and cells</li>
//     <li>TreeGrid headers and cells, including interactive open icon, checkbox icons</li>
//     <li>DynamicForm form items, including details of elements within those items</li>
// </ul>
//
// AutoTest
// @visibility external
// @group autoTest
//<

//> @type AutoTestLocator
// An autoTestLocator is an xpath-like string used by the AutoTest subsystem to robustly 
// identify DOM elements within a SmartClient application.
// <P>
// Typically AutoTestLocators will not be hand-written - they should be retrieved by a
// call to +link{AutoTest.getLocator()}. Note also that the +link{debugging,Developer Console}
// has built-in functionality to create and display autoTestLocators for a live app.
//
// @group autoTest
// @visibility external
//<



isc.defineClass("AutoTest");


isc.AutoTest.addClassMethods({
    
    //> @classMethod AutoTest.getLocator()
    // Returns the +link{type:Locator} associated with some DOM element in a SmartClient
    // application page.
    // @param DOMElement (DOMElement) DOM element within in the page. If null the locator for the last
    //  mouse event target will be generated
    // @param [checkForNativeHandling] (boolean) If this parameter is passed in, check whether
    //  the target element responds to native browser events directly rather than going through
    //  the SmartClient widget/event handling model. If we detect this case, return null rather
    //  than a live locator.  This allows us to differentiate between (for example) an event on
    //  a Canvas handle, and an event occurring directly on a simple <code>&lt;a href=...&gt;</code>
    //  tag written inside a Canvas handle.
    // @return (AutoTestLocator) Locator string allowing the AutoTest subsystem to find
    //   an equivalent DOM element on subsequent page loads.
    // @visibility external
    // @group autoTest
    //<
    
    getLocator : function (DOMElement, checkForNativeHandling) {
        var fromEvent;
        if (DOMElement == null) {
            fromEvent = true;
            DOMElement = isc.EH.lastEvent ? isc.EH.lastEvent.nativeTarget : null;
        }
        var canvas;
        if (isc.isA.Canvas(DOMElement)) canvas = DOMElement;
        else {
            canvas = isc.AutoTest.locateCanvasFromDOMElement(DOMElement);            
        }
        
        var locator = canvas ? canvas.getLocator(DOMElement, fromEvent) : "";
        
        
        
        if (checkForNativeHandling && locator && locator != "" &&
            canvas.checkLocatorForNativeElement(locator, DOMElement)) 
        {
            locator = "";
        }
        return locator;

    },
    
    
    //> @classMethod AutoTest.locateCanvasFromDOMElement()
    // Given an element in the DOM, returns the canvas containing this element, or null if
    // the element is not contained in any canvas handle.
    // @visibility external
    // @group autoTest
    //<
    locateCanvasFromDOMElement : function (element) {
        
        return isc.EH.getEventTargetCanvas(null, element);
    },
    
        
    // ------------------------------
    // Retrieving elements from the DOM based on locator string
    //> @classMethod AutoTest.getElement()
    // @param (AutoTestLocator) Locator String previously returned by +link{AutoTest.getLocator()}
    // @return (DOMElement) DOM element this locator refers to in the running application, or
    // null if not found
    // @visibility external
    // @group autoTest
    //<
    
    getElement : function (locator) {
        if (!locator) return null;

        // trim off quote chars from the start/end of the string
        
        if (locator.startsWith("'") || locator.startsWith('"')) locator = locator.substring(1);
        if (locator.endsWith("'") || locator.endsWith('"')) locator = locator.substring(0,locator.length-1);
        
        if (!locator.startsWith("//")) {
            // assume either just an ID or "ID=[ID]"
            if (locator.startsWith("ID=") || locator.startsWith("id=")) {
                locator = locator.substring(3);
            }
            locator = '//*any*[ID="' + locator + '"]';
        }

        var locatorArray = locator.split("/"),
            component;
            
        // account for the 2 slashes
        var baseComponentID = locatorArray[2];
        if (!baseComponentID) return null;
      
        // knock off the first 3 slots
        locatorArray = locatorArray.slice(3);     
        
        var baseComponent = this.getBaseComponentFromLocatorSubstring(baseComponentID);
        var elem = baseComponent ? baseComponent.getElementFromSplitLocator(locatorArray) : null;
        //this.logWarn("element :" + elem);
        return elem
    },
    
    
    getPageCoords : function (locator) {
        var element = this.getElement(locator);
        if (element == null) return;
        
        var canvas = this.locateCanvasFromDOMElement(element);
        return canvas ? canvas.getAutoTestLocatorCoords(locator, element) : null;
    },

    
    // getBaseComponentFromLocatorSubstring: This actually gets the *base* component from
    // a locator substring.
    // 2 possibilities:
    // - explicit ID (respect that)
    // - part of the array of top-level canvii
    getBaseComponentFromLocatorSubstring : function (substring) {
        var IDMatches = substring.match("(.*)\\[");
        var IDType = IDMatches ? IDMatches[1] : null;
        // if the recorded canvas had an auto-generated ID, try to find it by looking in the
        // top level (no parent) canvas array.
        // We'll look by name, title, then index by class, scClass and role!
        if (IDType == "autoID") {
            
            var config = isc.AutoTest.parseLocatorFallbackPath(substring),
                widgetConfig = config.config,
                strategy = "name",
                typeStrategy = "Class";
            return isc.Canvas.getCanvasFromFallbackLocator(
                        substring, widgetConfig, isc.Canvas._topCanvii, strategy, typeStrategy)
            
        } else {
        
            var className = IDType, 
                IDMatches = substring.match('ID=[\\"\'](.*)[\'\\"]'),
                ID = IDMatches ? IDMatches[1] : null;
//            this.logWarn("className/ID:" + [className,ID]);
            if (ID == null) return null;
            var baseComponent = window[ID];
            if (!baseComponent) return null;
            if (baseComponent && className != "*any*" &&
                (!isc.isA[className] || !isc.isA[className](baseComponent))) 
            {
                this.logWarn("AutoTest.getElement(): Component:"+ baseComponent + 
                            " expected to be of class:" + className);
            }
            return baseComponent;
        }
    },
    
     // Retrieving SC objects from locator string
    //> @classMethod AutoTest.getLocatorCanvas()
    // Returns the Canvas for some previously generated locator string.
    // @param (Locator) Locator String previously returned by +link{AutoTest.getLocator()}
    // @return (Canvas) Canvas associated with this locator
    // @visibility internal
    // @group autoTest
    //<
    getLocatorCanvas : function (locator) {
        
                
        // Simply get the DOM element and pick up the Canvas from it.
        // XXX this will not work if the Canvas is currently undrawn.
        /*
        var DOMElement = this.getElement(locator);
        if (DOMElement != null) {
            return this.locateCanvasFromDOMElement(DOMElement);
        }
        return null;
        */
        
        if (locator == null || isc.isAn.emptyString(locator)) return null;
        var locatorArray = locator.split("/"),
            component;
        //this.logWarn("locatorArray" + locatorArray);
        // account for the 2 slashes
        var baseComponentID = locatorArray[2];
        
        // knock off the first 3 slots
        var length = locatorArray.length;
        for (var i = 3; i < length; i++) {
            locatorArray[i-3] = locatorArray[i];
        }
        locatorArray.length = length-3;
        if (!baseComponentID) return null;
        
        var baseComponent = this.getBaseComponentFromLocatorSubstring(baseComponentID);
        if (baseComponent) {
            var i = 0,
                child = baseComponent.getChildFromLocatorSubstring(locatorArray[i], i, locatorArray);
            while (child != null) {
                i++;
                baseComponent = child;
                child = baseComponent.getChildFromLocatorSubstring(locatorArray[i], i, locatorArray);
            }
            return baseComponent;
        }
        return null;
    },
    
    //> @classMethod AutoTest.getLocatorFormItem()
    // Returns the FormItem for some previously generated locator string, or null if no
    // matching FormItem can be found.
    // @param (Locator) Locator String previously returned by +link{AutoTest.getLocator()}
    // @return (Canvas) Canvas associated with this locator
    // @visibility autoTest
    //<
    getLocatorFormItem : function (locator) {
        // Simply get the DOM element and pick up the DynamicForm/ FormItem from it.
        // XXX this will not work if the Canvas is currently undrawn.
        /*
        var DOMElement = this.getElement(locator);
        if (DOMElement != null) {
            var form = this.locateCanvasFromDOMElement(DOMElement);
            if (isc.isA.DynamicForm(form)) {
                var itemInfo = isc.DynamicForm._getItemInfoFromElement(DOMElement,form);
                if (itemInfo) return itemInfo.item;
            }
        }
        return null;
        */
            
        if (locator == null || isc.isAn.emptyString(locator)) return null;
        var locatorArray = locator.split("/"),
            component;
        // account for the 2 slashes
        var baseComponentID = locatorArray[2];
        
        // knock off the first 3 slots
        var length = locatorArray.length;
        for (var i = 3; i < length; i++) {
            locatorArray[i-3] = locatorArray[i];
        }
        locatorArray.length = length-3;
        if (!baseComponentID) return null;
        
        var baseComponent = this.getBaseComponentFromLocatorSubstring(baseComponentID);
        if (baseComponent) {
            
            var child = baseComponent.getChildFromLocatorSubstring(locatorArray[0], 0, locatorArray);
            while (child != null) {
                locatorArray.removeAt(0);
                baseComponent = child;
                child = baseComponent.getChildFromLocatorSubstring(locatorArray[0], 0, locatorArray);
            }
        }
        if (isc.isA.DynamicForm(baseComponent)) {
            return baseComponent.getItemFromSplitLocator(locatorArray);
        }
        return null;
    },
    
    // Fallback locator subsystem:
    // For cases where there is more than one possible way to identify a component or element
    // we generate a string similar to this:
    // "row[a=b||b=c||7]"
    
    // createLocatorFallbackPath()
    // Takes a locator name and an object of the format:
    //   {fieldName:value, fieldName:value}
    // and returns a string in the format
    //   name[fieldName=value||fieldName=value...]
    // standalone values (with no "=" may also be included -- to do this set the "fallback_valueOnlyField"
    // property on the object passed in
    // For example:
    //   var identifier = {a:"b"};
    //   identifier[isc.AutoTest.fallback_valueOnlyField] = "c";
    //   isc.AutoTest.createLocatorFallbackPath("test", identifier);
    // would give back:
    //   "test[a:b||c]"
    
    fallback_valueOnlyField:"_$_standaloneProperty",
    
    fallback_startMarker:"[",
    fallback_endMarker:"]",
    fallback_separator:"||",
    fallback_equalMarker:"=",
    
    // If a property name contains the "/" character we can't store it as we use
    // simple string.split to break up based on this char.
    // Fix this by just sub-ing in a customizable marker when generating locators.
    
    slashMarker:"$fs$",
    
    createLocatorFallbackPath : function (name, config) {
        
        var locator = [];
        
        for (var field in config) {
            var fieldVal = config[field];
            
            // If a string contains "[", "||", etc we can get very confused
            // use 'escape' to HTML encode the string -- we'll unescape when parsing
            // We have to escape actual slashes too as this will break our logic to 
            // break up stored locators.
            // use a regex to just replace them with a customizeable marker
            if (isc.isA.String(fieldVal)) {
                fieldVal = fieldVal.replace("/",this.slashMarker);
                fieldVal = escape(fieldVal);
            }
            
            // Not worrying about other data types for now
            // Numbers / bools will convert automatically
            // If it becomes necessary we could encode dates, arr's, objects etc
            // and unencode on the way back
                
            if (field == this.fallback_valueOnlyField) {
                locator.add(fieldVal);
            } else {
                locator.add(field + this.fallback_equalMarker + fieldVal);
            }
        }
        return name + this.fallback_startMarker + locator.join(this.fallback_separator) + 
                    this.fallback_endMarker;
    },
    
    // This method will take a generated locatorFallbackPath string and return a
    // standard config object as described above - property/field values will be unmapped
    // and any standalone value will be stored under the special
    // isc.AutoTest.fallback_valueOnlyField attribute name.
    parseLocatorFallbackPath : function (path) {
        var pathArr = path.split(this.fallback_startMarker);
        // don't crash if we were passed something we don't understand...
        if (pathArr == null || pathArr.length < 2) return;
        
        var name = pathArr[0],
            path = pathArr[1].substring(0, pathArr[1].length-this.fallback_endMarker.length);
            
        var configArr = path.split(this.fallback_separator),
            configObj = {};
        for (var i = 0; i < configArr.length; i++) {
            var string = configArr[i],
                equalsIndex = string.indexOf(this.fallback_equalMarker),
                fieldName;
                
            if (equalsIndex == -1) {
                fieldName = this.fallback_valueOnlyField;
            } else {
                fieldName = string.substring(0,equalsIndex);
                string = string.substring(equalsIndex+1);
            }
            
            // always unescape
            string = string.replace(this.slashMarker, "/");
            string = unescape(string);
            configObj[fieldName] = string;
            
        }
        
        // BackCompat: Standard locator format (pre March 2010) was always of the format
        // item[1][Class="Canvas"]
        // This is still used where we don't run through the fallback-path subsystem but is
        // being incrementally replaced.
        // If we're passed a string of that format, pull the class out of the string passed in
        // and attach it to the config object.
        // This means that for any old auto-test recordings with the previous identifier format
        // if they end up running through this subsystem we should still have predictable results
        if (pathArr[2] != null) {
            var string = pathArr[2].substring(0, pathArr[2].length-this.fallback_endMarker.length),
                equalsIndex = string.indexOf(this.fallback_equalMarker),
                
                key = string.substring(0,equalsIndex),
                val = string.substring(equalsIndex+1);
            // if the string was quoted, eat the quotes!
            if (val.startsWith("\"")) val = val.substring(1, val.length-1);
            
            configObj[key] = val; 
        }
        
        return {name:name, config:configObj};
    },
    
    
    
    // Generate a standard object "locator fallback path" identifier from an object,
    // similar to:
    //  member[title="foo"||index=1||Class="ImgButton"]
    //
    // Parameters:
    // - name attribute specifies the identifier type (in this example "member")
    // - canvas is the object to get an identifier for
    // - properties is an object specifying some default identifier properties to use which
    //   cannot be directly retrieved from the object. Typically used to specify the
    //   index of the object in the named array.
    // - mask is an object or array specifying properties to include in the locator string.
    //   If an array of strings, for each element store the same-named attribute from the object
    //   on the locator string
    //   If an object, for each entry, pick up the value field from the object and store it
    //   under the key on the locator string
    // * When getting properties from the object, use getters if present
    // * if AutoTest.fallback_valueOnlyField is included this will be included in the 
    //   locator string with no key - for example
    //   member[1]
    //
    
    getObjectLocatorFallbackPath : function (name, object, properties, mask) {
        
        if (properties == null) properties = {};
      
        if (mask == null) mask = {
            title:"title",
            // we do this because widget.getClass() gives us the class object whereas
            // widget.getClassName gives us the name of the smartclient class...
            Class:"ClassName"
        };
        
        if (isc.isAn.Array(mask)) {
                
            for (var i = 0; i < mask.length; i++) {
                var value = object.getProperty ? object.getProperty(mask[i]) : object[mask[i]];
                if (value != null && !isc.isAn.emptyString(value)) properties[mask[i]] = value;
            }
        } else {
            for (var field in mask) {
                var value = object.getProperty ? object.getProperty(mask[field]) : object[mask[field]];
                if (value != null && !isc.isAn.emptyString(value)) properties[field] = value;
            }
        }
        
        // This will turn that config object into a standard locator type string.
        return isc.AutoTest.createLocatorFallbackPath(name, properties);
    },
    
    
    // Auto Test locators use various strategies to attempt to locate widgets. In some cases
    // we return a "best guess" type locator string -- for example an index in the members array
    // of a layout -- this is prone to return the wrong element if the page is restructured.
    // When actually retrieving elements from the DOM, we have some hints as to the fact that
    // our locator may be returning the wrong thing -- number of matching elements has changed
    // might be one of them, or the role / class of the widget we think matches is different
    // from what we recorded.
    // In these cases we'll log a warning.
    // This is a generic warning text which we can append to these warnings about how to
    // make identifying more robust in the future
    robustLocatorWarning:"If you are seeing unexpected results in recorded tests, it is likely" +
    " that the application has been modified since the test was recorded. We would recommend re-recording" +
    " your test script with the latest version of your application. Note that you may be able to" +
    " avoid seeing this message in future by using the AutoChild subsystem or providing explicit" +
    " global IDs to components whose function within the page is unlikely to change.",
    logRobustLocatorWarning : function () {
        if (this._loggedWarning) return;
        this.logWarn(this.robustLocatorWarning, "AutoTest");
        this._loggedWarning = true;
    }
        

});

isc.ApplyAutoTestMethods = function () {


isc.Canvas.addClassMethods({
    // use fallback strategies to get at the right object from a stored path.
    getCanvasLocatorFallbackPath : function (name, canvas, sourceArray, properties, mask) {
        
       if (properties == null) properties = {};
        
        if (mask == null) mask = {};
        else if (isc.isAn.Array(mask)) {
            var maskObj = {};
            for (var i = 0; i <mask.length; i++) {
                maskObj[mask[i]] = mask[i];
            }
            mask = maskObj;
        }
        
        // Always pick up the following attributes directly from the widget, if present
        if (mask.title == null) mask.title = "title";
        if (mask.scRole == null) mask.scRole = "waiRole";
        if (mask.name == null) mask.name = "name";
        
        // ClassName / scClassName - this is more complex than just looking at attributes on
        // the widget:
        // We need to pick up the class name, and if that's not a core smartclient class, also
        // pick up the core superclass of that class so we can look at both
        var objectClassName = canvas.getClassName(),
            objectClass = canvas.getClass();
        
        properties.Class = objectClassName;
        
        var scClassName;
        if (!objectClass.isFrameworkClass) {
            scClassName = objectClass._scClassName;
        }
        if (scClassName != null) properties.scClass = scClassName;
        
        
        // We also want to pick up index-based locators from the source array
        // Record both the index and the current length
        // Locating by index is always imperfect: If a developer changes the orders of
        // members (for example), it'll break.
        // However if the length is different when a recorded locator is parsed, we have
        // a really good indication that the index based locator is probably unreliable.
        if (sourceArray != null) {
            
            // Raw position in the array
            properties.index = sourceArray.indexOf(canvas);
            properties.length = sourceArray.length;

            // position within widgets of this class in the array
            // Use case: the developer adds something like a 'status label' at the top
            // of an array of buttons
            var matchingClass = sourceArray.findAll("Class", objectClassName);
            properties.classIndex = matchingClass.indexOf(canvas);
            properties.classLength = matchingClass.length;

            // position within widgets of this SmartClient class in the array
            // Use case: The developer subclasses a SmartClient component as the app matures
            // but the application layout stays the same, so an array of buttons becomes
            // an array of custom button subclasses
            if (scClassName != null) {
                var matchingSCClass = sourceArray.findAll("_scClass", scClassName);
                properties.scClassIndex = matchingSCClass.indexOf(canvas);
                properties.scClassLength = matchingSCClass.length;
            }
            
            // Position within widgets with this role in the warray
            // Use case: The smart client class changes due to (say) reskinning (moving from
            // a button to a stretchImgButton), but the role is unchanged
            if (canvas.waiRole != null) {
                var matchingRoles = sourceArray.findAll("waiRole", canvas.waiRole);
                properties.roleIndex = matchingRoles.indexOf(canvas);
                properties.roleLength = matchingRoles.length;
            }
        }
        
        return isc.AutoTest.getObjectLocatorFallbackPath(name, canvas, properties, mask);
        
    },
    
    
    // substring param really just used for logging
    getCanvasFromFallbackLocator : function (substring, config, candidates, strategy, typeStrategy) {
        
        // Given an array of possible candidates attempt to match as follows:
        
        // - if a 'name' was recorded,
        //  - match by name and class name
        //  - otherwise by name and scClassName
        //  - otherwise by name and scRole
        // - if a title was recorded
        //  - match by title  and class name
        //  - title / scClassName
        //  - title / role
        //
        // Otherwise back off to matching by index:
        //  - try to match by class name / index (of candidates with that className)
        //  - then by scClassName / index
        //  - then by role / index
        //  - then by raw index
        
        // Robustness:
        // We have a big one-time warning to log when we think what we're returning is
        // likely unreliable (see AutoTest.logRobustLocatorWarning())
        // We do this:
        //  - if we find a match by name but it doesn't match class, scclass or role
        //  - if we find a match by title but it doesn't match class scclass or role
        //      (If there is more than one match by title we ignore this strategy and back
        //       off to index with a different warning)
        //  - if, when attempting to find a match by index (by class scClass or role, or by
        //    raw index), we find the array length has changed (meaning the array has
        //    changed, so the index is probably worthless).
        //
        // We also log a less "things are broken" warning everytime we return
        // by raw index as this is very fragile.
        var name = config.name;
        
        // Some common things we're always going to try:
        var className = config.Class,
            // scClass will not have been recorded separately if the recorded class
            // is already a core class.
            scClassName = config.scClass || config.Class,
            role = config.scRole;
     
        
        switch (strategy) {
            
        case "name":
                            
            if (name != null) {
                var nameMatch = candidates.find("name", name);
                // we could check uniqueness as we do with title, but this seems very unlikely
                // to be necessary - name is only really used as a unique ID within some
                // array (though not globally unique)
                
                if (nameMatch) {
                    
                    switch (typeStrategy) {
                        
                    case "Class": // scClass // role // none
                    
                      
                        if (className && isc.isA[className] && isc.isA[className](nameMatch)) {
                            if (this.logIsDebugEnabled("AutoTest")) {
                                this.logDebug("Locator string:" + substring + 
                                    " - returning widget with matching name and ClassName:" +
                                    nameMatch, "AutoTest");
                            }
                            return nameMatch;
                        }
                        
                    case "scClass":
                        
                        if (scClassName && isc.isA[scClassName] && isc.isA[scClassName](nameMatch)) 
                        {
                            if (this.logIsDebugEnabled("AutoTest")) {
                                this.logDebug("Locator string:" + substring + 
                                    " - returning widget with matching name and scClassName:" +
                                    nameMatch, "AutoTest");
                            }
                            return nameMatch;
                        }
                        
                    case "role":
                    
                        // If the classes don't match - see if the roles match
                        var scRole = config.scRole;
                        if (nameMatch.waiRole == scRole) {
                            if (this.logIsDebugEnabled("AutoTest")) {
                                this.logDebug("Locator string:" + substring + 
                                    " - returning widget with matching name and role:" +
                                    nameMatch, "AutoTest");
                            }
                            return nameMatch;
                        }
                    
                    default:
                        
                        // In this case we've got a matching name but we can't match it to
                        // class or role. This is still the most likely candidate (better than
                        // backing off to checking index), so log a warning and return it:
                        
                        if (typeStrategy != "none") {
                            isc.AutoTest.logRobustLocatorWarning();
                            this.logWarn("Locator string:" + substring + 
                                ". Returning closest match:" + nameMatch + ". This has the same name " +
                                "as the recorded component but does not match class or role. ", "AutoTest");
                        } else {
                             if (this.logIsDebugEnabled("AutoTest")) {
                                this.logDebug("Locator string:" + substring + 
                                    " - returning widget with matching name:" +
                                    nameMatch, "AutoTest");
                            }
                        }
                            
                        return nameMatch;
                    }
                }
            }
            
            
            
        case "title":
            var title = config.title;
            if (title != null) {
                var titleMatches = candidates.findAll("title", title);
                
                if (titleMatches && titleMatches.length > 0) {
                    var titleMatch;
                    
                    
                    switch (typeStrategy) {
                        
                    case "Class": // scClass // role // none
                        if (className) {
                            var titleInnerMatches = titleMatches.findAll("Class", className);
                            if (titleInnerMatches != null) {
                                titleMatch = titleInnerMatches[0];
                                if (titleInnerMatches.length == 1 && titleMatch) {
                                    if (this.logIsDebugEnabled("AutoTest")) {
                                        this.logDebug("Locator string:" + substring + 
                                            " - returning widget with matching title and ClassName:" +
                                            titleMatch, "AutoTest");
                                    }
                                    return titleMatch;
                                }
                            }
                        }
                        
                    case "scClass":
                        if (scClassName) {
                            var titleInnerMatches = titleMatches.findAll("_scClass", scClassName);
                            if (titleInnerMatches != null) {
                                if (titleInnerMatches.length ==1 || titleMatch == null)
                                    titleMatch = titleInnerMatches[0];
                               
                                if (titleInnerMatches.length == 1 && titleMatch) {
                                        
                                    if (this.logIsDebugEnabled("AutoTest")) {
                                        this.logDebug("Locator string:" + substring + 
                                            " - returning widget with matching name and scClassName:" +
                                            titleMatch, "AutoTest");
                                    }
                                    return titleMatch;
                                }
                            }
                        }
                    case "role":
                        if (role) {
                            var titleInnerMatches = titleMatches.findAll("waiRole", role);
                            if (titleInnerMatches != null) {
                                if (titleInnerMatches.length ==1 || titleMatch == null)
                                    titleMatch = titleInnerMatches[0];
                               
                                if (titleInnerMatches.length == 1 && titleMatch) {
                                    
                                    if (this.logIsDebugEnabled("AutoTest")) {
                                        this.logDebug("Locator string:" + substring + 
                                            " - returning widget with matching title and role:" +
                                            titleMatch, "AutoTest");
                                    }
                                    return titleMatch;
                                }
                            }
                        }
                        
                    default:
                        // In this case we've got a matching title but we can't match it to
                        // class or role.
                        // Log the "unreialble locator" one time warning -- the fact that
                        // we couldn't find a match by class as well as title implies things
                        // must have changed since the recording was made...
                        //
                        // Return the match if it's unique, otherwise ignore it and move on to 
                        // matching by index.
                        
                        if (titleMatches.length == 1) {
                            
                            if (typeStrategy != "none") {
                                isc.AutoTest.logRobustLocatorWarning();
                               
                                this.logWarn ("Locator string:" + substring + 
                                    ". Returning closest match:" + titleMatches[0] + ". This has the same title " +
                                    "as the recorded component but does not match class or role.", "AutoTest");
                            } else {
                                if (this.logIsDebugEnabled("AutoTest")) {
                                    this.logDebug("Locator string:" + substring + 
                                        " - returning widget with matching title:" +
                                        titleMatch, "AutoTest");
                                }
                            }
                            return titleMatches[0];
                        } else {
                            this.logWarn("Locator string:" + substring +
                                ", attempt to match by title failed -- multiple candidate components have this " +
                                "same title. Attempting to match by index instead.", "AutoTest");
                        }
                    } // end of inner switch
                }
            }
            
        // either strategy is "index" or we didn't find a title/name match
        default:
                
            
            // back off to index
            // We captured index per class name, per scClass and per role as well as the
            // raw index in the array.
            // Test them in that order.
            // Note that if the lengths have changed this is likely wrong!
             var classIndexMatch,
                scClassIndexMatch,
                roleIndexMatch;
                
             switch (typeStrategy) {
             case "Class": // scClass // role // none
              
    
                if (className && config.classIndex) {
                    var classMatches = candidates.findAll("Class", className);
                    if (classMatches && classMatches.length > 0) {
                        
                        classIndexMatch = classMatches[parseInt(config.classIndex)];
                        
                        if (classMatches.length == parseInt(config.classLength)) {
                        
                            if (this.logIsInfoEnabled("AutoTest")) {
                                this.logInfo("Locator string:" + substring + 
                                        " - returning widget with matching ClassName / index by ClassName:" +
                                        classIndexMatch, "AutoTest");
                            }
                            return classIndexMatch;
                        }
                        // If the lengths didn't match, the index is very likely unreliable
                        // Hang onto it to return it if we can't match by scClassName or role more
                        // reliably
                    }
                }
                
            case "scClass":
                
                if (scClassName && config.scClassIndex) {
                    
                    var scClassMatches = candidates.findAll("_scClass", scClassName);
                    if (scClassMatches && scClassMatches.length > 0) {
                        
                        scClassIndexMatch = scClassMatches[parseInt(config.scClassIndex)];
                        
                        if (scClassMatches.length == parseInt(config.scClassLength)) {
                        
                            if (this.logIsInfoEnabled("AutoTest")) {
                                this.logInfo("Locator string:" + substring + 
                                        " - returning widget with matching SmartClient superclass / index by ClassName:" +
                                        scClassIndexMatch, "AutoTest");
                            }
                            return scClassIndexMatch;
                        }
                        // If the lengths didn't match, the index is very likely unreliable
                        // Try roles before using this
                    }
                }
                
            case "role":
                
                if (role && config.roleIndex) {
                    
                    var roleMatches = candidates.findAll("waiRole", role);
                    if (roleMatches && roleMatches.length > 0) {
                        
                        roleIndexMatch = roleMatches[parseInt(config.roleIndex)];
                        
                        if (roleMatches.length == parseInt(config.roleLength)) {
                        
                            if (this.logIsInfoEnabled("AutoTest")) {
                                this.logInfo("Locator string:" + substring + 
                                        " - returning widget with matching role / index by role:" +
                                        roleIndexMatch, "AutoTest");
                            }
                            return roleIndexMatch;
                        }
                    }
                }
                
            default:
                
                // At this point if we had class/scClass or role, we know the lengths have changed
                // so index is very unreliable.
                // In this case, or if the overall length has changed, log the robustLocatorWarning
                //
                // Then return our best guess
                if ((typeStrategy != "none" && (className || scClassName || role)) || 
                    (config.length != null && (parseInt(config.length) != candidates.length))) 
                {
                    isc.AutoTest.logRobustLocatorWarning();
                }
                
                var match = classIndexMatch || scClassIndexMatch || roleIndexMatch;
                if (match == null) {
                    var index = config[isc.AutoTest.fallback_valueOnlyField];
                    if (index == null) index = config.index;
                    index = parseInt(index);
                    
                    match = candidates[index];
                }
                
                
                if (match) {
                    this.logWarn("Locator string:" + substring + 
                        " matching by index gave " + match +
                        ". Reliability cannot be guaranteed for matching by index if the underlying " +
                        "application undergoes any changes.", "AutoTest");
                    return match;
                }
            } // closes inner switch statement
        } // closes outer switch statement
        
        
        
        // if we're here, we didn't find any candidates, or didn't find a child within them.
        // This doesn't necessarily indicate any kind of failure: We use fallback locators
        // for elements within some components - EG list grid cells
        this.logDebug("AutoTest.getElement(): locator substring:" + substring + 
            " parsed to fallback locator name:" + name + 
            ", unable to find relevant child - may refer to inner element.", "AutoTest");
            
            
            
    }
});
    
isc.Canvas.addMethods({
    
    //> @method canvas.getLocator()
    // Get an abstract Locator String for an element contained within this Canvas
    // @param (DOMElement) DOM element contained within this Canvas
    // @return (Locator) abstract Locator String
    // @visibility autoTest
    //<
    // No apparent need to expose this directly, unless we are ready to support developers
    // writing their own locator logic in addition to the defaults
    ///
    // Additional 'fromEvent' param tells us we're actually retriving the target for the
    // current mouse event
    // In some cases we can use this to get additional info that isn't available from the
    // actual target element (EG target cell in a GR when showing a floating embedded componet)
    getLocator : function (element, fromEvent) {
        var baseLocator, parent;
        if (this._generated || this.locatorParent || this.creator || this._autoAssignedID) {
            parent = this.getLocatorParent();
        }
        if (!parent) {
            baseLocator = this.getLocatorRoot();
        } else {
            baseLocator = parent.getLocator() + "/" +
                          parent.getChildLocator(this);
        }
        if (element) return [baseLocator, this.getInteriorLocator(element,fromEvent)].join("/");
        return baseLocator;
    },
    
    _locatorRootTemplate:[
    "//",
    ,   // classname
    '[ID="',
    ,   // global ID
    '"]'
    ],
    getLocatorRoot : function () {
        
        if (!this.locatorRoot) {
            // If the widget has an explicitly specified ID always use it above all else!
            // Otherwise we'll use the "fallbackLocator" pattern to find it
            if (this._autoAssignedID && this.parentElement == null) {
                this.locatorRoot = "//" +
                    isc.Canvas.getCanvasLocatorFallbackPath("autoID", this, isc.Canvas._topCanvii);
            } else {
                this._locatorRootTemplate[1] = this.getClassName();
                this._locatorRootTemplate[3] = this.getID();
                this.locatorRoot = this._locatorRootTemplate.join(isc.emptyString);
            }
        }
        return this.locatorRoot;
    },
    
    containsLocatorChild : function (canvas) {
        if (this.namedLocatorChildren != null) {
            for (var i = 0; i < this.namedLocatorChildren.length; i++) {
                var name = this.namedLocatorChildren[i];
                if (isc.isAn.Object(name)) name = name.attribute;
                if (canvas == this[name]) {
                    return true;
                }
            }
        }
        return false;
    },
    
    getLocatorParent : function () {
        // locatorParent -- this is a generic entry point allowing special locator parent/child
        // behavior. 
        // To make use of this a widget could set itself as the locatorParent of some other
        // widget, and implement custom 'containsLocatorChild()' / 'getChildLocator()'  
        if (this.locatorParent && this.locatorParent.containsLocatorChild(this)) {
            return this.locatorParent;
        }
        if (this.creator && isc.isA.Canvas(this.creator)) {
            var autoChildName = this.creator.getAutoChildLocator(this);
            if (autoChildName == null) {
                // failed to find the child - most likely created via 'createAutoChild' but
                // never ran through addAutoChild() which would make it detectable in the
                // getAutoChildLocator() method
                // This is likely to happen if we are using the auto-child system to create
                // numerous auto-children with common properties, so it's not really a
                // failure.
                // Allow this to continue through the standard master-peer / parent-child
                // logic.
                this.logInfo("Locator code failed to find relationship between parent:"+
                            this.creator.getID() + " and autoChild:"+ this.getID(), "AutoTest");
            } else {
                return this.creator;
            }
        }
        return this.masterElement || this.parentElement;
    },
    
  
    //> @method canvas.getChildLocator()
    // Get the abstract Locator string for finding a child canvas within its parent element 
    // @param (Canvas)
    // @return (Locator) abstract Locator String for finding this child
    //<
    // Leave this internal - developers would call getLocator() directly
    _childLocatorTemplate:[
        ,   // "child" or "peer"
        "[",
        ,   // index of child/peer
        '][Class="',
        ,   // className of child/peer
        '"]'
    ],
     
    
    getChildLocator : function (canvas) {
        // special case scrollbars
        if (canvas == this.hscrollbar) {
            return "hscrollbar";
        }
        if (canvas == this.vscrollbar) {
            return "vscrollbar";
        }
        
        // More general behavior split into 2 parts for easy overriding - autoChildren are pretty
        // much always respected over other locators such as children / members array
        if (canvas.creator == this) {    
            var autoChildID = this.getAutoChildLocator(canvas);
            if (autoChildID) return autoChildID;
        }
        
        return this.getStandardChildLocator(canvas);
    },
    
    // Called when AutoTest.getLocator() is called with the checkNativeElement parameter.
    // This method tests for the case where we have an element that natively 
    // "has meaning" in terms of events (IE eventHandledNatively is true) and our generated
    // SC-locator won't get back to that element.
    // Example case: A link written into a canvas handle -- the locator will likely point to
    // the canvas, while the link itself is the element that should be recorded.
    // In this case testing tools such as selenium may be able to get a better identifier 
    // based on (EG) ID of the link element.
    //
    // We do have cases where a widget writes out a live element which will handle native events
    // but we already handle generating a full locator to get at them (rather than just the
    // canvas handle). Example case: link elements within the month view of a calendar widget.
    // 
    // We test for this case by doing a round-trip test - if the locator already directly
    // points to the element (via AutoTest.getElement()), we use the locator.
    //
    
    // Implemented at the Canvas level so we can override this in subclasses if appropriate.
    checkLocatorForNativeElement : function (locator, element) {
        if (element == null || locator == null) return false;
        
        return (isc.EventHandler.eventHandledNatively("mousedown", element, true) &&
                (isc.AutoTest.getElement(locator) != element))
    },

    
    // getCanvasLocatorFallbackPath
    // generates a standard 'fallback path' to locate a widget from within a pool of widgets.
    // Used for locating mutliple auto children with the same name, members, peers, children
    // and so on.
    // The concept is that this'll capture as much information as possible so we can
    // use fallback strategies to get at the right object from a stored path.
    getCanvasLocatorFallbackPath : function (name, canvas, sourceArray, properties, mask) {
        return isc.Canvas.getCanvasLocatorFallbackPath(name,canvas,sourceArray,properties,mask);
    },
    
    
    
    getAutoChildLocator : function (canvas) {
        
        if (this._createdAutoChildren) {
            var ID = canvas.getID();
            for (var childName in this._createdAutoChildren) {
                var children = this._createdAutoChildren[childName];
                if (children.contains(ID)) {
                    // common case this.header etc
                    if (canvas == this[childName]) return childName;
                    else {
                        // create an array of the *live* auto children (not just their IDs)
                        // this allows us to figure out our index in that array as well as
                        // our index based on role!
                        var liveChildren = [];
                        for (var i = 0; i < children.length; i++) {
                            liveChildren[i] = window[children[i]];
                        }
                        
                        return this.getCanvasLocatorFallbackPath(childName, canvas, liveChildren);
                        
                    }
                }
            }
        }
        return null
    },
    
    getNamedLocatorChildString : function (canvas) {
         
        // Fairly common pattern - this.<someAttribute> is set directly to the canvas
        // but for whatever reason it didn't go through the addAutoChild() subsystem.
        // We can handle this explicitly by:
        // - setting locatorParent on the child to point to this widget
        // - adding an entry to the "namedLocatorChildren" array with the attribute name
        if (canvas.locatorParent == this && this.namedLocatorChildren) {
            for (var i = 0; i < this.namedLocatorChildren.length; i++) {
                var name = this.namedLocatorChildren[i],
                    attrName = name;
                    
                // support an object of the format {name:"name", attribute:"attributeName"}
                // This allows us to defeat changing obfuscated names like "_editRowForm"
                if (isc.isA.Object(name)) {
                    attrName = name.attribute,
                    name = name.name;
                }
                if (canvas == this[attrName]) {
                    return name;
                }
            }
        }
    },
    
    getStandardChildLocator : function (canvas) {
        var nlcs = this.getNamedLocatorChildString(canvas);
        if (nlcs) return nlcs;
       
        
        if (canvas.masterElement == this) {
            return this.getCanvasLocatorFallbackPath("peer", canvas, this.peers);
            
        } else if (canvas.parentElement == this) {
            return this.getCanvasLocatorFallbackPath("child", canvas, this.children);
        } else {
            // Not clear what would cause this - we already catch the autoChild case, 
            // so this is really a sanity check only
            this.logWarn("unexpected error - failed to find relationship between parent:"+
                        this.getID() + " and child:"+ canvas.getID());
            // return the standard root ID for the canvas - when parsing the strings back
            // we will have to explicitly catch this case?
            return canvas.getLocatorRoot();
        }
    },
    
    //> @method canvas.getInteriorLocator()
    // Get a relative Locator for an element contained within this Canvas
    // @param (DOMElement) DOM element contained within this Canvas
    // @return (Locator) abstract Locator String
    //<
    // Overridden to provide standard "meaningful locations" for ListGrids, DynamicForm, etc
    getInteriorLocator : function (element, fromEvent) {
        if (element && this.useEventParts) {
            var partObj = this.getElementPart(element);
            if (partObj != null && partObj.part != null) {
                // This will be of the format "partType_partID"
                return (partObj.partID && partObj.partID != isc.emptyString) ? 
                                        partObj.part + "_" +  partObj.partID : partObj.part;
            }
        }
        return isc.emptyString;
    },
    
    
    // -------------------------
    // Retrieving dom elements from locator strings
    //> @method canvas.getElementFromSplitLocator()
    // Given a locator string split into an array, return a pointer to the appropriate DOM element.
    // @param (DOMElement) DOM element contained within this Canvas
    // @return (Locator Array) array of strings
    // @visibility internal
    //<
    // Internal - the parameter format does not match the Locator format returned by
    // canvas.getLocator -- developers should call AutoTest.getElement() rather than directly 
    // accessing this method
    getElementFromSplitLocator : function (locatorArray) {

        var child = this.getChildFromLocatorSubstring(locatorArray[0], 0, locatorArray);
        if (child) {
            locatorArray.removeAt(0);
            return child.getElementFromSplitLocator(locatorArray);
        }
        // split finding the element within our handle to a separate method for simpler override
        return this.getInnerElementFromSplitLocator(locatorArray);
    },
    
    // Given a substring extracted from a split locator array, return the child widget
    // that matches the specified substring.
    // If there is no matching child, return null - we'll then treat this widget as the
    // innermost child widget treat any remaining locator info as an interior locator
     
    getChildFromLocatorSubstring : function (substring, index, locatorArray) {
        if (substring == null || substring == "") return null;
        
        // Standard formats:
        // 
        // Attribute pointing directly to widget:
        // EG:
        // - vscrollbar/hscrollbar 
        // - named autoChild
        // - things in the "namedLocatorChildren" array
        if (isc.isA.Canvas(this[substring])) {
            return this[substring]
        }
        
        // - standard attribute<-->name mappings in the namedLocatorChildren array:
        if (this.namedLocatorChildren != null) {
            var rename = this.namedLocatorChildren.find("name", substring);
            if (rename != null) {
                var canvas = this[rename.attribute];
                if (isc.isA.Canvas(canvas)) return canvas;
                this.logWarn("Locator substring:" + substring 
                    + " remaps to attribute:" + rename.attribute + 
                    " but no canvas exists under that attribute name.", "AutoTest");
                // this is probably a failure - could return null here or keep going
                // - keep going in case some other strategy finds the component?
            }
        }
        
        // Fallback locators ([childType][fallback locator for specific child])
        // EG:
        // - autoChildName[<fallback locator within auto children>]
        // - children[<fallback locator>]
        // - members[<fallback locator>]
        var fallbackLocatorConfig =  isc.AutoTest.parseLocatorFallbackPath(substring);
        if (fallbackLocatorConfig != null) {
            return this.getChildFromFallbackLocator(substring, fallbackLocatorConfig);
        }
        
                
        // if we're here, we didn't find any candidates, or didn't find a child within them.
        // No need to warn here -- this is likely to happen if the remaining identifier is
        // an inner element locator
        return null;
        
    },
    
    //> @type LocatorStrategy
    // The AutoTest subsystem relies on generating and parsing identifier strings to identify
    // components on the page. A very common pattern is identifying a specific component
    // within a list of possible candidates. There are many many cases where this pattern
    // is used, for example - members in a layout,tabs in a tabset, sections in a section stack.
    // <P>
    // In order to make these identifiers as robust as possible across minor
    // changes to an application, (such as skin changes, minor layout changes, etc) the
    // system will store multiple pieces of information about a component when generating
    // an identification string to retrieve it from a list of candidates.
    // The system has a default strategy for choosing the order in which to look at these
    // pieces of information but in some cases this can be overridden by setting
    // a <code>LocatorStrategy</code>.
    // <p>
    // By default we use the following strategies in order to identify a component from a list of
    // candidates:
    // <UL><li><code>name</code>: Does not apply in all cases but in cases where a specified
    //   <code>name</code> attribute has meaning we will use it - for example for
    //  +link{SectionStackSection.name,sections in a section stack}.</li>
    // <li><code>title</code>: If a title is specified for the component this may be used
    //   as a legitimate identifier if it is unique within the component - for example
    //   differently titled tabs within a tabset.</li>
    // <li><code>index</code>: Locating by index is typically less robust than by name or
    //   title as it is likely to be effected by layout changes on the page.</li>
    // </UL>
    // If an explicit strategy is specified, that will be used to locate the component if 
    // possible. If no matching component is found using that strategy, we will continue to
    // try the remaining strategies in order as described above. In other words setting
    // a locatorStrategy to "title" will skip attempting to find a component by name, and
    // instead attempt to find by title - or failing that by index.
    // <P>
    // Note that we also support matching by type (see +link{type:LocatorTypeStrategy}).
    // Matching by type is used if we were unable to match by name or title or to disambiguate
    // between multiple components with a matching title.
    //
    // @value "name" Match by name if possible.
    // @value "title" Match by title if possible.
    // @value "index" Match by index
    // @visibility external
    // @group autoTest
    //<
    
    //> @type LocatorTypeStrategy
    // When attempting to identify a component from within a list of possible candidates
    // as described +link{type:LocatorStrategy,here}, if we are unable to find a unique match
    // by name or title, we will use the recorded "type" of the component to verify
    // an apparent match.
    // <P>
    // By default we check the following properties in order:
    // <ul><li>Does the Class match?</li>
    //     <li>If this is not a +link{Class.isFrameworkClass,framework class}, does the
    //         core framework superclass match?</li>
    //     <li>Does the <code>role</code> match?</li>
    // </ul>
    // In some cases an explicit locatorTypeStrategy can be specified to modify this
    // behavior. As with +link{type:LocatorStrategy}, if we are unable to match using the
    // specified type strategy we continue to test against the remaining strategies in order - 
    // so if a type strategy of "scClass" was specified but we were unable to find a match
    // with the appropriate core superclass, we will attempt to match by role.
    // Possible values are:
    // @value "Class" Match by class if possible
    // @value "scClass" Ignore specific class and match by the SmartClient framework superclass.
    // @value "role" Ignore class altogether and attempt to match by role
    // @value "none" Don't attempt to compare type in any way
    // @visibility external
    // @group autoTest
    //<

    //> @attr Canvas.locateChildrenBy (LocatorStrategy : null : IRWA)
    // Strategy to use when locating children in this canvas from an autoTest locator string.
    // 
    // @visibility external
    // @group autoTest
    //<
    
    //> @attr Canvas.locateChildrenType (LocatorTypeStrategy : null : IRWA)
    // +link{type:LocatorTypeStrategy} to use when finding children within this canvas.
    // @visibility external
    // @group autoTest
    //<
    
    //> @attr Canvas.locatePeersBy (LocatorStrategy : null : IRWA)
    // Strategy to use when locating peers of this canvas from an autoTest locator string.
    // 
    // @visibility external
    // @group autoTest    
    //<
    
    //> @attr Canvas.locatePeersType (LocatorTypeStrategy : null : IRWA)
    // +link{type:LocatorTypeStrategy} to use when finding peers of this canvas.
    // @visibility external
    // @group autoTest
    //<
    
    // given a childType -- for example "peers"
    // figure out the specified child locator strategy.
    // Works by looking for this.locate[pluralName]By -- EG
    // locatePeersBy
    getChildLocatorStrategy : function (childType) {
        if (isc.AutoTest.locStrategyNames == null) {
            isc.AutoTest.locStrategyNames = {};
        }
        
        var attrName = isc.AutoTest.locStrategyNames[childType];
        if (attrName == null) {
            var pluralName = childType;
            if (isc.isA.String(this._locatorChildren[childType])) pluralName = this._locatorChildren[childType];
            attrName = isc.AutoTest.locStrategyNames[childType] =
                        "locate" + 
                        pluralName.substring(0,1).toUpperCase() + pluralName.substring(1) +
                        "By";
        }
        
        return this[attrName];
    },
    // Same type of logic for type-identifiers
    // checks for this.locate[pluralName]Type -- EG: locatePeersType
    getChildLocatorTypeStrategy : function (childType) {
           
        if (isc.AutoTest.locStrategyTypes == null) {
            isc.AutoTest.locStrategyTypes = {};
        }
        
        var attrName = isc.AutoTest.locStrategyTypes[childType];
        if (attrName == null) {
            var pluralName = childType;
            if (isc.isA.String(this._locatorChildren[childType])) pluralName = this._locatorChildren[childType];
            attrName = isc.AutoTest.locStrategyTypes[childType] =
                        "locate" + 
                        pluralName.substring(0,1).toUpperCase() + pluralName.substring(1) +
                        "Type";
        }
        
        return this[attrName];
    },
    
    
    // substring param really just used for logging
    getChildFromFallbackLocator : function (substring, fallbackLocatorConfig) {
    
        var type = fallbackLocatorConfig.name,
            config = fallbackLocatorConfig.config;
        // default logic:
        // we use the "name" to find candidate widgets, then use the config to
        // figure out which candidate we actually want
        var candidates = this.getFallbackLocatorCandidates(type);
        if (candidates && candidates.length > 0) {
            
                  
            var strategy = this.getChildLocatorStrategy(type);
            if (strategy == null) strategy = "name";
            var typeStrategy = this.getChildLocatorTypeStrategy(type);
            if (typeStrategy == null) typeStrategy = "Class";
            
            var match = isc.Canvas.getCanvasFromFallbackLocator(
                            substring, config, candidates, 
                            strategy, typeStrategy);
            if (match != null) return match;
        }
        
        // if we're here, we didn't find any candidates, or didn't find a child within them.
        // This doesn't necessarily indicate any kind of failure: We use fallback locators
        // for elements within some components - EG list grid cells
        this.logDebug("AutoTest.getElement(): locator substring:" + substring + 
            " parsed to fallback locator name:" + type + 
            ", unable to find relevant child - may refer to inner element.", "AutoTest");
            
    },
    
    
    _locatorChildren:{
        peer:"peers",
        child:"children"
    },
    getFallbackLocatorCandidates : function (name) {
    
        var candidates;
        
        // check _createdAutoChildren for autoChildren by autoChild name
        if (this._createdAutoChildren != null && this._createdAutoChildren[name] != null) {
            var IDs = this._createdAutoChildren[name];
            candidates = [];
            for (var i = 0; i < IDs.length; i++) {
                candidates[i] = window[IDs[i]];
            }
            
        // _locatorChildren object: This specifies a mapping between known cases where
        // we have an attribute on this widget containing an array of candidates
        // (EG the children array) and a known 'locator' childType name (EG "child")
        
        } else if (isc.isA.String(this._locatorChildren[name])) {
            candidates = this[this._locatorChildren[name]];
        
        // Also support the 'name' pointing directly to an attribute on this widget 
        // containing an array of candidate objects (So could store "children" directly
        // rather than using the remapping above).
        } else if (this[name] && isc.isAn.Array(this[name])) {
            candidates = this[name];
        }
        return candidates;
    },
    
    
    emptyLocatorArray : function (locatorArray) {
        return locatorArray == null || locatorArray.length == 0 ||
                (locatorArray.length == 1 && locatorArray[0] == "");
    },
    
    getInnerElementFromSplitLocator : function (locatorArray) {
        if (!this.emptyLocatorArray(locatorArray)) {
            // support event-parts in all canvii
            if (locatorArray.length == 1) {
                
                var parts = locatorArray[0].split("_");
                
                var part = {
                        part:parts[0],
                        partID:parts[1]
                    };
                var element = this.getPartElement(part);
                if (element) return element;
            }
            
        }
        return this.getHandle();
    },
      
    // Retrieving coordinates based on element / locator string
    getAutoTestLocatorCoords : function (locator, element) {

        // we assume both are present for now
        if (locator == null || element == null) return null;

        var rect = isc.Element.getElementRect(element);
        // return the center of the element
        
        var left = rect[0],
            width = rect[2];
        left += Math.floor(width/2);
        
        var top = rect[1],
            height = rect[3];
            
        top += Math.floor(height/2);
        
        return [left,top];
    }

});
          
// -----------------------------------------------------------------
// Override getChildLocator() for special cases

if (isc.Layout) {
    isc.Layout.addProperties({
            
        //> @attr Layout.locateMembersBy (LocatorStrategy : null : IRWA)
        // Strategy to use when locating members from within this Layout's members array.
        // 
        // @visibility external
        // @group autoTest
        //<
        
        //> @attr Layout.locateMembersType (LocatorTypeStrategy : null : IRWA)
        // +link{type:LocatorTypeStrategy} to use when finding members within this layout.
        // @visibility external
        // @group autoTest
        //<
        
            
        getStandardChildLocator : function (canvas) {
            var nlcs = this.getNamedLocatorChildString(canvas);
            if (nlcs) return nlcs;
            
            if (this.members.contains(canvas)) {
                return this.getCanvasLocatorFallbackPath("member", canvas, this.members);
            }
            
            return this.Super("getStandardChildLocator", arguments);
        },
        
        
        _locatorChildren:{
            member:"members",
            peer:"peers",
            child:"children"
        }
    });
}

if (isc.Window) {
    isc.Window.addProperties({
        // Code in Window.js sets up Windows as the 'locatorParent' of their items
        containsLocatorChild : function (canvas) {
            if (this.items && this.items.contains(canvas)) return true;
            return this.Super("containsLocatorChild", arguments);
        },
        getStandardChildLocator : function (canvas) {
        
            if (this.items && this.items.contains(canvas)) {
                var template = this._childLocatorTemplate;
                template[0] = "item";
                template[2] = this.items.indexOf(canvas);
                template[4] = canvas.getClassName();
                
                return template.join(isc.emptyString);
            }
            
            return this.invokeSuper(isc.Window, "getStandardChildLocator", canvas);            
        },
        
        _locatorChildren:{
            item:"items",
            member:"members",
            peer:"peers",
            child:"children"
        }
    });
}
//  - in a Window, for an non-autoChild item, item[itemIndex][Class="className"]

if (isc.SectionStack) {
    
    // add the _locatorChildren for SectionHeader / ImgSectionHeader - this will
    // allow them to parse the item[fallbacklocator] generated by the
    // sectionStack standard child locator override below
    isc.ImgSectionHeader.changeDefaults("_locatorChildren", {item:"items"});
    isc.SectionHeader.changeDefaults("_locatorChildren", {item:"items"});

    
    // add sections to locatorChildren for SectionStack - allows it to parse the
    // section[fallbackLocator] we create below
    isc.SectionStack.changeDefaults("_locatorChildren", {section:"sections"});
    
    isc.SectionStack.addProperties({
            
        // override getStandardChildLocator - for sections return 
        //  section[name="name"||title="title"||3]
        // for items, append
        //  item[0]
        getStandardChildLocator : function (canvas) {
            var sections = this.sections || [],
                locatorString;
            for (var i = 0; i < sections.length; i++) {
    
                var items = sections[i].items,
                    section, item;
                if (canvas == sections[i]) {
                    section = canvas;
                    
                } else if (items && items.contains(canvas)) {
                    
                    section = sections[i];
                    item = canvas;
                }
                
                if (section != null) {
                    
                    // This will pick up name by default, then title, index, etc
                    locatorString = this.getCanvasLocatorFallbackPath("section", section, this.sections);
                }
                
                if (item != null) {
                    locatorString += "/" + this.getCanvasLocatorFallbackPath("item", item, section.items);
                }
                if (locatorString != null) return locatorString;
            }
            
            return this.Super("getStandardChildLocator", arguments);
        }
           
        //> @attr SectionStack.locateSectionsBy (LocatorStrategy : null : IRWA)
        // When +link{isc.AutoTest.getElement()} is used to parse locator strings generated by
        // link{isc.AutoTest.getLocator()}, how should sections within this stack be identified?
        // By default if section has a specified +link{Section.name} this will always be used.
        // For sections with no name, the following options are available:
        // <ul>
        // <li><code>"title"</code> use the title as an identifier</li>
        // <li><code>"index"</code> use the index of the section in the sections array as an identifier</li>
        // </ul>
        // 
        // If unset, and the section has no specified name, default behavior is to
        // identify by title (if available), otherwise by index.
        // @visibility external
        // @group autoTest
        //<
        
        //> @attr SectionStack.locateSectionsType (LocatorTypeStrategy : null : IRWA)
        // +link{type:LocatorTypeStrategy} to use when finding Sections within this section Stack.
        // @visibility external
        // @group autoTest
        //<
        

        // This will be picked up automatically based on the _locatorChildren object and
        // the standard "getLocatorStrategy()" logic
        
            
    });  
}

// --------------------------------------------------
// Interior locators

if (isc.StretchImg) {
isc.StretchImg.addProperties({
    getInteriorLocator : function (element, fromEvent) {
        // We don't use the useEventParts flag in StretchImgs but in some cases we need to tell the
        // difference between events on different items
        // (EG a track-click and a button click)
        var origElement = element,
            handle = this.getHandle(), canvasName = this.getCanvasName();

        while (element && element != handle && element.getAttribute) {
            // check the "name" property for the open-icon 
            var ID = element.getAttribute("name");
            if (ID && ID.startsWith(canvasName)) {
                return ID.substring(canvasName.length);
            }
            element = element.parentNode;
        }
        return this.Super("getInteriorLocator", [origElement,fromEvent]);
    },
    
    getInnerElementFromSplitLocator : function (locatorArray) {
        
        // check for "name" - used for parts
        if (!this.emptyLocatorArray(locatorArray) && locatorArray.length == 1) {
            var image = this.getImage(locatorArray[0]);
            if (image) return image;
        }
        return this.Super("getInnerElementFromSplitLocator", arguments);
    }
    
    
    
});
}


// label.icon already handled via standard canvas 'eventPart' handling

if (isc.DynamicForm) {
    isc.DynamicForm.addProperties({
        getInteriorLocator : function (element) {
            var itemInfo = isc.DynamicForm._getItemInfoFromElement(element, this);
            // itemInfo format:
            // {item:item, overElement:boolean, overTitle:boolean, overTextBox:boolean,
            //  overControlTable:boolean, overIcon:string}
            if (!itemInfo.item) return this.Super("getInteriorLocator", arguments);
            var item = itemInfo.item,
                locator = [this.getItemLocator(item), '/'];
                
            if (itemInfo.overElement) locator[locator.length] = "element";
            else if (itemInfo.overTitle) locator[locator.length] = "title";
            else if (itemInfo.overTextBox) locator[locator.length] = "textbox";
            else if (itemInfo.overControlTable) locator[locator.length] = "controltable";
            else if (itemInfo.overIcon) locator[locator.length] = "[icon='" + itemInfo.overIcon + "']"
            
            return locator.join(isc.emptyString);
        },
        
        getItemLocator : function (item) {
            
            // containerItems contain sub items, which point back up to them via the
            // parentItem attribute
            // If we hit a sub-item of a container item, call getItemLocator on that so
            // the item is located within the containerItem's items array
            // This method is copied from DF to containerItems below
            // the check for item.parentItem != this is required - if this is running
            // on a container item and we contain an item in our items array we need to
            // allow standard identifier construction to continue or we'd have an infinite loop
            if (item.parentItem && (item.parentItem != this)) {
                return this.getItemLocator(item.parentItem) + "/" + 
                            item.parentItem.getItemLocator(item);
            }
            
            var itemIdentifiers = {};
            
            if (item.name != null) itemIdentifiers.name = item.name;
            
            // Title - default strategy if no name
            var title = item.getTitle();
            if (title != null) itemIdentifiers.title = title;
            
            // Value - useful for things like header items where value is pretty much
            // a valid identifier
            var value = item.getValue();
            if (value != null) itemIdentifiers.value = value;
            
            // Index - cruder identifier
            itemIdentifiers.index = this.getItems().indexOf(item);
            
            // ClassName: Not used by default
            itemIdentifiers.Class = item.getClassName();
            
            var IDString = isc.AutoTest.createLocatorFallbackPath("item", itemIdentifiers);
            return IDString;
        },
        
        containsLocatorChild : function (canvas) {
            if (isc.isA.DateChooser(canvas) && canvas.callingForm == this) return true;
            return this.Super("containsLocatorChild", arguments);
        },
        getChildLocator : function (canvas) {
            if (canvas.canvasItem) {
                var item = canvas.canvasItem;
                return this.getItemLocator(item) + "/canvas";
            }
            if (isc.isA.PickListMenu(canvas)) {
                var item = canvas.formItem;
                return this.getItemLocator(item) + "/pickList";
            }
            if (isc.isA.DateChooser(canvas)) {
                var item = canvas.callingFormItem;
                return this.getItemLocator(item) + "/picker";
            }
            
            return this.Super("getChildLocator", arguments);
        },
        
        getItemFromSplitLocator : function (locatorArray) {
            var fullItemID = locatorArray[0],
                className;

            // BackCompat note: Old format for identifying form items was
            //   item[name="foo"][Class="TextItem"]
            // new format is
            //   item[name=foo||title=moo||index=2||Class=TextItem]
            // Handle the old format for backCompat
            if (fullItemID.contains("[Class=")) {
                var split = fullItemID.match(
                    "item\\[(.+)'\\]\\[Class=\"(.+)\"\\]"
                );
                className = split[1].substring(6, split[1].length-2);
                fullItemID = split[0];
            }
            var itemConfig = isc.AutoTest.parseLocatorFallbackPath(fullItemID);
            
            if (itemConfig && itemConfig.name == "item" && itemConfig.config != null) {
                var config = itemConfig.config;
                
                // className is stored even if we don't identify by it.
                className = config.Class;
                
                // if we have a valid name, always have it take precedence
                var item;
                if (config.name != null) {
                    //this.logWarn("locating by name" + config.name);
                    item = this.getItem(config.name);
                } else {
                    //this.logWarn("item locator:" +fullItemID + " has no name - checking for " +
                    //    " title etc.");
                    
                    // no name - check for the item 'locateItemBy' setting
                    // Options are by title or by value
                    for (var i = 0; i < this.items.length; i++) {
                        var testItem = this.items[i],
                            locateItemBy = testItem.locateItemBy;
                        if (locateItemBy == null) locateItemBy = "title";
                        //this.logWarn("item:" + testItem + ", locate by:" + locateItemBy + 
                        //    "config[locateBy:" + config[locateItemBy]);
                        if (locateItemBy == "title" && config.title != null && 
                            testItem.title == config.title) 
                        {
                            item = testItem;
                        } else if (locateItemBy == "value" && config.value != null && 
                                    testItem.getValue() == config.value) 
                        {
                            item = testItem;
                        }
                    }
                    
                    // If we couldn't find the item by title or value (or locateItemBy was
                    // specified explicitly as index) - locate by index
                    if (item == null) {
                        var index = config.index;
                        if (isc.isA.String(index)) {
                            if (index.startsWith("'") ||
                                index.startsWith('"')) 
                            {
                                index = index.substring(1);
                            }
                            index = parseInt(index);
                        }
                        item = this.items[index];
                    }
                }
                if (!item) {
                    this.logWarn("AutoTest.getElement(): Unable to find item from " +
                        "locator string:" + fullItemID);
                    return null;
                }
                if (!isc.isA[className] || !isc.isA[className](item)) {
                    this.logWarn("AutoTest.getElement(): identifier:"+ fullItemID + 
                                " returned an item of class:"+ item.getClassName());
                }
                return item;
            }
            
            return null;
        },
        
        getInnerElementFromSplitLocator : function (locatorArray) {
            if (this.emptyLocatorArray(locatorArray)) {
                return this.getHandle();
            }
            
            var item = this.getItemFromSplitLocator(locatorArray);
            if (item != null) {
                locatorArray.removeAt(0);
                return item.getInnerElementFromSplitLocator(locatorArray);
            }
            return this.getHandle();
        }
    });
    
    // containerItems contain sub items
    // copy methods across to them to form locators for sub items and
    // identify sub items from split locators
    isc.ContainerItem.addProperties({
        // getItemLocator -- called directly by DynamicForm.getItemLocator if
        // an item has a parentItem specified
        getItemLocator:isc.DynamicForm.getPrototype().getItemLocator,
        
        
        // getInnerElementFromSplit locator - override to check for the presence of items
        getItemFromSplitLocator:isc.DynamicForm.getPrototype().getItemFromSplitLocator,
        getInnerElementFromSplitLocator:function (locatorArray) {
            if (!this.emptyLocatorArray(locatorArray)) {
                var subItem = this.getItemFromSplitLocator(locatorArray);
                if (subItem != null) {
                    locatorArray.removeAt(0);
                    return subItem.getInnerElementFromSplitLocator(locatorArray);
                }
            }
            return this.Super("getInnerElementFromSplitLocator", arguments);
        }
    });
    
    
    isc.FormItem.addProperties({
        
        //> @attr FormItem.locateItemBy (string : null : IRWA)
        // When +link{isc.AutoTest.getElement()} is used to parse locator strings generated by
        // link{isc.AutoTest.getLocator()} for this form item, should the item be identified?
        // By default if the item has a name this will always be used, however for items with
        // no name, the following options are available:
        // <ul>
        // <li><code>"title"</code> use the title as an identifier within this form</li>
        // <li><code>"value"</code> use the value of the item to identify it (often used
        //  for items with a static defaultValue such as HeaderItems</li>
        // <li><code>"index"</code> use the index within the form's items array.
        // </ul>
        // 
        // If unset, and the item has no specified name, default behavior is to
        // identify by title (if available), otherwise by index.
        // @visibility external
        // @group autoTest
        //<
        
        getInnerElementFromSplitLocator : function (locatorArray) {
            if (!this.emptyLocatorArray(locatorArray)) {
                var part = locatorArray[0];
                if (part == "element") return this.getDataElement();
                if (part == "title") return this.form.getTitleCell(this);
                if (part == "textbox") return this._getTextBoxElement();
                if (part == "controltable") return this._getControlTabelElement();
                
                // canvasItems
                if (part == "canvas") {
                    if (this.canvas) {
                        locatorArray.removeAt(0);
                        return this.canvas.getElementFromSplitLocator(locatorArray);
                    }
                }
                
                // picker (EG date picker)
                if (part == "picker") {
                    if (this.picker) {
                        locatorArray.removeAt(0);
                        return this.picker.getElementFromSplitLocator(locatorArray);
                    }
                }
                
                // pickList
                if (part == "pickList") {
                    if (!this.pickList) this.makePickList(false);
                    locatorArray.removeAt(0);
                    return this.pickList.getElementFromSplitLocator(locatorArray);
                }
                
                // If passed an icon, return a pointer to the img element 
                // Event if there is a link element, it'll be above that in the DOM
                var iconSplit = part.match("\\[icon='(.+)'\\]"),
                    iconID = iconSplit ? iconSplit[1] : null;
                if (iconID) {
                    return this._getIconImgElement(iconID);
                }
            }
        },
        
        // copy the 'emptyLocatorArray()' helper function across
        emptyLocatorArray:isc.Canvas.getPrototype().emptyLocatorArray
    });
    
    
    isc.HeaderItem.addProperties({
        //> @attr HeaderItem.locateItemBy (string : "value" : IRWA)
        // Default to locating header items by value
        // @visibility autoTest
        //<
        locateItemBy:"value"
    });
    
    
    if (isc.PickListMenu) {
        isc.PickListMenu.addProperties({
            getLocatorParent : function () {
                if (this.formItem) return this.formItem.form;
                return this.Super("getLocatorParent", arguments);
            }
        });
    }
}


if (isc.GridRenderer) {
    
    isc.GridRenderer.addProperties({
        getInteriorLocator : function (element, fromEvent) {
            var cell = this.getCellFromDomElement(element);
            if (cell == null) return this.Super("getInteriorLocator", [element, fromEvent]);
            
            var rowNum = cell[0], colNum = cell[1];
            
            return this.getCellLocator(rowNum, colNum);
            
        },
        
        //> @method gridRenderer.getCellFromDomElement() [A]
        // Given a pointer to an element in the DOM, this method will check whether this
        // element is contained within a cell of the gridRenderer, and if so return a
        // 2 element array denoting the <code>[rowNum,colNum]</code> of the element
        // in question.
        // @param element (DOM element) DOM element to test
        // @return (Array) 2 element array containing rowNum and colNum, or null if the
        //   element is not contained in any cell in this gridRenderer
        // @group autoTest
        // @visibility external
        //<
        getCellFromDomElement : function (element) {
            var handle = this.getHandle(),
                table = this.getTableElement();
                
            if (!table) return null;
                
            var rows = table.rows,
                tagName,
                row, cell,
                tr = "tr", TR = "TR",
                td = "td", TD = "TD";
            
            while (element && element != table && element != handle) {
                
                tagName = element.tagName;           
                // document whether it's upper / lower case by default
                if (tagName == td || tagName == TD) {
                    cell = element;
                }
                
                // document whether it's upper / lower case by default
                if (tagName == tr || tagName == TR) {
                    row = element;
                }
                // keep going in case there are nested tables, etc
                element = element.parentNode;
            }
            if (!row || !cell) return null;
            
            var rows = table.rows, rowNum, logicalRowNum;
            for (var i = 0; i < rows.length; i++) {
                if (rows[i] == row) {
                    rowNum = i;
                    break;
                }
            }
            var cells = row.cells, colNum, logicalColNum;
            for (var i = 0; i < cells.length; i++) {
                if (cells[i] == cell) {
                    colNum = i;
                    break;
                }
            }
            logicalRowNum = rowNum + (this._firstDrawnRow || 0);
            logicalColNum = colNum + (this._firstDrawnCol || 0);
            
            return [logicalRowNum,logicalColNum];
        },
        
        getCellLocator : function (rowNum, colNum) {
            return "row[" + rowNum + "]/col[" + colNum + "]"
        },
        
        getInnerElementFromSplitLocator : function (locatorArray) {
            
            if (this.emptyLocatorArray(locatorArray)) return this.getHandle();
            
            // Format should be [row[index], col[index]]
            if (locatorArray.length == 2) {
                var cell = this.getCellFromLocator(locatorArray[0], locatorArray[1]),
                    rowNum = cell[0], colNum = cell[1];
                
                if (isc.isA.Number(rowNum) && isc.isA.Number(colNum)) {
                    // We suppress all events on row/cols during row animation
                    // in this case suppress the element entirely so auto-test engines
                    // don't attempt to fire events on them.
                    
                    if (this._suppressEventHandling()) return null;
            
                    return this.getTableElement(rowNum, colNum);
                }
            }
            return this.Super("getInnerElementFromSplitLocator", arguments);
        },
        
        // assumes rowLocator is row[rowNum]
        // colLocator is col[colNum]
        getCellFromLocator : function (rowLocator, colLocator) {
            // This is a straight parse - to support being passed a fuller format and
            // just extracting the index, if present, we'd want to have 
            // AutoTest.parseFallbackLocator run and then extract the standalone field value
            // knowing that's an index.
            var rowString = rowLocator.substring(4, rowLocator.length-1),
                colString = colLocator.substring(4, colLocator.length-1);
            return [rowNum,colNum];
        }
    })

}
if (isc.ListGrid) {
    isc.ListGrid.addProperties({
        // we explicitly set up the locatorParent pointers on these widgets
        // in ListGrid.js
        namedLocatorChildren:[
            "header", "frozenHeader", "body", "frozenBody", {attribute:"_editRowForm", name:"editRowForm"}
        ]
    });
      
      
    // We want to handle identifying cells by fieldName, record primary key etc as well
    // as simple rowNum / colNum.
    // We also need to handle the fact that with the option to freeze fields we can end up
    // with a logical cell that was in one sub-component (the frozen body, say)
    // is now in another (the standard body).
    
    // Implementation:
    // - when generating the Locator string include 'body' / 'frozenBody' as normal but
    //   have getCellLocator overridden in gridBody to record information about the fieldName etc
    //   as well as simple rowNum / colNum
    // - when parsing Locator strings, have the listGrid catch the case where we'd usually
    //   pass through to the body and handle it directly - figuring out which body the
    //   cell is in, and calling 'getTableElement()' on that
  
    isc.GridBody.addProperties({
  
        // override 'getInteriorLocator()' -- if an event occurred over an embedded component such
        // as a rollOverCanvas with eventProxy pointing back to us, we can't rely on the
        // DOM element
        // In the case where we're getting a locator from the event actually handle this by getting
        // coordinates from the event
        
        getInteriorLocator : function (element, fromEvent) {
            if (fromEvent) {
                var children = this.children;
                if (children != null && children.length > 0) {
                    for (var i = 0; i < children.length; i++) {
                        var child = children[i];
                        if (child && child.eventProxy == this) {
                            var handle = child.getHandle();
                            if (handle != null) {
                                var testElement = element;
                                while (testElement != this.getHandle() && testElement != null) 
                                {
                                    if (testElement == handle) {
                                        var rowNum = this.getEventRow(),
                                            colNum = this.getEventColumn();
                                        return this.getCellLocator(rowNum,colNum);
                                        
                                    }
                                    testElement = testElement.parentNode;
                                }
                            }
                        }
                    }
                }
            }
            return this.Super("getInteriorLocator", arguments);
        },
        
        getCellLocator : function (rowNum, colNum) {
            var grid = this.grid;
            if (grid == null) return this.Super("getCellLocator", arguments);
            return grid.getCellLocator(this, rowNum, colNum);
        }
              
    });
    
    isc.ListGrid.addProperties({
        // getCellLocator -- called by the grid body to generate the identifier
        getCellLocator : function (body, rowNum, colNum) {
            var rowLocatorOptions = this.getRowLocatorOptions(body, rowNum, colNum),
                colLocatorOptions = this.getColLocatorOptions(body, rowNum, colNum);
            return isc.AutoTest.createLocatorFallbackPath("row", rowLocatorOptions) +
                    "/" + isc.AutoTest.createLocatorFallbackPath("col", colLocatorOptions);
        },
        
        // builds a config type object that we'll pass to createLocatorFallbackPath
        getRowLocatorOptions : function (body, rowNum, colNum) {
            
            var locatorOptions = {},
                gridColNum = this.getFieldNumFromLocal(colNum, body),
                record = this.getCellRecord(rowNum, gridColNum),
                ds = this.getDataSource();
                
            if (record != null) {
                if (ds != null) {
                    var pk = ds.getPrimaryKeyFieldName();
                    if (pk != null && record[pk] != null) {
                        locatorOptions[pk] = record[pk];
                    }
                }
                
                var titleField = this.getTitleField();
                if (titleField != null && record[titleField] != null) {
                    locatorOptions[titleField] = record[titleField];
                }
                var fieldName = this.getFieldName(gridColNum);
                if (fieldName != null && record[fieldName] != null) {
                    locatorOptions[fieldName] = record[fieldName];
                }
            }
            // also store the rowNum
            locatorOptions[isc.AutoTest.fallback_valueOnlyField] = rowNum;
            
            return locatorOptions;
        },
        
        getColLocatorOptions : function (body, rowNum, colNum) {
            var locatorOptions = {},
                gridColNum = this.getFieldNumFromLocal(colNum, body);
            var field = this.getField(gridColNum);
            if (this.isCheckboxField(field)) {
                locatorOptions.isCheckboxField = true;
            } else {
                var fieldName = this.getFieldName(gridColNum);
                if (fieldName != null) locatorOptions.fieldName = fieldName;
            }
            locatorOptions[isc.AutoTest.fallback_valueOnlyField] = colNum;
            return locatorOptions;  
            
        },
        
        
        // if the child substring is "frozenBody' / "body", return null - we'll handle
        // finding the element at the ListGrid level
        getChildFromLocatorSubstring : function (substring, index, locatorArray) {
            if (substring == "frozenBody" || substring == "body") {
                // actually do check for the case where we're looking for a cell within the
                // body - we don't want to avoid normal handling if we're looking for
                // the body handle itself, for example
                if (locatorArray.length == index+3 &&
                    locatorArray[index+1].startsWith("row[") &&
                    locatorArray[index+2].startsWith("col[")) 
                {
                    return null;
                }
            }
            return this.Super("getChildFromLocatorSubstring", arguments);
        },
        
        // Override getInnerElementFromSplitLocator to handle cells in the body/frozenBody
        getInnerElementFromSplitLocator : function (locatorArray) {
            
            if (this.emptyLocatorArray(locatorArray)) return this.getHandle();
            
            // expected format: "frozenBody", row[...], col[...]"
            var body = locatorArray[0];
                
            if (locatorArray.length == 3 && (body == "body" || body == "frozenBody")) {
                // Start with the field!
                var colLocator = locatorArray[2],
                    colLocatorConfig = isc.AutoTest.parseLocatorFallbackPath(colLocator);

                // colLocatorConfig will have name:"col", config:{config object}
                // The 'getChildFromLocatorSubstring() method already checks for this but
                // as a sanity check verify the name of the col locator
                if (colLocatorConfig.name != "col") {
                    this.logWarn("Error parsing locator:" + locatorArray.join("") + 
                        " returning ListGrid handle");
                    return this.getHandle();
                }
                
                var field = this.getFieldFromColLocatorConfig(colLocatorConfig.config),
                    localColNum;
                // If no fieldName stored, use the previous colNum instead
                // [we stored the colNum relative to the body in question]
                if (field == null) {
                    localColNum = parseInt(colLocatorConfig.config[isc.AutoTest.fallback_valueOnlyField]);
                    
                    if (body == "frozenBody" && this.frozenBody == null) {
                        body = "body";
                    }
                    // convert to string to a pointer to the widget
                    body = this[body];
                } else {
                    localColNum = this.getLocalFieldNum(this.getFieldNum(field)); 
                    
                    if (this.fieldIsFrozen(field)) body = this.frozenBody;
                    else body = this.body;
                }
                // Bail if we haven't created the right body for some reason
                
                if (body == null) return null;
                
                // At this point we know what body it's in and what the colNum is within that
                // body.
                // Now find the row
                
                var rowLocator = locatorArray[1],
                    rowLocatorConfig = isc.AutoTest.parseLocatorFallbackPath(rowLocator),
                    rowNum = this.getRowNumFromLocatorConfig(rowLocatorConfig.config);
                
                if (isc.isA.Number(rowNum) && isc.isA.Number(localColNum)) {
                    // We suppress all events on row/cols during row animation
                    // in this case suppress the element entirely so auto-test engines
                    // don't attempt to fire events on them.
                    
                    if (body._suppressEventHandling()) return null;
            
                    return body.getTableElement(rowNum, localColNum);
                }
            }
            
            return this.Super("getInnerElementFromSplitLocator", arguments);
        },
        
        // helper to pick up field based on 'checkboxField' status and name
        // If neither work, we will use field num instead
        getFieldFromColLocatorConfig : function (colConfig) {
            //this.logWarn("colConfig:" + this.echo(colConfig));
            if (colConfig.isCheckboxField != null) {
            
                for (var i = 0; i < this.fields.length; i++) {
                    if (this.isCheckboxField(this.fields[i])) {
                        return this.fields[i];
                    }
                    // In this case we didn't find a checkbox field - test is probably
                    // invalid
                    this.logWarn("AutoTest stored a locator for interaction with " +
                            "checkbox field - but this grid is not showing a checkbox field - " +
                            "recorded test may be invalid.", "AutoTest");
                    // returning -1 here - this causes use to not return some other random
                    // unrelated cell (typically the first column in the grid)
                    return -1;
                }
            } else {
       
                var locateColsBy = this.locateColumnsBy;
                //locateColsBy will be one of ("fieldName", "index")    
    
                if (locateColsBy == "fieldName" || locateColsBy == null) {
                    var fieldName = colConfig.fieldName;
                    if (fieldName != null) {
                        return this.getField(fieldName);
                    }
                }
            }
        },
        
        
        getRowNumFromLocatorConfig : function (rowConfig) {
            //this.logWarn("rowConfig:" + this.echo(rowConfig));
            var locateRowsBy = this.locateRowsBy;
       
            if (locateRowsBy == null) locateRowsBy = "primaryKey";
            switch(locateRowsBy) {
                case "primaryKey":
                    //this.logWarn("trying to locate by pk");
                    var ds = this.getDataSource();
                    if (ds != null) {
                        var pkField = ds.getPrimaryKeyFieldName();
                        if (ds != null && rowConfig[pkField] != null) {
                            return this.findRowNum(rowConfig);
                        }
                    }
                    // don't break - if we were unable to use PK, fall back through
                    // titleField / cell value before index
                        
                case "titleField":
                    //this.logWarn("trying to locate by title field");
                    var titleField = this.getTitleField();
                    if (titleField != null && rowConfig[titleField] != null) {
                        var data = this.data;
                        return data.findIndex(titleField, rowConfig[titleField]);
                    }
                    
                case "targetCellValue":
                    //this.logWarn("trying to locate by target");
                    // Assertion: In this case, there was no titleField or primary key
                    // on the config object.
                    // This relies on the fact that we wouldn't store "null"s on that object
                    // when creating the locator options.
                    // All that's left is the original index under the fallback_valueOnlyField
                    // array and the target row cell value
                    for (var fieldName in rowConfig) {
                        if (fieldName == isc.AutoTest.fallback_valueOnlyField) continue;
                        
                        if (rowConfig[fieldName] != null) {
                            return this.data.findIndex(fieldName, rowConfig[fieldName]);
                        }
                    }
                default:
                    //this.logWarn("locate by rowNum");
                    // Final fallback option- original rowNum as stored
                    // Technically this is locateRowsBy "index"
                    return parseInt(rowConfig[isc.AutoTest.fallback_valueOnlyField]);
            }
        }
    });
}


if (isc.TreeGrid) {
    isc.TreeGridBody.addProperties({
        getInteriorLocator : function (element) {
            var origElement = element;
            
            var handle = this.getHandle(),
                tableElement = this.getTableElement();
                
            if (!element || !handle || !tableElement) return isc.emptyString;
            var openAreaPrefix = this.grid.getCanvasName() + this.grid._openIconIDPrefix,
                rowNum, colNum;
                
            // The checkbox icon shows in the "extra icon" slot so
            // we'll have one or the other (not both) and can just store "extraIcon" as an 
            // identifier
            var extraIconPrefix = this.grid.getCanvasName() + this.grid._extraIconIDPrefix;
                
          
            // optimization - we could duplicate the logic from GR here and avoid double-iterating
            // through the DOM if we're NOT in the open area of the TG.
            while (element != this.tableElement && element != handle && element.getAttribute) {
                // check the "name" property for the open-icon 
                var ID = element.getAttribute("name");
                if (ID) {
                    if (ID.startsWith(openAreaPrefix)) {
                        rowNum =  parseInt(ID.substring(openAreaPrefix.length));
                        colNum = this.grid.getTreeFieldNum();
                        
                        return this.getCellLocator(rowNum,colNum) + "/open";
                    }
                    if (ID.startsWith(extraIconPrefix)) {
                        rowNum =  parseInt(ID.substring(extraIconPrefix.length));
                        colNum = this.grid.getTreeFieldNum();
                        return this.getCellLocator(rowNum,colNum) + "/extra";
                    }
                }
                element = element.parentNode;
            }
            
            return this.Super("getInteriorLocator", [origElement]);
        },
        
        
          
        getInnerElementFromSplitLocator : function (locatorArray) {
            
            if (this.emptyLocatorArray(locatorArray)) return this.getHandle();
            
            // Additional Format is: [row[index], col[index], open]
            if (locatorArray.length == 3) {
                if (locatorArray[2] == "open") {
                    // We suppress all events on row/cols during row animation
                    // Also suppress toggleFolder event target in this case.
                    
                    if (this._suppressEventHandling()) return null;
                    
                    
                    var rowLocator = locatorArray[0];
                    var rowNum;
                    
                    //   old format was row3
                    // new format is a standard row locator like
                    //   row[pkFieldValue=foo|3]
                    // Test for old format explicitly since parseLocatorFallbackPath doesn't
                    // handle it.
                    if (rowLocator.charAt(3) != "[") {
                        rowNum = parseInt(rowLocator.substring(3))
                    } else {
                        var rowLocatorConfig = isc.AutoTest.parseLocatorFallbackPath(rowLocator);
                        if (rowLocatorConfig == null || rowLocatorConfig.name != "row") {
                            
                            this.logInfo("Locator appears to be click-in-open-area locator but " +
                                "doesn't contain row/col info? returning null.\n" 
                                + locatorArray.join("/"),
                                "AutoTest");
                            
                        }
                        rowNum = this.grid.getRowNumFromLocatorConfig(rowLocatorConfig.config);
                    }
                    // we recorded the colNum but we don't need it!
                    //var colNum = this.grid.getTreeFieldNum();
    
                    // use getImage since we write a name into the opener icon.
                    var openerID = this.grid._openIconIDPrefix + rowNum,
                        image = this.grid.getImage(openerID);
                    if (image) return image;
                    
                // exactly the same logic for the "extraIcon", which is also used for
                // the checkbox icon when doing checkbox / cascading selection
                } else if (locatorArray[2] == "extra") {
                    if (this._suppressEventHandling()) return null;
                                
                    
                     
                    var rowLocator = locatorArray[0];
                    var rowNum;
                    
                    //   old format was row3
                    // new format is a standard row locator like
                    //   row[pkFieldValue=foo|3]
                    // Test for old format explicitly since parseLocatorFallbackPath doesn't
                    // handle it.
                    if (rowLocator.charAt(3) != "[") {
                        rowNum = parseInt(rowLocator.substring(3))
                    } else {
                        var rowLocatorConfig = isc.AutoTest.parseLocatorFallbackPath(rowLocator);
                        if (rowLocatorConfig == null || rowLocatorConfig.name != "row") {
                            
                            this.logInfo("Locator appears to be click-in-open-area locator but " +
                                "doesn't contain row/col info? returning null.\n" 
                                + locatorArray.join("/"),
                                "AutoTest");
                            
                        }
                        rowNum = this.grid.getRowNumFromLocatorConfig(rowLocatorConfig.config);
                    }
                    // we recorded the colNum but we don't need it!
                    //var colNum = this.grid.getTreeFieldNum();
    
                    // use getImage since we write a name into the opener icon.
                    var openerID = this.grid._extraIconIDPrefix + rowNum,
                        image = this.grid.getImage(openerID);
                    if (image) return image;
                
                }
            }
            return this.Super("getInnerElementFromSplitLocator", arguments);
        },
        
        getAutoTestLocatorCoords : function (locator, element) {
            
            
            var coords = this.Super("getAutoTestLocatorCoords", arguments);
            if (coords == null) return coords;
            
            var tg = this.grid;
            // if we're picking up either icon (opener or other icon) coords will be position
            // of icon so return it
            // Otherwise, if the element is a cell in the tree field we need to modify the coords
            // to be beyond the icons to avoid triggering an open/close.
            if (tg == null || locator.endsWith("open") || locator.endsWith("extra")) return coords;
            
            var y = coords[1],
                rowNum = this.getEventRow(y),
                colNum = this.getEventColumn(coords[0]),
                
                data = tg.data,
                node = tg.getRecord(rowNum),
                isTreeField = tg.getTreeFieldNum() == tg.getFieldNumFromLocal(colNum, this);
            
            if (isTreeField && tg.data && tg.data.isFolder(node)) {
                // use the openAreaWidth calculation already performed by the TreeGrid
                // and put the event in the middle of the remaining space in the row.
                var openAreaWidth = tg.getOpenAreaWidth(node),
                    rect = isc.Element.getElementRect(element),
                    left = (rect[0] +openAreaWidth),
                    width = rect[2] - left;
                
                coords[0] = left+Math.floor(width/2);
            }
            return coords;

        }
    })
}





// TabSets:
// We want to be able to locate tabs by ID or title rather than just index so if the order
// changes they continue to be accessable
if (isc.TabSet) {
    isc.TabSet.addProperties({
        
        // Relevant logic outside this file:
        //
        // In TabSet: tabBarControls layout has locatorParent / namedLocatorChildren set such that
        // the tabset will point directly to that auto-child by name.
        
        
        // In TabBar we have logic in makeButton to set 'locatorParent' on tabs to point
        // straight to the TabSet
        
        // Need to update containsLocatorChild / getStandardChildLocator / ... ? _locatorChildren??
        containsLocatorChild : function (canvas) {
            if (this.Super("containsLocatorChild", arguments)) return true;
            
            if (this.getTabNumber(canvas) != -1) return true;
            return false;
        },
        
        getStandardChildLocator : function (canvas) {
            var tabNum = this.getTabNumber(canvas);
            if (tabNum != -1) {
                var tabObj = this.getTabObject(tabNum);
                
                var locatorConfig = {};
                // locate by ID, title or index
                if (tabObj.ID != null) locatorConfig.ID = tabObj.ID;
                if (tabObj.title != null) locatorConfig.title = tabObj.title;
                locatorConfig.index = tabNum;
                
                return isc.AutoTest.createLocatorFallbackPath("tab", locatorConfig);
            
            }
            return this.Super("getStandardChildLocator", arguments);
        },
        
        //> @attr TabSet.locateTabsBy (string : null : IRWA)
        // When +link{isc.AutoTest.getElement()} is used to parse locator strings generated by
        // link{isc.AutoTest.getLocator()}, how should tabs within this tabset be identified?
        // By default if tab has a specified +link{Tab.ID} this will always be used.
        // For tabs with no ID, the following options are available:
        // <ul>
        // <li><code>"title"</code> use the title as an identifier</li>
        // <li><code>"index"</code> use the index of the tab in the tabset as an identifier</li>
        // </ul>
        // 
        // If unset, and the tab has no specified ID, default behavior is to
        // identify by title (if available), otherwise by index.
        // @visibility external
        // @group autoTest
        //<
        getChildFromLocatorSubstring : function (substring) {
            
            // this startsWith("tab[") is a bit of a hack we will probably just pass the
            // substring to AutoTest.parseLocatorFallbackPath directly and look at the returned
            // 'name' property -- however not sure if it'll handle all formats right now.
            if (substring && substring.startsWith("tab[")) {
                var fallbackConfig = isc.AutoTest.parseLocatorFallbackPath(substring),
                    config = fallbackConfig.config;
                
                // If ID is present, always respect it:
                
                if (config.ID != null) {
                    return this.getTab(config.ID);
                }
                var locateTabsBy = this.locateTabsBy;
                if (locateTabsBy == null) locateTabsBy = "title";
                
                if (config.title && locateTabsBy == "title") {
                    var tabNum = this.tabs.findIndex("title", config.title);
                    return this.getTab(tabNum);
                }
                // last case -- we want to use the raw tab index.
                return this.getTab(parseInt(config.index));
            }
            
            return this.Super("getChildFromLocatorSubstring", arguments);
        }
        
    });
}

// ----------------------------------------------
// Returning element from interior locator

if (isc.StatefulCanvas) {
    isc.StatefulCanvas.addProperties({
          
        getInnerElementFromSplitLocator : function (locatorArray) {
            // label floats over statefulCanvas - if we have a specified part, assume it occurred
            // in the label since that's where we write out our icon, etc.
            if (!this.emptyLocatorArray(locatorArray) && this.label) {
                return this.label.getInnerElementFromSplitLocator(locatorArray);
            }
            return this.Super("getInnerElementFromSplitLocator", arguments);    
        }
    });
}



// DateChooser

if (isc.DateChooser) {
    
    
    isc.DateChooser.addMethods({
        
        getInteriorLocator : function (element) {
            
            // We don't write any kind of unique DOM IDs or attrs to our buttons which
            // would simplify determining the purpose of the buttons except our click handler.
            
            // 2 possible approaches:
            // 1) Crudely look at the current cell position in whichever table its in - likely to 
            //    break if we rework ... wait a sec
            
            // Ok so playback / record type stuff
            // 
            // Lets say we store this as prevYear, prevMo, moLauncher, nextMo, nextYear and
            // dateCell[rowNum,colNum] and today, cancel
            
            // we can then get back the appropriate button when rerun
            // however if the default date has changed (which it likely will, tests are unlikely 
            // to work unless the user changes to a specific month first....
            
            
            // Alternative would be to remember dateCell[datestamp] - then we simply won't find the element if the date changes. Ok that seems better - won't get WRONG behavior
            
            // If we have a header, the event may have occurred in it
            var handle = this.getHandle();
            if (!handle || !element) return "";
            
            var cachedString = element._cachedLocatorString;
            if (cachedString != null && cachedString != "") return cachedString;
            
            return element._cachedLocatorString = this._getInteriorLocator(element, handle);
        },
        
        _getInteriorLocator : function (element, handle) {
            var targetCell = element;
            while (targetCell && targetCell != null) {
                if (targetCell == handle) {
                    targetCell = null;
                    break;
                }
                if (targetCell.tagName && targetCell.tagName.toLowerCase() == "td") {
                    break;
                }
                targetCell = targetCell.parentElement;
            }
            if (targetCell == null) return "";
            
            
            var childNodes = handle.childNodes,
                tables = [];
            for (var i = 0; i < childNodes.length; i++) {
                if (!childNodes[i].tagName || childNodes[i].tagName.toLowerCase() != "table") {
                    continue;
                }
                tables[tables.length] = childNodes[i];
            }
            
            var headerTable = tables.length == 2 ? tables[0] : null,
                bodyTable = tables.length == 2 ? tables[1] : tables[0];
            
            if (headerTable != null && targetCell.offsetParent == headerTable) {
                // could look at position within rows array -- but then we'd have to also
                // look at the various 'showMonthChooser' etc configurations -- instead
                // lets look directly at the onclick handler
                var clickFunction = targetCell.onclick,
                    clickFunctionString = clickFunction ? clickFunction.toString() : null;
                if (!clickFunctionString) return "";
                if (clickFunctionString.contains("showPrevYear")) {
                    return "prevYearButton";
                } else if (clickFunctionString.contains("showNextYear")) {
                    return "nextYearButton";
                } else if (clickFunctionString.contains("showPrevMonth")) {
                    return "prevMonthButton";
                } else if (clickFunctionString.contains("showNextMonth")) {
                    return "nextMonthButton";
                } else if (clickFunctionString.contains("showMonthMenu")) {
                    return "monthMenuButton";
                } else if (clickFunctionString.contains("showYearMenu")) {
                    return "yearMenuButton";
                }
                return "";
                
            } else if (bodyTable != null && targetCell.offsetParent == bodyTable) {
                // If the event was in the body, return the appropriate dateClick string
                var clickFunction = targetCell.onclick,
                    clickFunctionString = clickFunction ? clickFunction.toString() : null;
                if (!clickFunctionString) return "";
                
                if (clickFunctionString.contains("cancelClick")) return "cancelButton";
                else if (clickFunctionString.contains("todayClick")) return "todayButton";
                else {
                    var dateClick = clickFunctionString.match("dateClick\\(\(.*\)\\)");
                    if (dateClick && dateClick[1]) {
                        var dateArr = dateClick[1].split(",");
                        for (var i = 0; i < dateArr.length; i++) {
                            dateArr[i] = dateArr[i].trim();
                        }        
                        return dateArr.join("/");
                    }
                }
            }
            return "";
        },
        
        
        getInnerElementFromSplitLocator : function (locatorArray) {

            if (this.emptyLocatorArray(locatorArray)) return this.getHandle();

            var handle = this.getHandle();
            if (handle == null) return;
            
            var isDateButton = (locatorArray.length == 3);
            if (!isDateButton) {
                
                var locatorString = locatorArray[0];
                
                if (locatorString == "") return handle;
                
                var isTodayButton = (locatorString == "todayButton"),
                    isCancelButton = !isTodayButton ? (locatorString == "cancelButton") : false;
                    
                var childNodes = handle.childNodes;
                    
                // today / cancel button show up in the "body" table
                if (isTodayButton || isCancelButton) {
                        
                    if (isTodayButton && !this.showTodayButton) {
                        this.logWarn("DateChooser attempting to locate element for " +
                            "'todayButton' but showTodayButton is false. Returning handle.",
                            "AutoTest");
                        return handle;
                    }
                    if (isCancelButton && !this.showCancelButton) {
                        this.logWarn("DateChooser attempting to locate element for " +
                            "'cancelButton' but showCancelButton is false. Returning handle.",
                            "AutoTest");
                        return handle;
                    }
                    
                    var bodyTable;
                    // we show two tables if the header is showing, or just one if not.
                    // Either way the table we want is the last table in the handle.
                    for (var i = childNodes.length-1; i >= 0; i--) {
                        if (childNodes[i].tagName && 
                             childNodes[i].tagName.toLowerCase() == "table") 
                        {
                            bodyTable = childNodes[i];
                            break;
                        }
                    }
                    
                    // today/cancel button cells are in the last row of the table
                    var lastRow = bodyTable.rows[bodyTable.rows.length-1],
                        cells = lastRow.cells;
                    for (var i = 0; i < cells.length; i++) {
                        if (this.getInteriorLocator(cells[i]) == locatorString) {
                            return cells[i];
                        }
                    }
                    
                } else {
                    
                    // Other buttons show up in the header table
                    if (!this.showHeader) {
                        this.logWarn("DateChooser attempting to locate element for " + locatorArray +
                          " but this.showHeader is false so this element will not be present. " +
                          "Returning handle.", "AutoTest");
                        return handle;
                    }
                    
                    var headerTable
                        // we show two tables if the header is showing, so grab the first table in the
                        // childNodes array
                        for (var i = 0; i < childNodes.length; i++) {
                            if (childNodes[i].tagName && 
                                 childNodes[i].tagName.toLowerCase() == "table") 
                            {
                                headerTable = childNodes[i];
                                break;
                            }
                        }
                        
                        // controls show up in the first row of cells
                        var row = headerTable.rows[0],
                            cells = row.cells;
                        for (var i = 0; i < cells.length; i++) {
                             if (this.getInteriorLocator(cells[i]) == locatorString) {
                                return cells[i];
                            }
                        }
                }
                    

            // Date Buttons. Only releveant if we're showing the date in question!
            } else {
                
                // If we're showing a different year, obviously we're not showing the date button
                var year = locatorArray[0],
                    month = locatorArray[1],
                    date = locatorArray[2];
                // month may differ but only for the few 'spillover' days at the beginning/end
                // of the week - so if the month is off by more than one we're not showing the
                // button
                if ((year == this.year) &&
                        (this.month == month || this.month == month+1 || this.month == month-1))
                {
                    // We could iterate through all the visible buttons looking at locators and
                    // see if they match, or we could figure out the rowNum/colNum in which
                    // the date will be showing (if it is) and pick the cell that way.
                    // We'll take the second approach
                    var buttonDate = new Date(year,month,date),
                        buttonDay = buttonDate.getDay();
                        
                    // only continue if we're showing weekends, or the button doesn't fall on 
                    // a weekend
                    if (this.showWeekends || !Date.getWeekendDays().contains(buttonDay)) {
                        
                        // figure out the first date we have a cell for
                        var start = new Date(this.year, this.month, 1);
                            // go back to the first day of the week for the start date
                            
                            var startDay = start.getDay(),
                                startOffset = startDay + this.firstDayOfWeek -
                                            // start date may have a lower "day number"
                                            // (sun=0 thru sat=6) than firstDayOfWeek,
                                            // in which case we need to adjust back by a week
                                            (startDay < this.firstDayOfWeek ? 7 : 0);
                                              
                            start.setDate(start.getDate() - startOffset);
                                        

                        // if the date is earlier than the first cell we're showing, we
                        // don't have a cell for it
                        // (this != comparison checks for opposite case - where we should continue)
                        if (Date.compareDates(buttonDate, start) != 1) {
                            
                            // get a pointer to the last day in the visible month
                            var end = new Date(this.year, this.month+1, 1);
                            end.setTime(end.getTime() - 86400000);
                            
                            // note that we show a few extra buttons for rest of the week
                            // (potentially)
                            var endDOW = end.getDay(),
                                lastDOW = this.firstDayOfWeek + 6;
                            if (lastDOW > 6) lastDOW -= 7;
                            
                            var dayDelta = lastDOW > endDOW ? 
                                                lastDOW-endDOW : lastDOW+7 - endDOW;
                            if (dayDelta != 0) {
                                end.setTime(end.getTime() + (86400000*dayDelta));
                            }
                            
                            // if the buttonDate is <= 'end' we are showing a cell for it
                            if (Date.compareDates(buttonDate, end) != -1) {
                                // rowNum will be day-delta between button date and first 
                                // visible date / 7

                                var rowNum = Math.floor( ((parseInt(date) + startOffset) / 7)) 
                                                           
                                // we always show day of week headers in the first row.
                                
                                rowNum += 1;
                                var firstDay = this.firstDayOfWeek;
                                if (!this.showWeekends) {
                                    while (Date.getWeekendDays().contains(firstDay)) {
                                        firstDay += 1;
                                        if (firstDay == 7) firstDay = 0;
                                    }
                                }
                                var colNum = buttonDate.getDay() - firstDay;
                                if (colNum < 0) colNum += 7;
                                // Ok - we have a rowNum/colNum
                                var childNodes = handle.childNodes,
                                    bodyTable;
                                // we show two tables if the header is showing, or just one if not.
                                // Either way the table we want is the last table in the handle.
                                for (var i = childNodes.length-1; i >= 0; i--) {
                                    if (childNodes[i].tagName && 
                                         childNodes[i].tagName.toLowerCase() == "table") 
                                    {
                                        bodyTable = childNodes[i];
                                        break;
                                    }
                                }                                
                                if (bodyTable) return bodyTable.rows[rowNum].cells[colNum];
                            } else {
                                this.logInfo("DateChooser Passed ID for a date after end. " +
                                    "end date:"+ [end.getFullYear(), end.getMonth(),
                                                    end.getDay()] +
                                    " vs:" + [year, month, date], "AutoTest");
                            }
                        } else {
                            this.logInfo("DateChooser Passed ID for a date before start date. " +
                            "startDate:"+ [start.getFullYear(), start.getMonth(), start.getDay()] +
                            " vs:" + [year, month, date], "AutoTest");
                        }
                    } else {
                        this.logInfo("DateChooser Passed ID for a weekend - not showing weekends", "AutoTest");
                    }
                } else {
                    this.logInfo("DateChooser passed ID for the wrong year or month - passed:" + 
                        locatorArray + ", showing:" + [this.year,this.month], "AutoTest");
                }
                
                this.logWarn("DateChooser - passed inner locator for date (" +
                            locatorArray.join("/") + ") -- not currently showing this date.",
                            "AutoTest");
            }
                        
            this.logWarn("DateChooser, unable to find element for inner locator:"+
                locatorArray + " returning handle");
            return handle;
        }
    });
}

}


// We want to respond to interaction with calendar events based on event name and 
// potentially date /title.
// Cases to handle:
// - interacting with cells in the 3 standard ListGrid views (day, week, month)
// - interacting with event links within cells in the month view
// - interacting with eventWindow auto-children associated with existing events

// Note we also need to handle interaction with various auto-children -- the date picker,
// prev/next buttons, etc. These should be handled by the standard "single auto child" 
// subsystem rather than needing any special logic.

// Putting this into a method (customizeCalendar()) means we can call this at the end
// of Calendar.js rather than having to worry about whether the module has been loaded or not.
isc.AutoTest.customizeCalendar = function () {
    
    
    // locateCellsBy
    //  - date
    //      - implies date AND time
    //  - index
    //      - rowNum/colNum
    
    
    // locateEventsBy
    //  - name
    //  - title
    //  ? event type
    //  - startDate
    isc._commonCalenderViewFunctions = {
            
        // Override the method to set up the 'rowLocator' - this should store
        // date and time and use that for preference over other locators
        // Note: we're leaving the columns alone here: for the day view we show only
        // two columns -- the label and the day column -- we'll already identify the correct
        // one based on field name
            
        // builds a config type object that we'll pass to createLocatorFallbackPath
        getRowLocatorOptions : function (body, rowNum, colNum) {
            
            // Pick up standard options - this will get the rowNum for us
            // (Other options, such as primary key don't have much use here)
            var options = this.Super("getRowLocatorOptions", arguments);
            
            var date = this.creator.chosenDate;
            options.date = date.toSchemaDate("date");
            
            // time always starts at 12am
            // We show 2 rows per hour.
            // so just count rows to get time
            options.minutes = rowNum * 30;
            return options;
            
        },
        
        
        // parse a stored locator configuration back to the appropriate cell
        // If we're identifying by date, use the stored date / minutes
        // otherwise just use index
        getRowNumFromLocatorConfig : function (rowConfig) {
            var locateCellsBy = this.creator.locateCellsBy;
            if ((locateCellsBy == "date" || locateCellsBy == null) &&
                rowConfig.date != null)
            {
                var date = isc.Date.parseSchemaDate(rowConfig.date);
                if (!this.showingDate(date)) {
                    this.logWarn("Locator for cell in this calendar day-view grid has date " +
                        "stored as:" + date.toUSShortDate() + ", but we're currently showing " +
                        this.creator.chosenDate.toUSShortDate() +
                        ". The stored date doesn't map to a visible cell so not returning a cell " +
                        "- if this is not the intended behavior in this test case you may need to " +
                        "set calendar.locateCellsBy to 'index'.", "AutoTest");
                    return -1;
                }
                // map the stored minutes to the appropriate rowNum
                
                return parseInt(rowConfig.minutes) / 30;
            }
            this.locateRowsBy = "index";
            return this.Super("getRowNumFromLocatorConfig", arguments);
        },
        
        showingDate : function (date) {
            return (isc.Date.compareLogicalDates(date, this.creator.chosenDate) == 0);
        }
    }
    isc.DaySchedule.addProperties(isc._commonCalenderViewFunctions);
    
    // WeekView - has fields for each day of the week (plus the label field)
    // field names are arbitrary ("day1", "day2" etc, not mapping to days of week).
    // However field objects have year, month, day stored as _yearNum, _dateNum, _monthNum
    // so we don't need to calculate based on location, etc
    isc.WeekSchedule.addProperties(isc._commonCalenderViewFunctions,{
            
        // override 'showingDate' -- we show a range of dates (a week's worth)
        // we could look at this.creator.chosenDate again but seems like it'd be easier just
        // to check the date values already stored on each visible field
        showingDate : function (date) {
            for (var i = 0; i < this.fields.length; i++) {
                var field = this.fields[i];
                if (field._yearNum == null) continue;
                if (Date.compareLogicalDates(
                        new Date(field._yearNum, field._monthNum, field._dateNum),
                        date
                    ) == 0) 
                {
                    this.logWarn("does contain date" + date.toShortDate());
                    return true;
                }
                this.logWarn("date passed in:" + date.toShortDate() +
                    "compared with:" + new Date(field._yearNum, field._monthNum, field._dateNum).toShortDate());
            } 
            
            this.logWarn("doesn't contain date:" + date);
            return false;
        },
        
        
        // Month view has meaningful fields - each column is one day
        // Store date information on our column locators and use it when
        // retrieving columns
        getColLocatorOptions : function (body, rowNum, colNum) {
            
            var locatorOptions = this.Super("getColLocatorOptions", arguments),
                gridColNum = this.getFieldNumFromLocal(colNum, body),
                field = this.getField(gridColNum);                
            // the label field has no associated date, of course
            if (field && field._dateNum != null) {
                // the month is zero based - add one to it so it looks like the schema date
                // not really necessary but that way the date on the rowNum (derived from
                // this.chosenDate, using getSchemaDate()) will match the
                // date on the colNum in the locator string!
                locatorOptions.date = [field._yearNum, (field._monthNum+1), field._dateNum].join("-");
            }
            
            return locatorOptions;
        },
        
        // helper to pick up field based on 'checkboxField' status and name
        // If neither work, we will use field num instead
        getFieldFromColLocatorConfig : function (colConfig) {
            
            if ((this.locateCellsBy == "date" || this.locateCellsBy == null) &&
                (colConfig.date != null)) 
            {
                var dateArr = colConfig.date.split("-");
                // we can ignore the month and year - if the chosen date wasn't already in
                // the range, rowNum will be -1 anyway so we won't return a cell.
                return this.getFields().find("_dateNum", dateArr[2]);
                
            }
            
            return this.Super("getFieldFromColLocatorConfig", arguments);
        }
    });

    // Month view:
    // MonthSchedule is a subclass of ListGrid as well - it shows one column per day of the
    // week, and 2 rows per week -- one row is the header containing date values
    // second row is the actual events
    // Events are embedded in the cells as link elements
    // We'll need to react to
    //  - click on header rows (goes to day view)
    //  - click on empty cells (shows window to add an event)
    //  - click on stored event links (shows window to edit event)
    // Once again we'll use locateCellsBy "date" to find cells
    // If set to index we'll just set locateRowsByIndex and let standard handling occur
    // for the month
    
    // each field is named 'day1' [, 'day2', ...]
    // Each record has a 'day1' value which matches that of the field header, and a
    //  date1 value which actually specifies the date the row represents
    // The "1", "2", etc is specified by looking at field._dayIndex
    //
    // So the 'date1' value in the first row (a header row) matches the 'date1' value in the
    // second row (an 'event' row), and is the date we're showing in the 'day1' column (the
    // first column) and so on...
    
    // rows that are actual dates have an events array attached to them -- usually empty
    // rows that are not actual dates (so header rows) have no events array
    
    // Events within Month cells:
    // We record info about the event rather than the cell (essentially the date) it's located
    // in for event-links within month cells.
    // When we attempt to find the event links we can therefore have the Calendar find the
    // event and then try to find the link associated with that event in our view.
    
    // Event links call 'monthViewEventClick(rowNum,colNum,index)' on the calendar, so we
    // will parse this href string to determine which event is being interacted with...
    
    isc.MonthSchedule.addProperties({
            
        getRowLocatorOptions : function (body, rowNum, colNum) {
            
            // Pick up standard options - this will get the rowNum for us
            // (Other options, such as primary key don't have much use here)
            var options = this.Super("getRowLocatorOptions", arguments);
            var record = this.getRecord(rowNum);
            if (!record) return options; // sanity check only
            
            var field = this.getField(colNum);
            
            var dayIndex = field._dayIndex;
            options.dayIndex = dayIndex;
            var date = record["date" + dayIndex];
            options.date = date.toSchemaDate("date");
            
            var events = record["event" + dayIndex];
            if (events == null) {
                options.isHeaderRow = true;
            } else {
                options.isHeaderRow = false;
            }
            return options;
        },
        
        getRowNumFromLocatorConfig : function (rowConfig) {

            var locateCellsBy = this.creator.locateCellsBy;
            if ((locateCellsBy == "date" || locateCellsBy == null) &&
                rowConfig.date != null)
            {
                var date = isc.Date.parseSchemaDate(rowConfig.date),
                    headerRow = (rowConfig.isHeaderRow == "true"),
                    dateField = "date" + rowConfig.dayIndex,
                    eventField = "event" + rowConfig.dayIndex;
                for (var i = 0; i < this.data.length; i++) {
                    var isHeader = (this.data[i][eventField] == null);
                    if (isHeader == headerRow) {
                        if (Date.compareLogicalDates(this.data[i][dateField], date) == 0) {
                            return i;
                        }
                    }
                }
                // no matching record (by date)
                return -1;
            }
            this.locateRowsBy = "index";
            return this.Super("getRowNumFromLocatorConfig", arguments);
        },
        
        getColLocatorOptions : function (body, rowNum, colNum) {
            var options = this.Super("getColLocatorOptions", arguments);
            // if we just record the dayIndex we can use that to find the column.
            // If the configuration changes such that (for example) the date isn't showing,
            // we'll just fail to find the cell so return -1 from getRowNumFromLocatorConfig
            options.dayIndex = this.getField(colNum)._dayIndex;
            return options;
        },
        
        getColNumFromLocatorConfig : function (colConfig) {
            var locateCellsBy = this.locateCellsBy;
            if (locateCellsBy == null || locateCellsBy == "date") {
                return this.fields.findIndex("_dayIndex", parseInt(colConfig.dayIndex));
            }
            
            this.locateColsBy = "index";
            return this.Super("getColNumFromLocatorConfig", arguments);
        }
        
    });
    
     
    isc.MonthScheduleBody.addProperties({
    
        // override getInterior locator to actually identify event link locators
        // (based on event rather than cell location)
        getInteriorLocator : function (element) {
            if (element.tagName.toLowerCase() == "a") {
                var href = element.href;
                if (href != null) {
                    // We're using the href -- this is pretty hokey but no
                    // other info is written into the DOM element...
                    // It should be robust across page reloades etc since the
                    // stored locator is based on the event directly -- not on the
                    // href directly -- we just use that to find the event (and then to
                    // find tha ppropriate link from the event when parsing locators)
                    
                    // double escaping necessary -- first is eaten by quotes
                    var match = href.match("javascript:.*monthViewEventClick\\((\\d+),(\\d+),(\\d+)\\);");
                    //this.logWarn("match!:" + match);
                    if (match) {
                        var row = parseInt(match[1]),   
                            col = parseInt(match[2]),
                            index = parseInt(match[3]);
                        var events = this.grid.getEvents(row,col),
                            event = events[index];
                           
                        if (event == null) {
                            this.logWarn("Unable to determine event associated with apparent event " +
                                "link element -- returning cell");
                            return this.Super("getInteriorLocator", arguments);
                        }
                        
                        var calendar = this.grid.creator,
                            config = calendar.getEventLocatorConfig(event);
                        var string = isc.AutoTest.createLocatorFallbackPath("eventLink", config);
                        //this.logWarn("string:" + string);
                        return string;
                    }
                }
            }
            
            return this.Super("getInteriorLocator", arguments);
        },
        
         getInnerElementFromSplitLocator : function (locatorArray) {
            
            if (this.emptyLocatorArray(locatorArray)) return this.getHandle();
            
            // if it starts with "eventLink" - get the relevant event from the Calendar
            // and then find it in our body if possible
            if (locatorArray.length == 1 && locatorArray[0].startsWith("eventLink")) {
                var fullConfig = isc.AutoTest.parseLocatorFallbackPath(locatorArray[0]);
                
                var calendar = this.grid.creator;
                var event = calendar.getEventFromLocatorConfig(fullConfig.config);
                
                var cell = this.grid.getEventCell(event);
                
                if (cell != null) {
                    var data = this.grid.data,
                        rowNum = cell[0],
                        colNum = cell[1],
                        dayIndex = this.grid.getField(colNum)._dayIndex;
            
                    var cellElement = this.getTableElement(rowNum,colNum),
                        links = cellElement.getElementsByTagName("A");
                    if (links != null) {
                        for (var iii = 0; iii < links.length; iii++) {
                            var href = links[iii].href;
                            if (href != null) {
                                // double escaping necessary -- first is eaten by quotes
                                var match = href.match("javascript:.*monthViewEventClick\\((\\d+),(\\d+),(\\d+)\\);");
                                if (match && data[rowNum]["event"+dayIndex][parseInt(match[3])] 
                                    == event) 
                                {
                                    return links[iii];
                                }
                            }
                        }
                    }
                }
                return this.Super("getInnerElementFromSplitLocator", arguments);
                
            }
            
         }
            
    });
    
    // Events:
    // Calendars are dataBound components where this.data is a set of events to show
    // (May come from a dataSource).
    // In Day and Week views, events show up as windows floating over the grid body
    // In Month view events are embedded directly in the cells
    // Modify the standard row locator / parsing logic to store / retrieve events
    // and find the appropriate windows (or link elements in the month view)
    isc.Calendar.addProperties({
            
        // this method gets called automatically for autoChildren.
        // Pick up eventWindows and store information based on the event they represent
        getCanvasLocatorFallbackPath : function (name, canvas, sourceArray, properties, mask) {
            if (name == "eventWindow") {
                var options = this.getEventLocatorConfig(canvas.event);
                return isc.AutoTest.createLocatorFallbackPath("eventWindow", options);
            }
            return this.Super("getCanvasLocatorFallbackPath", arguments);
        },
        
        getEventLocatorConfig : function (event) {
            //this.logWarn("event:" + this.echo(event));
            var config = {};
            if (this.dataSource) {
                var pkField = this.getDataSource().getPrimaryKeyFieldName();
                config[pkField] = event[pkField];
            }
            
            var nameField = this.nameField;
            config[nameField] = event[nameField];
            
            var startField = this.startDateField;
            var startTime = event[startField];
            config[startField] = startTime.toSchemaDate();
            
            var endField = this.endDateField;
            var endTime = event[endField];
            config[endField] = endTime.toSchemaDate();
            
            config.index = this.data.indexOf(event);
            //this.logWarn("event config: " + this.echo(config));
            return config;
        },
        
        // substring param really just used for logging
        getChildFromFallbackLocator : function (substring, fallbackLocatorConfig) {
            var type = fallbackLocatorConfig.name,
                config = fallbackLocatorConfig.config;
                
            if (type == "eventWindow") {
                var viewName = this.mainView.getSelectedTab().viewName;
                if (viewName == "day") {
                    var children = this.dayView.body.children;
                } else if (viewName == "week") {
                    var children = this.weekView.body.children;
                }
                
                if (children != null) {
                    var event = this.getEventFromLocatorConfig(config),
                        eWindow = children.find("event", event);
                    return eWindow;
                }
                this.logWarn("unable to find event window associated with event:" + this.echo(event) +
                    " based on locator string:" + substring + 
                    ". It's possible that this event is not visible in the current view of " +
                    "this Calendar", "AutoTest");
                return null;
            }
            
            return this.Super("getChildFromFallbackLocator", arguments);
        },
        
        // we need date support.
        // So we need to be able to customize fields to record
        
        getEventFromLocatorConfig : function (config) {
            var locateBy = this.locateEventsBy;
            if (locateBy == null) locateBy = "primaryKey";
            
            switch (locateBy) {
            case "primaryKey":
                var ds = this.getDataSource();
                if (ds) {
                    var pkField = ds.getPrimaryKeyFieldName();
                    if (pkField && config[pkField] != null) {
                        return this.data[this.data.findByKey(config)];
                    }
                }
                
            case "name":
                var name = config[this.nameField];
                if (name != null) return this.data.find(this.nameField, name);
                
                
            case "date":
                // we could convert these to dates, and then compare via compareDate but
                // that could trip up on millisecond differences, etc -- this seems a
                // safer approach.
                var startTime = config[this.startDateField],
                    endTime = config[this.endDateField];
                
                // we're going to have to find all dates where start AND end time match
                // we could get more sophisticated and match start / end separately too
                // but that seems like an odd use case
                for (var i = 0; i < this.data.length; i++) {
                    var testEvent = this.data.get(i);
                    if (testEvent == null) continue;
                    
                    if (testEvent[this.startDateField].toSchemaDate() == startTime &&
                        testEvent[this.endDateField].toSchemaDate() == endTime)
                    {
                        return testEvent;
                    }
                    this.logWarn("attempt to match calendar event by startDate / endDate " +
                        "unable to locate any events. Backing off to index within data array");
                }
                
            // back off to locating by index within this.data
            default:
                var index = parseInt(config.index);
                return this.data.get(index);
                
                
            }
        }
        
    });
    
}
if (isc.Calendar) isc.AutoTest.customizeCalendar();


// Hold off applying the AutoTest interface methods to widget classes until the page is done loading
// This ensures we don't depend on module load order

if (!isc.Page.isLoaded()) {
    isc.Page.setEvent("load", "isc.ApplyAutoTestMethods()");
} else {
    isc.ApplyAutoTestMethods();
}
