(in-package "RTL")

(include-book "round")

(include-book "arithmetic-5/top" :dir :system)

;; This file contains proofs of rnd-const-cor and rnd-drnd-inject (the final lemma of Chapter 6).

;; I found that the proof requires generalizing rnd-const-thm to include the case n = 1,
;; which in turn requires generalizations of rne-power-2, rna-power-2, rtz-power-2, rne-imp, and rna-imp.
;; I've named the new lemmas rnd-const-lemma, rne-pow-2, rna-pow-2, rtz-pow-2, rne-impl, and rna-impl.
;; To be consistent, I've renamed raz-imp razx-impl.

;;---------------------------------------------------------------------------------------

;; I also found that the lemma rna-force, which is listed in the book, was not in round.lisp, so it is proved here:

(local-defthmd rnaf-1
  (let ((e (expo x)))
    (implies (and (rationalp x)
                  (> x 0)
	  	  (posp n))
	     (> (expt 2 e) (expt 2 (- e n))))))

(local-defthmd rnaf-2
  (let ((e (expo x)))
    (implies (and (rationalp x)
                  (> x 0)
	  	  (posp n))
	     (> (- x (expt 2 (- e n))) 0)))
  :hints (("Goal" :use (expo-lower-bound rnaf-1) :in-theory '(abs))))

(local-defthmd rnaf-3
  (let ((e (expo x)))
    (implies (and (rationalp x)
                  (> x 0)
	  	  (posp n)
		  (rationalp y)
		  (exactp y n)
		  (< (abs (- x y)) (expt 2 (- e n))))
	     (> y 0)))
  :hints (("Goal" :use (rnaf-2))))

(local-defthmd rnaf-4
  (let ((e (expo x))
        (z (rna x n)))
    (implies (and (rationalp x)
                  (> x 0)
	  	  (posp n))
	     (>= (expo z) e)))
  :hints (("Goal" :use (expo-rna))))

(local-defthmd rnaf-5
  (let ((e (expo x)))
    (implies (and (rationalp x)
                  (> x 0)
	  	  (posp n)
		  (rationalp y)
		  (exactp y n)
		  (< (abs (- x y)) (expt 2 (- e n))))
	     (>= (expo y) e)))
  :hints (("Goal" :in-theory (enable exactp-2**n)
                  :use (rnaf-3 expo-lower-bound
		        (:instance expo>= (x y) (n (expo x)))
			(:instance fp-2 (x (expt 2 (expo x))))))))

(local-defthmd rnaf-6
  (let ((e (expo x))
        (z (rna x n)))
    (implies (and (rationalp x)
                  (> x 0)
	  	  (posp n)
		  (rationalp y)
		  (exactp y n)
		  (< (abs (- x y)) (expt 2 (- e n))))
	     (< (abs (- y z))
	        (expt 2 (1+ (- e n))))))
  :hints (("Goal" :nonlinearp t :use (rna-diff))))

(local-defthmd rnaf-7
  (let ((e (expo x))
        (z (rna x n)))
    (implies (and (rationalp x)
                  (> x 0)
	  	  (posp n)
		  (rationalp y)
		  (exactp y n)
		  (< (abs (- x y)) (expt 2 (- e n))))
	     (< z (fp+ y n))))
  :hints (("Goal" :nonlinearp t :use (rnaf-6 rnaf-5))))

(local-defthmd rnaf-8
  (let ((e (expo x))
        (z (rna x n)))
    (implies (and (rationalp x)
                  (> x 0)
	  	  (posp n)
		  (rationalp y)
		  (exactp y n)
		  (< (abs (- x y)) (expt 2 (- e n))))
	     (<= z y)))
  :hints (("Goal" :use (rnaf-3 rnaf-7 (:instance fp+2 (x y) (y (rna x n)))))))

(local-defthmd rnaf-9
  (let ((e (expo x))
        (z (rna x n)))
    (implies (and (rationalp x)
                  (> x 0)
	  	  (posp n)
		  (rationalp y)
		  (exactp y n)
		  (< (abs (- x y)) (expt 2 (- e n))))
	     (< y (fp+ z n))))
  :hints (("Goal" :nonlinearp t :use (rnaf-6 rnaf-4))))

(local-defthmd rnaf-10
  (let ((e (expo x))
        (z (rna x n)))
    (implies (and (rationalp x)
                  (> x 0)
	  	  (posp n)
		  (rationalp y)
		  (exactp y n)
		  (< (abs (- x y)) (expt 2 (- e n))))
	     (<= y z)))
  :hints (("Goal" :use (rnaf-3 rnaf-9 (:instance fp+2 (y y) (x (rna x n)))))))

(local-defthm rnaf-11
  (let ((e (expo x))
        (z (rna x n)))
    (implies (and (rationalp x)
                  (> x 0)
	  	  (posp n)
		  (rationalp y)
		  (exactp y n)
		  (< (abs (- x y)) (expt 2 (- e n))))
	     (= y z)))
  :rule-classes ()
  :hints (("Goal" :use (rnaf-8 rnaf-10))))

(local-defthm rnaf-12
  (let ((e (expo x))
        (z (rna x n)))
    (implies (and (rationalp x)
                  (< x 0)
	  	  (posp n)
		  (rationalp y)
		  (exactp y n)
		  (< (abs (- x y)) (expt 2 (- e n))))
	     (= y z)))
  :rule-classes ()
  :hints (("Goal" :use (rna-minus (:instance rnaf-11 (x (- x)) (y (- y)))))))

(defthm rna-force
    (implies (and (integerp n)
		  (> n 0)
                  (rationalp x)
                  (not (= x 0))
		  (rationalp y)
                  (exactp y n)
		  (< (abs (- x y)) (expt 2 (- (expo x) n))))
	     (= y (rna x n)))
  :hints (("Goal" :use (rnaf-11 rnaf-12)))
  :rule-classes ())


;;---------------------------------------------------------------------------------------

(local-defthm p2-1
  (implies (and (rationalp x) (> x 0)
                (>= (+ x (expt 2 (1- (expo x))))
	            (expt 2 (1+ (expo x))))
	        (not (= (rne x 1) (expt 2 (1+ (expo x))))))
           (equal x (- (expt 2 (1+ (expo x))) (expt 2 (1- (expo x))))))
  :hints (("Goal" :use (expo-upper-bound
                        (:instance rne-force (n 1) (y (expt 2 (1+ (expo x)))))
                        (:instance exactp-2**n (m 1) (n (1+ (expo x)))))))
  :rule-classes ())

(local-defthmd p2-2
  (implies (and (rationalp x) (> x 0))
           (equal (- (expt 2 (1+ (expo x))) (expt 2 (1- (expo x))))
	          (* 3 (expt 2 (1- (expo x)))))))

(local-defthmd p2-3
  (implies (and (rationalp x) (> x 0))
           (equal (rne (* 3 (expt 2 (1- (expo x)))) 1)
	          (expt 2 (1+ (expo x)))))
  :hints (("Goal" :use ((:instance rne-shift (x 3) (n 1) (k (1- (expo x))))))))

(local-defthm p2-4
  (implies (and (rationalp x) (> x 0)
	        (>= (+ x (expt 2 (1- (expo x))))
	            (expt 2 (1+ (expo x)))))
	   (= (rne x 1)
	      (expt 2 (1+ (expo x)))))
  :hints (("Goal" :use (p2-1 p2-2 p2-3)
                  :in-theory (theory 'minimal-theory)))
  :rule-classes ())

(defthm rne-pow-2
  (implies (and (rationalp x) (> x 0)
                (not (zp n))
	        (>= (+ x (expt 2 (- (expo x) n)))
	            (expt 2 (1+ (expo x)))))
	   (= (rne x n)
	      (expt 2 (1+ (expo x)))))
  :hints (("Goal" :use (rne-power-2 p2-4)))
  :rule-classes ())

;;---------------------------------------------------------------------------------------

(local-defthmd p2-5
  (implies (and (rationalp x) (> x 0) (not (zp n)))
           (equal (expo (- (expt 2 (1+ (expo x))) (expt 2 (- (expo x) n))))
		  (expo x)))
  :hints (("Goal" :use ((:instance expo>= (x (- (expt 2 (1+ (expo x))) (expt 2 (- (expo x) n)))) (n (expo x)))
                        (:instance expo<= (x (- (expt 2 (1+ (expo x))) (expt 2 (- (expo x) n)))) (n (expo x)))))))

(local-defthmd p2-6
  (implies (and (rationalp x) (> x 0) (not (zp n)))
           (equal (rna (- (expt 2 (1+ (expo x))) (expt 2 (- (expo x) n))) n)
	          (expt 2 (1+ (expo x)))))
  :hints (("Goal" :use (expo-lower-bound p2-5
                        (:instance rna-imp (x (- (expt 2 (1+ (expo x))) (expt 2 (- (expo x) n)))))
			(:instance rtz-exactp-b (x (expt 2 (1+ (expo x)))))
                        (:instance exactp-2**n (m n) (n (1+ (expo x))))))))

(defthm rna-pow-2
  (implies (and (rationalp x) (> x 0)
                (not (zp n))
	        (>= (+ x (expt 2 (- (expo x) n)))
	            (expt 2 (1+ (expo x)))))
	   (= (rna x n)
	      (expt 2 (1+ (expo x)))))
  :hints (("Goal" :use (p2-6 expo-upper-bound
                        (:instance rna-monotone (x (- (expt 2 (1+ (expo x))) (expt 2 (- (expo x) n)))) (y x))
			(:instance rna-exactp-b (x (expt 2 (1+ (expo x)))))
			(:instance rna-monotone (y (expt 2 (1+ (expo x)))))
			(:instance exactp-2**n (m n) (n (1+ (expo x)))))))
  :rule-classes ())

;;---------------------------------------------------------------------------------------

(defthm rtz-pow-2
  (implies (and (rationalp x) (> x 0)
                (not (zp n))
	        (>= (+ x (expt 2 (- (expo x) n)))
	            (expt 2 (1+ (expo x)))))
	   (= (rtz (+ x (expt 2 (- (expo x) n))) n)
	      (expt 2 (1+ (expo x)))))
  :hints (("Goal" :nonlinearp t
                  :use (expo-upper-bound
                        (:instance fp+2 (y (rtz (+ x (expt 2 (- (expo x) n))) n)) (x (expt 2 (1+ (expo x)))))
			(:instance rtz-exactp-c (a (expt 2 (1+ (expo x)))) (x (+ x (expt 2 (- (expo x) n)))))
			(:instance rtz-upper-pos (x (+ x (expt 2 (- (expo x) n)))))
			(:instance exactp-2**n (m n) (n (1+ (expo x)))))))
  :rule-classes ())

;;---------------------------------------------------------------------------------------

(local-defthmd ri-1
    (implies (and (rationalp x) (> x 0)
                  (>= (+ x (expt 2 (1- (expo x))))
	              (expt 2 (1+ (expo x)))))
	     (equal (rne x 1)
	            (rtz (+ x (expt 2 (1- (expo x)))) 1)))
  :hints (("Goal" :use ((:instance rne-pow-2 (n 1))
                        (:instance rtz-pow-2 (n 1))))))

(local-defthmd ri-2
    (implies (and (rationalp x) (> x 0)
                  (< (+ x (expt 2 (1- (expo x))))
	             (expt 2 (1+ (expo x)))))
	     (equal (expo (rne x 1)) (expo x)))
  :hints (("Goal" :nonlinearp t
                  :use( (:instance expo-rne (n 1))
                        (:instance rne-diff (n 1))))))

(local-defthmd ri-3
    (implies (and (rationalp x) (> x 0)
                  (< (+ x (expt 2 (1- (expo x))))
	             (expt 2 (1+ (expo x)))))
             (equal (expo (+ x (expt 2 (1- (expo x))))) (expo x)))
  :hints (("Goal" :nonlinearp t
                  :use(expo-lower-bound
		       (:instance expo<= (x (+ x (expt 2 (1- (expo x))))) (n (expo x)))
                       (:instance expo>= (x (+ x (expt 2 (1- (expo x))))) (n (expo x)))))))

(local-defthmd ri-4
    (implies (and (rationalp x) (> x 0)
                  (exactp x 1)
                  (< (+ x (expt 2 (1- (expo x))))
	             (expt 2 (1+ (expo x)))))
             (<= x (rtz (+ x (expt 2 (1- (expo x)))) 1)))
  :hints (("Goal" :use ((:instance rtz-exactp-c (a x) (n 1) (x (+ x (expt 2 (1- (expo x))))))))))

(local-defthmd ri-5
    (implies (and (rationalp x) (> x 0)
                  (exactp x 1)
                  (< (+ x (expt 2 (1- (expo x))))
	             (expt 2 (1+ (expo x)))))
             (> (fp+ x 1) (rtz (+ x (expt 2 (1- (expo x)))) 1)))
  :hints (("Goal" :nonlinearp t :use ((:instance rtz-upper-pos (x (+ x (expt 2 (1- (expo x))))) (n 1))))))

(local-defthmd ri-6
    (implies (and (rationalp x) (> x 0)
                  (exactp x 1)
                  (< (+ x (expt 2 (1- (expo x))))
	             (expt 2 (1+ (expo x)))))
             (>= x (rtz (+ x (expt 2 (1- (expo x)))) 1)))
  :hints (("Goal" :use (ri-5
                        (:instance fp+2 (y (rtz (+ x (expt 2 (1- (expo x)))) 1)) (n 1))))))

(local-defthmd ri-7
    (implies (and (rationalp x) (> x 0)
                  (exactp x 1)
                  (< (+ x (expt 2 (1- (expo x))))
	             (expt 2 (1+ (expo x)))))
             (equal (rne x 1)
	            (rtz (+ x (expt 2 (1- (expo x)))) 1)))
  :hints (("Goal" :use (ri-4 ri-6
                        (:instance rne-exactp-b (n 1))))))

(local-defthmd ri-8
    (implies (and (rationalp x) (> x 0)
                  (not (exactp x 2))
                  (< (+ x (expt 2 (1- (expo x))))
	             (expt 2 (1+ (expo x)))))
             (not (equal (rne x 1)
	                 (- x (expt 2 (1- (expo x)))))))
  :hints (("Goal" :use ((:instance exactp-<= (x (rne x 1)) (m 1) (n 2))
                        (:instance expo-rne (n 1))
			(:instance fp+1 (x (rne x 1)) (n 2))))))

(local-defthmd ri-9
    (implies (and (rationalp x) (> x 0)
                  (not (exactp x 2))
                  (< (+ x (expt 2 (1- (expo x))))
	             (expt 2 (1+ (expo x)))))
             (> (rne x 1)
	        (- x (expt 2 (1- (expo x))))))
  :hints (("Goal" :use (ri-8 (:instance rne-diff (n 1))))))

(local-defthmd ri-10
    (implies (and (rationalp x) (> x 0)
                  (not (exactp x 2))
                  (< (+ x (expt 2 (1- (expo x))))
	             (expt 2 (1+ (expo x)))))
             (<= (rne x 1)
	         (rtz (+ x (expt 2 (1- (expo x)))) 1)))
  :hints (("Goal" :use ((:instance rne-diff (n 1))
                        (:instance rtz-exactp-c (a (rne x 1)) (x (+ x (expt 2 (1- (expo x))))) (n 1))))))

(local-defthmd ri-11
    (implies (and (rationalp x) (> x 0)
                  (not (exactp x 2))
                  (< (+ x (expt 2 (1- (expo x))))
	             (expt 2 (1+ (expo x)))))
             (< (rtz (+ x (expt 2 (1- (expo x)))) 1)
	        (+ (rne x 1) (expt 2 (expo x)))))
  :hints (("Goal" :nonlinearp t
                  :use (ri-9
                        (:instance rtz-upper-pos (x (+ x (expt 2 (1- (expo x))))) (n 1))))))

(local-defthmd ri-12
    (implies (and (rationalp x) (> x 0)
                  (not (exactp x 2))
                  (< (+ x (expt 2 (1- (expo x))))
	             (expt 2 (1+ (expo x)))))
             (< (rtz (+ x (expt 2 (1- (expo x)))) 1)
	        (expt 2 (1+ (expo x)))))
  :hints (("Goal" :nonlinearp t
                  :use (ri-9
                        (:instance rtz-upper-pos (x (+ x (expt 2 (1- (expo x))))) (n 1))))))

(local-defthmd ri-13
    (implies (and (rationalp x) (> x 0)
                  (not (exactp x 2))
                  (< (+ x (expt 2 (1- (expo x))))
	             (expt 2 (1+ (expo x)))))
             (<= (rtz (+ x (expt 2 (1- (expo x)))) 1)
	         (rne x 1)))
  :hints (("Goal" :use (ri-11 ri-12
                       (:instance expo-rne (n 1))
                       (:instance fp+2 (n 1) (x (rne x 1)) (y (rtz (+ x (expt 2 (1- (expo x)))) 1)))))))

(local-defthmd ri-14
    (implies (and (rationalp x) (> x 0)
                  (exactp x 2)
                  (not (exactp x 1)))
             (equal (sig x) 3/2))
  :hints (("Goal" :in-theory (enable exactp bvecp)
		  :use (sig-upper-bound sig-lower-bound (:instance bvecp-member (x (* 2 (sig x))) (n 2))))))

(local-defthmd ri-15
    (implies (and (rationalp x) (> x 0)
                  (exactp x 2)
                  (not (exactp x 1)))
             (equal (* 3/2 (expt 2 (expo x))) x))
  :hints (("Goal" :use (fp-abs ri-14))))

(local-defthmd ri-16
    (implies (and (rationalp x) (> x 0)
                  (exactp x 2)
                  (not (exactp x 1)))
             (equal (rne x 1) (expt 2 (1+ (expo x)))))
  :hints (("Goal" :use (ri-15 (:instance rne-shift (x 3/2) (k (expo x)) (n 1))))))

(local-defthmd ri-17
    (implies (and (rationalp x) (> x 0)
                  (exactp x 2)
                  (not (exactp x 1)))
             (equal (rne x 1) (rtz (+ x (expt 2 (1- (expo x)))) 1)))
  :hints (("Goal" :nonlinearp t
                  :use (ri-15 ri-16
                        (:instance rtz-exactp-b (n 1) (x (expt 2 (1+ (expo x)))))
			(:instance exactp-2**n (n (1+ (expo x))) (m 1))))))

(local-defthmd ri-18
    (implies (and (rationalp x) (> x 0))
             (equal (rne x 1) (rtz (+ x (expt 2 (1- (expo x)))) 1)))
  :hints (("Goal" :use (ri-1 ri-7 ri-10 ri-13 ri-17))))

(defthmd rne-impl
    (implies (and (rationalp x) (> x 0)
		  (not (zp n)))
	     (equal (rne x n)
		    (if (and (> n 1) (exactp x (1+ n)) (not (exactp x n)))
		        (rtz (+ x (expt 2 (- (expo x) n))) (1- n))
		      (rtz (+ x (expt 2 (- (expo x) n))) n))))
  :hints (("Goal" :use (rne-imp ri-18))))

 ;;---------------------------------------------------------------------------------------

(local-defthmd ria1
    (implies (and (rationalp x) (> x 0)
                  (>= (+ x (expt 2 (1- (expo x))))
	              (expt 2 (1+ (expo x)))))
	     (equal (rna x 1)
	            (rtz (+ x (expt 2 (1- (expo x)))) 1)))
  :hints (("Goal" :use ((:instance rna-pow-2 (n 1))
                        (:instance rtz-pow-2 (n 1))))))

(local-defthmd ria2
    (implies (and (rationalp x) (> x 0)
                  (< (+ x (expt 2 (1- (expo x))))
	             (expt 2 (1+ (expo x)))))
	     (equal (expo (rna x 1)) (expo x)))
  :hints (("Goal" :nonlinearp t
                  :use( (:instance expo-rna (n 1))
                        (:instance rna-diff (n 1))))))

(local-defthmd ria3
    (implies (and (rationalp x) (> x 0)
                  (< (+ x (expt 2 (1- (expo x))))
	             (expt 2 (1+ (expo x)))))
             (equal (expo (+ x (expt 2 (1- (expo x))))) (expo x)))
  :hints (("Goal" :nonlinearp t
                  :use(expo-lower-bound
		       (:instance expo<= (x (+ x (expt 2 (1- (expo x))))) (n (expo x)))
                       (:instance expo>= (x (+ x (expt 2 (1- (expo x))))) (n (expo x)))))))

(local-defthmd ria4
    (implies (and (rationalp x) (> x 0)
                  (exactp x 1)
                  (< (+ x (expt 2 (1- (expo x))))
	             (expt 2 (1+ (expo x)))))
             (<= x (rtz (+ x (expt 2 (1- (expo x)))) 1)))
  :hints (("Goal" :use ((:instance rtz-exactp-c (a x) (n 1) (x (+ x (expt 2 (1- (expo x))))))))))

(local-defthmd ria5
    (implies (and (rationalp x) (> x 0)
                  (exactp x 1)
                  (< (+ x (expt 2 (1- (expo x))))
	             (expt 2 (1+ (expo x)))))
             (> (fp+ x 1) (rtz (+ x (expt 2 (1- (expo x)))) 1)))
  :hints (("Goal" :nonlinearp t :use ((:instance rtz-upper-pos (x (+ x (expt 2 (1- (expo x))))) (n 1))))))

(local-defthmd ria6
    (implies (and (rationalp x) (> x 0)
                  (exactp x 1)
                  (< (+ x (expt 2 (1- (expo x))))
	             (expt 2 (1+ (expo x)))))
             (>= x (rtz (+ x (expt 2 (1- (expo x)))) 1)))
  :hints (("Goal" :use (ria5
                        (:instance fp+2 (y (rtz (+ x (expt 2 (1- (expo x)))) 1)) (n 1))))))

(local-defthmd ria7
    (implies (and (rationalp x) (> x 0)
                  (exactp x 1)
                  (< (+ x (expt 2 (1- (expo x))))
	             (expt 2 (1+ (expo x)))))
             (equal (rna x 1)
	            (rtz (+ x (expt 2 (1- (expo x)))) 1)))
  :hints (("Goal" :use (ria4 ria6
                        (:instance rna-exactp-b (n 1))))))

(local-defthmd ria8
    (implies (and (rationalp x) (> x 0)
                  (not (exactp x 2))
                  (< (+ x (expt 2 (1- (expo x))))
	             (expt 2 (1+ (expo x)))))
             (not (equal (rna x 1)
	                 (- x (expt 2 (1- (expo x)))))))
  :hints (("Goal" :use ((:instance exactp-<= (x (rna x 1)) (m 1) (n 2))
                        (:instance expo-rna (n 1))
			(:instance fp+1 (x (rna x 1)) (n 2))))))

(local-defthmd ria9
    (implies (and (rationalp x) (> x 0)
                  (not (exactp x 2))
                  (< (+ x (expt 2 (1- (expo x))))
	             (expt 2 (1+ (expo x)))))
             (> (rna x 1)
	        (- x (expt 2 (1- (expo x))))))
  :hints (("Goal" :use (ria8 (:instance rna-diff (n 1))))))

(local-defthmd ria10
    (implies (and (rationalp x) (> x 0)
                  (not (exactp x 2))
                  (< (+ x (expt 2 (1- (expo x))))
	             (expt 2 (1+ (expo x)))))
             (<= (rna x 1)
	         (rtz (+ x (expt 2 (1- (expo x)))) 1)))
  :hints (("Goal" :use ((:instance rna-diff (n 1))
                        (:instance rtz-exactp-c (a (rna x 1)) (x (+ x (expt 2 (1- (expo x))))) (n 1))))))

(local-defthmd ria11
    (implies (and (rationalp x) (> x 0)
                  (not (exactp x 2))
                  (< (+ x (expt 2 (1- (expo x))))
	             (expt 2 (1+ (expo x)))))
             (< (rtz (+ x (expt 2 (1- (expo x)))) 1)
	        (+ (rna x 1) (expt 2 (expo x)))))
  :hints (("Goal" :nonlinearp t
                  :use (ria9
                        (:instance rtz-upper-pos (x (+ x (expt 2 (1- (expo x))))) (n 1))))))

(local-defthmd ria12
    (implies (and (rationalp x) (> x 0)
                  (not (exactp x 2))
                  (< (+ x (expt 2 (1- (expo x))))
	             (expt 2 (1+ (expo x)))))
             (< (rtz (+ x (expt 2 (1- (expo x)))) 1)
	        (expt 2 (1+ (expo x)))))
  :hints (("Goal" :nonlinearp t
                  :use (ria9
                        (:instance rtz-upper-pos (x (+ x (expt 2 (1- (expo x))))) (n 1))))))

(local-defthmd ria13
    (implies (and (rationalp x) (> x 0)
                  (not (exactp x 2))
                  (< (+ x (expt 2 (1- (expo x))))
	             (expt 2 (1+ (expo x)))))
             (<= (rtz (+ x (expt 2 (1- (expo x)))) 1)
	         (rna x 1)))
  :hints (("Goal" :use (ria11 ria12
                       (:instance expo-rna (n 1))
                       (:instance fp+2 (n 1) (x (rna x 1)) (y (rtz (+ x (expt 2 (1- (expo x)))) 1)))))))

(local-defthmd ria14
    (implies (and (rationalp x) (> x 0)
                  (exactp x 2)
                  (not (exactp x 1)))
             (equal (sig x) 3/2))
  :hints (("Goal" :in-theory (enable exactp bvecp)
		  :use (sig-upper-bound sig-lower-bound (:instance bvecp-member (x (* 2 (sig x))) (n 2))))))

(local-defthmd ria15
    (implies (and (rationalp x) (> x 0)
                  (exactp x 2)
                  (not (exactp x 1)))
             (equal (* 3/2 (expt 2 (expo x))) x))
  :hints (("Goal" :use (fp-abs ria14))))

(local-defthmd ria16
    (implies (and (rationalp x) (> x 0)
                  (exactp x 2)
                  (not (exactp x 1)))
             (equal (rna x 1) (expt 2 (1+ (expo x)))))
  :hints (("Goal" :use (ria15 (:instance rna-shift (x 3/2) (k (expo x)) (n 1))))))

(local-defthmd ria17
    (implies (and (rationalp x) (> x 0)
                  (exactp x 2)
                  (not (exactp x 1)))
             (equal (rna x 1) (rtz (+ x (expt 2 (1- (expo x)))) 1)))
  :hints (("Goal" :nonlinearp t
                  :use (ria15 ria16
                        (:instance rtz-exactp-b (n 1) (x (expt 2 (1+ (expo x)))))
			(:instance exactp-2**n (n (1+ (expo x))) (m 1))))))

(local-defthmd ria18
    (implies (and (rationalp x) (> x 0))
             (equal (rna x 1) (rtz (+ x (expt 2 (1- (expo x)))) 1)))
  :hints (("Goal" :use (ria1 ria7 ria10 ria13 ria17))))

(defthmd rna-impl
    (implies (and (rationalp x) (> x 0)
		  (not (zp n)))
	     (equal (rna x n)
		    (rtz (+ x (expt 2 (- (expo x) n))) n)))
  :hints (("Goal" :use (rna-imp ria18))))

;;---------------------------------------------------------------------------------------

;; Just so that the names are consistent:

(defthmd raz-impl
    (implies (and (rationalp x)
		  (> x 0)
		  (integerp n)
		  (> n 0)
		  (integerp m)
		  (>= m n)
		  (exactp x m))
	     (equal (raz x n)
		    (rtz (+ x
		    	    (expt 2 (- (1+ (expo x)) n))
			    (- (expt 2 (- (1+ (expo x)) m))))
		         n)))
  :hints (("Goal" :use (raz-imp))))

;;---------------------------------------------------------------------------------------

(defthmd rnd-const-lemma
    (implies (and (common-mode-p mode)
		  (not (zp n))
		  (not (zp x))
		  (implies (or (= mode 'raz) (= mode 'rup))
		           (>= (expo x) (1- n))))
	     (equal (rnd x mode n)
		    (if (and (eql mode 'rne)
		             (> n 1)
			     (exactp x (1+ n))
                             (not (exactp x n)))
                        (rtz (+ x (rnd-const (expo x) mode n)) (1- n))
                      (rtz (+ x (rnd-const (expo x) mode n)) n))))
  :hints (("Goal" :use (rnd-const-thm rne-impl rna-impl expo-upper-bound (:instance raz-imp (m (1+ (expo x)))))
                  :in-theory (enable bvecp rnd common-mode-p ieee-rounding-mode-p rup rdn))))

;;---------------------------------------------------------------------------------------

;; I proved m-lemma-a and m-lemma-b below on the way to proving rnd-drnd-inject.
;; Then I realized that rnd-const-cor-a and rnd-const-cor-b would be good to have,
;; so I derived them easily as special cases of m-lemma-*.
;; Then I realized that it woyld be more natural to prove rnd-const-cor-* first
;; and then derive m-lemma-* from them.  That's what I do in the book, but I was
;; too lazy to do the ACL2 proof over again.

(local-defund c (k p mode)
  (rnd-const (1- k) mode p))

(local-defund sum (m k p mode)
  (+ m (c k p mode)))

(local-defund sl (m k p mode)
  (bits (sum m k p mode) (1- (- k p)) 0))

(local-defund sh (m k p mode)
  (bits (sum m k p mode) k (- k p)))

(local-defthm int-sum
  (implies (and (bvecp m k) (natp k) (natp k) (> k p) (common-mode-p mode))
           (integerp (sum m k p mode)))
  :rule-classes (:type-prescription :rewrite)
  :hints (("Goal" :in-theory (enable sum c rnd-const))))

;;---------------------------------------------------------------------------------------

(local-defthmd dia-1
  (implies (and (natp k) (natp p) (>= k (+ p 2)) (>= p 1)
                (bvecp m k)
		(>= (expo m) (- k p)))
           (let ((n (- (+ (expo m) p 1) k)))
	     (and (natp n)
	          (>= n 1)
		  (iff (= n 1) (= (expo m) (- k p)))
		  (= (- (expo m) n) (1- (- k p)))))))

(local-defthmd dia-2
  (implies (and (common-mode-p mode)
                (natp k) (natp p) (>= k (+ p 2)) (>= p 1))
	   (bvecp (c k p mode) (- k p)))
  :hints (("Goal" :in-theory (enable bvecp c rnd-const))))

(local-defthmd dia-3
  (implies (and (common-mode-p mode)
                (natp k) (natp p) (>= k (+ p 2)) (>= p 1)
                (bvecp m k)
		(>= (expo m) (- k p))
                (not (= (expo (sum m k p mode)) (expo m))))
	   (> (expo (sum m k p mode)) (expo m)))
  :hints (("Goal" :in-theory (enable bvecp sum)
                  :use (dia-2
			(:instance expo-monotone (x m) (y (sum m k p mode)))))))

(local-defthmd dia-4
  (implies (and (common-mode-p mode)
                (natp k) (natp p) (>= k (+ p 2)) (>= p 1)
                (bvecp m k)
		(>= (expo m) (- k p))
                (not (= (expo (sum m k p mode)) (expo m))))
	   (let ((e (expo (sum m k p mode))))
	     (and (<= (expt 2 e) (sum m k p mode))
	          (< (sum m k p mode) (+ (expt 2 e) (expt 2 (- k p)))))))
  :hints (("Goal" :in-theory (enable sum bvecp)
                  :use (dia-2 dia-3
		        (:instance expo-lower-bound (x (sum m k p mode)))
		        (:instance expo-upper-bound (x m))
			(:instance expo>= (x m) (n (expo (sum m k p mode))))))))

(local-defthmd dia-5
  (implies (and (common-mode-p mode)
                (natp k) (natp p) (>= k (+ p 2)) (>= p 1)
                (bvecp m k)
		(>= (expo m) (- k p)))
           (bvecp (sum m k p mode) (1+ k)))
  :hints (("Goal" :in-theory (enable sum bvecp)
                  :nonlinearp t
		  :use (dia-2))))

(local-defthmd dia-6
  (implies (and (common-mode-p mode)
                (natp k) (natp p) (>= k (+ p 2)) (>= p 1)
                (bvecp m k)
		(>= (expo m) (- k p))
                (> (expo (sum m k p mode)) (expo m)))
	   (let ((e (expo (sum m k p mode))))
	     (<= (expt 2 (- e (- k p)))
	         (/ (sum m k p mode) (expt 2 (- k p))))))
  :hints (("Goal" :nonlinearp t :use (dia-4))))

(local-defthmd hack-1
  (implies (and (integerp a) (integerp b) (rationalp x) (< x (+ (expt 2 a) (expt 2 b))))
           (< (/ x (expt 2 a)) (1+ (expt 2 (- b a)))))
  :hints (("Goal" :nonlinearp t)))

(local-defthmd dia-7
  (implies (and (common-mode-p mode)
                (natp k) (natp p) (>= k (+ p 2)) (>= p 1)
                (bvecp m k)
		(>= (expo m) (- k p))
                (> (expo (sum m k p mode)) (expo m)))
	   (let ((e (expo (sum m k p mode))))
	     (and (< (/ (sum m k p mode) (expt 2 (- k p)))
		     (1+ (expt 2 (- e (- k p))))))))
  :hints (("Goal" :use (dia-4 (:instance hack-1 (a (- k p)) (b (expo (sum m k p mode))) (x (sum m k p mode)))))))

(local-defthmd hack-2
  (implies (and (integerp n) (rationalp x) (<= n x) (< x (1+ n)))
           (equal (fl x) n)))

(local-defthmd dia-8
  (implies (and (common-mode-p mode)
                (natp k) (natp p) (>= k (+ p 2)) (>= p 1)
                (bvecp m k)
		(>= (expo m) (- k p))
                (> (expo (sum m k p mode)) (expo m)))
	   (let ((e (expo (sum m k p mode))))
	     (equal (fl (/ (sum m k p mode) (expt 2 (- k p))))
		    (expt 2 (- e (- k p))))))
  :hints (("Goal" ;:in-theory (enable sum)
                  :use (dia-6 dia-7 (:instance hack-2 (n (expt 2 (- (expo (sum m k p mode)) (- k p)))) (x (/ (sum m k p mode) (expt 2 (- k p)))))))))

(local-defthmd dia-9
  (implies (and (common-mode-p mode)
                (natp k) (natp p) (>= k (+ p 2)) (>= p 1)
                (bvecp m k)
		(>= (expo m) (- k p))
                (> (expo (sum m k p mode)) (expo m)))
	   (let ((e (expo (sum m k p mode))))
	     (equal (sh m k p mode)
	            (expt 2 (- e (- k p))))))
  :hints (("Goal" :use (dia-8 dia-5) :in-theory (enable sh bvecp bits))))

(local-defthmd dia-10
  (implies (and (common-mode-p mode)
                (natp k) (natp p) (>= k (+ p 2)) (>= p 1)
                (bvecp m k)
		(>= (expo m) (- k p))
                (> (expo (sum m k p mode)) (expo m)))
	   (let ((e (expo (sum m k p mode))))
	     (equal (* (expt 2 (- k p)) (sh m k p mode))
	            (expt 2 e))))
  :hints (("Goal" :use (dia-9 dia-5))))

(local-defthmd dia-11
  (implies (and (common-mode-p mode)
                (natp k) (natp p) (>= k (+ p 2)) (>= p 1)
                (bvecp m k)
		(>= (expo m) (- k p))
                (> (expo (sum m k p mode)) (expo m)))
	   (<= (expo (sum m k p mode)) k))
  :hints (("Goal" :in-theory (enable bvecp)
                  :use (dia-5 (:instance expo<= (x (sum m k p mode)) (n k))))))

(local-defthmd dia-12
  (implies (and (common-mode-p mode)
                (natp k) (natp p) (>= k (+ p 2)) (>= p 1)
                (bvecp m k)
		(>= (expo m) (- k p))
                (> (expo (sum m k p mode)) (expo m)))
	   (let ((e (expo (sum m k p mode))))
	     (equal (bits (expt 2 (- e (- k p))) p 1)
	            (expt 2 (1- (- e (- k p)))))))
  :hints (("Goal" :use (dia-11
                        (:instance bitn-expt-0 (i (- (expo (sum m k p mode)) (- k p))) (n 0))
			(:instance bits-plus-bitn (x (expt 2 (- (expo (sum m k p mode)) (- k p)))) (n p) (m 0)))
                  :in-theory (enable bvecp))))

(local-defthmd dia-13
  (implies (and (common-mode-p mode)
                (natp k) (natp p) (>= k (+ p 2)) (>= p 1)
                (bvecp m k)
		(>= (expo m) (- k p))
                (> (expo (sum m k p mode)) (expo m)))
	   (let ((e (expo (sum m k p mode))))
	     (equal (* (expt 2 (1+ (- k p))) (bits (sh m k p mode) p 1))
	            (expt 2 e))))
  :hints (("Goal" :use (dia-12 dia-9))))

(local-defthmd dia-14
  (implies (and (common-mode-p mode)
                (natp k) (natp p) (>= k (+ p 2)) (>= p 1)
                (bvecp m k)
		(>= (expo m) (- k p))
                (> (expo (sum m k p mode)) (expo m)))
           (equal (expo (rnd m mode (- (+ (expo m) p 1) k)))
	          (expo (sum m k p mode))))
  :hints (("Goal" :in-theory (enable sum c)
                  :use ((:instance rnd-const-lemma (x m) (n (- (+ (expo m) p 1) k)))
		        (:instance expo-rtz (x (sum m k p mode)) (n (- (+ (expo m) p 1) k)))
		        (:instance expo-rtz (x (sum m k p mode)) (n (- (+ (expo m) p) k)))))))

(local-defthmd dia-15
  (implies (and (common-mode-p mode)
                (natp k) (natp p) (>= k (+ p 2)) (>= p 1)
                (bvecp m k)
		(>= (expo m) (- k p)))
	   (<= (expo (sum m k p mode)) (1+ (expo m))))
  :hints (("Goal" :in-theory (enable bvecp sum)
                  :nonlinearp t
                  :use (dia-2
		        (:instance expo<= (x (sum m k p mode)) (n (1+ (expo m))))
		        (:instance expo-upper-bound (x m))))))

(local-defthmd dia-16
  (implies (and (common-mode-p mode)
                (natp k) (natp p) (>= k (+ p 2)) (>= p 1)
                (bvecp m k)
		(>= (expo m) (- k p))
                (> (expo (sum m k p mode)) (expo m)))
           (equal (rnd m mode (- (+ (expo m) p 1) k))
	          (expt 2 (expo (sum m k p mode)))))
  :hints (("Goal" :use (dia-14 dia-15
                        (:instance expo-rnd (x m) (n (- (+ (expo m) p 1) k)))))))

(local-defthmd dia-17
  (implies (and (common-mode-p mode)
                (natp k) (natp p) (>= k (+ p 2)) (>= p 1)
                (bvecp m k)
		(>= (expo m) (- k p))
                (not (= (expo (sum m k p mode)) (expo m))))
           (and (equal (rnd m mode (- (+ (expo m) p 1) k))
	               (* (expt 2 (1+ (- k p))) (bits (sh m k p mode) p 1)))
		(equal (rnd m mode (- (+ (expo m) p 1) k))
	               (* (expt 2 (- k p)) (sh m k p mode)))))
  :hints (("Goal" :use (dia-16 dia-13 dia-10 dia-3))))

;;---------------------------------------------------------------------------------------------------------

(local-defthmd dia-18
  (implies (and (natp k) (natp p) (>= k (+ p 2)) (>= p 1)
                (bvecp m k)
		(>= (expo m) (- k p)))
           (let ((n (- (+ (expo m) p 1) k)))
	     (iff (exactp m n)
	          (integerp (* (expt 2 (- p k)) m)))))
  :hints (("Goal" :in-theory (enable exactp2))))

(local-defthmd dia-19
  (implies (and (natp x) (not (zp n)) (< x (expt 2 n)))
           (iff (integerp (* (expt 2 (- n)) x))
	        (= x 0))))

(local-defthmd dia-20
  (implies (and (natp k) (natp p) (>= k (+ p 2)) (>= p 1)
                (bvecp m k))
           (iff (integerp (* (expt 2 (- p k)) m))
                (= (bits m (1- (- k p)) 0) 0)))
  :hints (("Goal" :use ((:instance bits-plus-bits (x m) (n (1- k)) (p (- k p)) (m 0))
		        (:instance dia-19 (n (- k p)) (x (bits m (1- (- k p)) 0)))
		        (:instance bits-bounds (x m) (i (1- (- k p))) (j 0))))))

(local-defthmd dia-21
  (implies (and (natp k) (natp p) (>= k (+ p 2)) (>= p 1)
                (bvecp m k)
		(>= (expo m) (- k p)))
           (let ((n (- (+ (expo m) p 1) k)))
	     (iff (exactp m n)
	          (= (bits m (1- (- k p)) 0) 0))))
  :hints (("Goal" :use (dia-18 dia-20))))

(local-defthmd dia-22
  (implies (and (natp k) (natp p) (>= k (+ p 2)) (>= p 1)
                (bvecp m k)
		(>= (expo m) (- k p)))
           (let ((n (- (+ (expo m) p 1) k)))
	     (iff (exactp m (1+ n))
	          (integerp (* (expt 2 (1+ (- p k))) m)))))
  :hints (("Goal" :in-theory (enable exactp2))))

(local-defthmd dia-23
  (implies (and (natp k) (natp p) (>= k (+ p 2)) (>= p 1)
                (bvecp m k))
           (iff (integerp (* (expt 2 (1+ (- p k))) m))
                (= (bits m (- (- k p) 2) 0) 0)))
  :hints (("Goal" :use ((:instance bits-plus-bits (x m) (n (1- k)) (p (1- (- k p))) (m 0))
		        (:instance dia-19 (n (1- (- k p))) (x (bits m (- (- k p) 2) 0)))
		        (:instance bits-bounds (x m) (i (- (- k p) 2)) (j 0))))))

(local-defthmd dia-24
  (implies (and (natp k) (natp p) (>= k (+ p 2)) (>= p 1)
                (bvecp m k)
		(>= (expo m) (- k p)))
           (let ((n (- (+ (expo m) p 1) k)))
	     (iff (exactp m (1+ n))
	          (= (bits m (- (- k p) 2) 0) 0))))
  :hints (("Goal" :use (dia-22 dia-23))))

(local-defthmd dia-25
  (implies (and (natp k) (natp p) (>= k (+ p 2)) (>= p 1)
                (bvecp m k)
		(>= (expo m) (- k p)))
           (let ((n (- (+ (expo m) p 1) k)))
	     (iff (and (exactp m (1+ n))
	               (not (exactp m n)))
		  (= (bits m (1- (- k p)) 0)
		     (expt 2 (1- (- k p)))))))
  :hints (("Goal" :use (dia-21 dia-24
                        (:instance bitn-0-1 (x m) (n (1- (- k p))))
			(:instance bitn-plus-bits (x m) (n (1- (- k p))) (m 0))
		        (:instance bits-bounds (x m) (i (- (- k p) 2)) (j 0))))))

(local-defthmd dia-26
  (implies (and (integerp x) (not (zp a)))
           (iff (= (mod (+ x (expt 2 (1- a))) (expt 2 a))
	           0)
	        (= (mod x (expt 2 a))
		   (expt 2 (1- a)))))
  :hints (("Goal" :nonlinearp t
                  :use ((:instance mod-plus-mod (n (expt 2 a)) (a x) (b (expt 2 (1- a))) (c (expt 2 (1- a))))
                        (:instance mod-plus-mod (n (expt 2 a)) (a (+ x (expt 2 (1- a)))) (b (expt 2 a)) (c (- (expt 2 (1- a)))))))))

(local-defthmd dia-27
  (implies (and (natp k) (natp p) (>= k (+ p 2)) (>= p 1)
                (bvecp m k)
		(>= (expo m) (- k p)))
           (let ((n (- (+ (expo m) p 1) k)))
	     (iff (and (exactp m (1+ n))
	               (not (exactp m n)))
		  (= (bits (+ m (expt 2 (1- (- k p)))) (1- (- k p)) 0)
		     0))))
  :hints (("Goal" :in-theory (enable bits-mod)
                  :use (dia-25
                        (:instance dia-26 (x m) (a (- k p)))))))

(local-defthmd dia-28
  (implies (and (natp k) (natp p) (>= k (+ p 2)) (>= p 1)
                (bvecp m k)
		(>= (expo m) (- k p))
		(= mode 'rne))
           (let ((n (- (+ (expo m) p 1) k)))
	     (iff (and (exactp m (1+ n))
	               (not (exactp m n)))
		  (= (sl m k p mode)
		     0))))
  :hints (("Goal" :in-theory (enable sl sum c rnd-const)
                  :use (dia-27))))

(local-defthmd dia-29
  (implies (and (common-mode-p mode)
                (natp k) (natp p) (>= k (+ p 2)) (>= p 1)
                (bvecp m k)
		(>= (expo m) (- k p)))
           (equal (rnd m mode (- (+ (expo m) p 1) k))
	          (if (and (= mode 'rne)
		           (not (= (expo m) (- k p)))
		           (= (sl m k p mode) 0))
		      (rtz (sum m k p mode) (+ (expo m) (- p k)))
		    (rtz (sum m k p mode) (1+ (+ (expo m) (- p k)))))))
  :hints (("Goal" :in-theory (enable sum c)
                  :use (dia-28 (:instance rnd-const-lemma (x m) (n (- (+ (expo m) p 1) k)))))))

(local-defthmd dia-30
  (implies (and (common-mode-p mode)
                (natp k) (natp p) (>= k (+ p 2)) (>= p 1)
                (bvecp m k)
		(>= (expo m) (- k p))
	        (= (expo (sum m k p mode)) (expo m)))
           (equal (rtz (sum m k p mode) (+ (expo m) (- p k)))
	          (* (expt 2 (1+ (- k p)))
		     (bits (sum m k p mode) (expo m) (1+ (- k p))))))
  :hints (("Goal" :use (dia-5 (:instance bits-rtz (x (sum m k p mode)) (n (1+ (expo m))) (k (- (+ (expo m) p) k)))))))

(local-defthmd dia-31
  (implies (and (common-mode-p mode)
                (natp k) (natp p) (>= k (+ p 2)) (>= p 1)
                (bvecp m k)
		(>= (expo m) (- k p))
	        (= (expo (sum m k p mode)) (expo m)))
           (equal (rtz (sum m k p mode) (+ (expo m) 1 (- p k)))
	          (* (expt 2 (- k p))
		     (bits (sum m k p mode) (expo m) (- k p)))))
  :hints (("Goal" :use (dia-5 (:instance bits-rtz (x (sum m k p mode)) (n (1+ (expo m))) (k (- (+ (expo m) 1 p) k)))))))

(local-defthm dia-32
  (implies (and (= mode 'rne)
                (natp k) (natp p) (>= k (+ p 2)) (>= p 1)
                (bvecp m k))
           (equal (sum m k p mode)
	          (+ m (expt 2 (1- (- k p))))))
  :hints (("Goal" :in-theory (enable sum c rnd-const))))

(local-defthmd dia-33
  (implies (and (= mode 'rne)
                (= (expo m) (- k p))
		(= (sl m k p mode) 0)
                (natp k) (natp p) (>= k (+ p 2)) (>= p 1)
                (bvecp m k)
		(>= (expo m) (- k p))
	        (= (expo (sum m k p mode)) (expo m)))
           (>= (bits m (1- (- k p)) 0)
	       (expt 2 (1- (- k p)))))
  :hints (("Goal" :nonlinearp t :in-theory (enable bits-mod sl))))

(local-defthmd dia-34
  (implies (and (= mode 'rne)
                (= (expo m) (- k p))
		(= (sl m k p mode) 0)
                (natp k) (natp p) (>= k (+ p 2)) (>= p 1)
                (bvecp m k)
	        (= (expo (sum m k p mode)) (expo m)))
           (>= (bits m (1- k) (- k p)) 1))
  :hints (("Goal" :use ((:instance bits-plus-bits (x m) (n (1- k)) (p (- k p)) (m 0))
			(:instance bits-bounds (x m) (i (1- (- k p))) (j 0))
			(:instance expo-lower-bound (x m))))))

(local-defthmd dia-35
  (implies (and (= mode 'rne)
                (= (expo m) (- k p))
		(= (sl m k p mode) 0)
                (natp k) (natp p) (>= k (+ p 2)) (>= p 1)
                (bvecp m k)
	        (= (expo (sum m k p mode)) (expo m)))
           (>= m (+ (expt 2 (- k p)) (expt 2 (1- (- k p))))))
  :hints (("Goal" :nonlinearp t
                  :use (dia-33 dia-34
                        (:instance bits-plus-bits (x m) (n (1- k)) (p (- k p)) (m 0))))))

(local-defthmd dia-36
  (implies (and (= mode 'rne)
                (= (expo m) (- k p))
		(= (sl m k p mode) 0)
                (natp k) (natp p) (>= k (+ p 2)) (>= p 1)
                (bvecp m k)
	        (= (expo (sum m k p mode)) (expo m)))
           (>= (sum m k p mode) (expt 2 (1+ (- k p)))))
  :hints (("Goal" :nonlinearp t
                  :use (dia-35))))

(local-defthmd dia-37
  (implies (and (= mode 'rne)
                (= (expo m) (- k p))
		(= (sl m k p mode) 0)
                (natp k) (natp p) (>= k (+ p 2)) (>= p 1)
                (bvecp m k)
	        (= (expo (sum m k p mode)) (expo m)))
           (< (sum m k p mode)
	      (+ (expt 2 (1+ (- k p)))
	         (expt 2 (1- (- k p))))))
  :hints (("Goal" :use ((:instance expo-upper-bound (x m))))))

(local-in-theory (disable dia-32))

(local-defthmd dia-38
  (implies (and (= mode 'rne)
                (= (expo m) (- k p))
		(= (sl m k p mode) 0)
                (natp k) (natp p) (>= k (+ p 2)) (>= p 1)
                (bvecp m k)
	        (= (expo (sum m k p mode)) (expo m)))
           (< (bits (sum m k p mode) (- k p) 0)
	      (expt 2 (1- (- k p)))))
  :hints (("Goal" :use (dia-37 dia-36 (:instance mod-force (n (expt 2 (1+ (- k p)))) (a 1)))
                  :nonlinearp t
                  :in-theory (enable bits-mod))))

(local-defthmd dia-39
  (implies (and (= mode 'rne)
                (= (expo m) (- k p))
		(= (sl m k p mode) 0)
                (natp k) (natp p) (>= k (+ p 2)) (>= p 1)
                (bvecp m k)
	        (= (expo (sum m k p mode)) (expo m)))
           (equal (bitn (sum m k p mode) (- k p)) 0))
  :hints (("Goal" :nonlinearp t
                  :use (dia-38
                        (:instance bitn-plus-bits (x (sum m k p mode)) (n (- k p)) (m 0))
			(:instance bits-bounds (x (sum m k p mode)) (i (1- (- k p))) (j 0))
			(:instance bitn-0-1 (x (sum m k p mode)) (n (- k p)))))))

(local-defthmd dia-40
  (implies (and (= mode 'rne)
                (= (expo m) (- k p))
		(= (sl m k p mode) 0)
                (natp k) (natp p) (>= k (+ p 2)) (>= p 1)
                (bvecp m k)
	        (= (expo (sum m k p mode)) (expo m)))
           (equal (bits (sum m k p mode) (expo m) (- k p))
	          (* 2 (bits (sum m k p mode) (expo m) (1+ (- k p))))))
  :hints (("Goal" :use (dia-39 (:instance bits-plus-bitn (x (sum m k p mode)) (n (expo m)) (m (- k p)))))))

(local-defthmd dia-41
  (implies (and (common-mode-p mode)
                (natp k) (natp p) (>= k (+ p 2)) (>= p 1)
                (bvecp m k)
		(>= (expo m) (- k p))
	        (= (expo (sum m k p mode)) (expo m)))
           (equal (rnd m mode (- (+ (expo m) p 1) k))
	          (if (and (= mode 'rne)
		           (= (sl m k p mode) 0))
		      (* (expt 2 (1+ (- k p)))
		         (bits (sum m k p mode) (expo m) (1+ (- k p))))
		    (* (expt 2 (- k p))
		       (bits (sum m k p mode) (expo m) (- k p))))))
  :hints (("Goal" :use (dia-29 dia-30 dia-31 dia-40))))

(local-defthmd dia-42
  (implies (and (common-mode-p mode)
                (natp k) (natp p) (>= k (+ p 2)) (>= p 1)
                (bvecp m k)
		(>= (expo m) (- k p))
	        (= (expo (sum m k p mode)) (expo m)))
           (and (< (sum m k p mode) (expt 2 (1+ k)))
	        (< (sum m k p mode) (expt 2 (1+ (expo m))))))
  :hints (("Goal" :in-theory (enable bvecp):nonlinearp t
                  :use ((:instance expo<= (x m) (n (1- k)))
		        (:instance expo>= (x (sum m k p mode)) (n k))
			(:instance expo-upper-bound (x (sum m k p mode)))))))

(local-defthmd dia-43
  (implies (and (common-mode-p mode)
                (natp k) (natp p) (>= k (+ p 2)) (>= p 1)
                (bvecp m k)
		(>= (expo m) (- k p))
	        (= (expo (sum m k p mode)) (expo m)))
           (equal (sum m k p mode)
	          (bits (sum m k p mode) (expo m) 0)))
  :hints (("Goal" :in-theory (enable bvecp)
                  :use (dia-5 dia-42))))

(local-defthmd dia-44
  (implies (and (common-mode-p mode)
                (natp k) (natp p) (>= k (+ p 2)) (>= p 1)
                (bvecp m k)
		(>= (expo m) (- k p))
	        (= (expo (sum m k p mode)) (expo m)))
           (<= (expo m) (1- k)))
  :hints (("Goal" :in-theory (enable bvecp)
                  :use (dia-5 (:instance expo<= (x m) (n (1- k)))))))

(local-defthmd dia-45
  (implies (and (common-mode-p mode)
                (natp k) (natp p) (>= k (+ p 2)) (>= p 1)
                (bvecp m k)
		(>= (expo m) (- k p))
	        (= (expo (sum m k p mode)) (expo m)))
           (equal (sh m k p mode)
	          (bits (sum m k p mode) (expo m) (- k p))))
  :hints (("Goal" :in-theory (enable sh bits-bits)
                  :use (dia-5 dia-43 dia-44))))

(local-defthmd dia-46
  (Implies (and (common-mode-p mode)
                (natp k) (natp p) (>= k (+ p 2)) (>= p 1)
                (bvecp m k)
		(>= (expo m) (- k p))
	        (= (expo (sum m k p mode)) (expo m)))
           (equal (bits (sh m k p mode) p 1)
	          (bits (sum m k p mode) (expo m) (1+ (- k p)))))
  :hints (("Goal" :in-theory (enable dia-45 bits-bits)
                  :use (dia-5 dia-44))))

(local-defthmd m-lemma-a
  (implies (and (common-mode-p mode)
                (natp k) (natp p) (>= k (+ p 2)) (>= p 1)
                (bvecp m k)
		(>= (expo m) (- k p)))
           (equal (rnd m mode (- (+ (expo m) p 1) k))
	          (if (and (= mode 'rne)
		           (= (sl m k p mode) 0))
		      (* (expt 2 (1+ (- k p)))
		         (bits (sh m k p mode) p 1))
		    (* (expt 2 (- k p))
		       (sh m k p mode)))))
  :hints (("Goal" :use (dia-17 dia-41 dia-46 dia-45))))

;;---------------------------------------------------------------------------------------------------------

;; This should be in basic.lisp:

(defthm mod-plus-mod-iff
    (implies (and (integerp a)
		  (integerp b)
		  (integerp c)
		  (not (zp n)))
             (iff (= (mod a n) (mod b n))
	          (= (mod (+ a c) n) (mod (+ b c) n))))
  :hints (("Goal" :use (mod-plus-mod (:instance mod-plus-mod (a (+ a c)) (b (+ b c)) (c (- c))))))
  :rule-classes ())

(local-defthmd dia-48
  (implies (and (common-mode-p mode)
                (natp k) (natp p) (>= k (+ p 2)) (>= p 1)
                (bvecp m k))
           (iff (= (sl m k p mode) (c k p mode))
	        (= (mod m (expt 2 (- k p))) 0)))
  :hints (("Goal" :in-theory (enable bits-mod sl sum bvecp)
                  :use (dia-2 (:instance mod-plus-mod-iff (a m) (b 0) (c (c k p mode)) (n (expt 2 (- k p))))))))

(local-defthmd dia-49
  (implies (and (common-mode-p mode)
                (natp k) (natp p) (>= k (+ p 2)) (>= p 1)
                (bvecp m k)
		(>= (expo m) (- k p)))
           (iff (exactp m (- (+ (expo m) p 1) k))
	        (= (mod m (expt 2 (- k p))) 0)))
  :hints (("Goal" :in-theory (enable exactp2) :use ((:instance mod-def (x m) (y (expt 2 (- k p))))))))

(local-defthmd dia-50
  (implies (and (common-mode-p mode)
                (natp k) (natp p) (>= k (+ p 2)) (>= p 1)
                (bvecp m k)
		(>= (expo m) (- k p)))
           (iff (= (sl m k p mode) (c k p mode))
	        (exactp m (- (+ (expo m) p 1) k))))
  :hints (("Goal" :use (dia-48 dia-49))))

(local-defthmd m-lemma-b
  (implies (and (common-mode-p mode)
                (natp k) (natp p) (>= k (+ p 2)) (>= p 1)
                (bvecp m k)
		(>= (expo m) (- k p)))
           (iff (= (sl m k p mode) (c k p mode))
	        (= m (rnd m mode (- (+ (expo m) p 1) k)))))
  :hints (("Goal" :use (dia-50 (:instance rnd-exactp-b (x m) (n  (- (+ (expo m) p 1) k)))))))

;;---------------------------------------------------------------------------------------------------------

(defthmd rnd-const-cor-a
  (implies (and (common-mode-p mode)
                (posp n)
                (posp m)
		(> (expo m) n))
	   (let* ((e (expo m))
	          (sum (+ m (rnd-const e mode n)))
		  (sh (bits sum (1+ e) (1+ (- e n))))
		  (sl (bits sum (- e n) 0)))
             (equal (rnd m mode n)
	            (if (and (= mode 'rne) (= sl 0))
		        (* (expt 2 (+ 2 (- e n))) (bits sh n 1))
		      (* (expt 2 (1+ (- e n))) sh)))))
  :hints (("Goal" :use ((:instance m-lemma-a (p n) (k (1+ (expo m))))
                        (:instance expo-upper-bound (x m)))
		  :in-theory (enable sum sl sh c bvecp))))

(defthmd rnd-const-cor-b
  (implies (and (common-mode-p mode)
                (posp n)
                (posp m)
		(> (expo m) n))
	   (let* ((e (expo m))
                  (c (rnd-const e mode n))
	          (sum (+ m c))
		  (sl (bits sum (- e n) 0)))
             (iff (= m (rnd m mode n))
	          (= sl c))))
  :hints (("Goal" :use ((:instance m-lemma-b (p n) (k (1+ (expo m))))
                        (:instance expo-upper-bound (x m)))
		  :in-theory (enable sum sl sh c bvecp))))

;;---------------------------------------------------------------------------------------------------------

(defthm posp-prec
  (implies (formatp f)
           (posp (prec f)))
  :rule-classes (:type-prescription :rewrite)
  :hints (("Goal" :in-theory (enable formatp prec))))

(local-defund m (k z)
  (* (expt 2 (1- k)) z))

(local-defthm bvecp-m
  (implies (and (rationalp z)
		(< 0 z)
		(< z 2)
		(natp k)
		(natp (m k z)))
	   (bvecp (m k z) k))
  :hints (("Goal" :in-theory (enable bvecp m) :nonlinearp t)))

(local-defthmd expo-s
  (implies (formatp f)
           (equal (expt 2 (expo (spn f)))
	          (spn f)))
  :hints (("Goal" :in-theory (enable spn))))

(local-defthm expo-m
  (implies (and (formatp f)
                (natp k)
                (rationalp z)
		(< 0 z))
	   (equal (expo (m k z))
	          (+ (expo (* (spn f) z))
	             (1- k)
		     (- (expo (spn f))))))
  :hints (("Goal" :use (expo-s
                        (:instance expo-shift (x z) (n (1- k)))
                        (:instance expo-shift (x z) (n (expo (spn f)))))
		  :in-theory (enable m))))

(set-ignore-ok t)

(local-defthmd rdia-1
  (let* ((p (prec f))
         (s (spn f))
  	 (x (* s z))
	 (m (m k z))
	 (sum (sum m k p mode))
	 (sl (sl m k p mode))
	 (sh (sh m k p mode)))
    (implies (and (formatp f)
                  (rationalp z)
		  (< 0 z)
		  (< z 2)
		  (common-mode-p mode)
		  (natp k)
		  (>= k (+ p 2))
		  (natp m)
		  (>= (expo m) (- k p)))
	    (equal (drnd x mode f)
	           (rnd x mode (- (+ (expo m) 1 p) k)))))
  :hints (("Goal" :in-theory (enable expo-m drnd))))

(local-defthmd rdia-2
  (let* ((p (prec f))
         (s (spn f))
  	 (x (* s z))
	 (m (m k z))
	 (sum (sum m k p mode))
	 (sl (sl m k p mode))
	 (sh (sh m k p mode)))
    (implies (and (formatp f)
                  (rationalp z)
		  (< 0 z)
		  (< z 2)
		  (common-mode-p mode)
		  (natp k)
		  (>= k (+ p 2))
		  (natp m)
		  (>= (expo m) (- k p)))
	    (equal (* (expt 2 (+ (- 1 k))) s (rnd (m k z) mode (- (+ (expo m) 1 p) k)))
	           (rnd x mode (- (+ (expo m) 1 p) k)))))
  :hints (("Goal" :in-theory (enable m)
                  :use (expo-s (:instance rnd-shift (x (m k z)) (n (- (+ (expo (m k z)) 1 (prec f)) k)) (k (- (1+ (expo (spn f))) k)))))))

(local-defthmd rdia-3
  (let* ((p (prec f))
         (s (spn f))
  	 (x (* s z))
	 (m (m k z))
	 (sum (sum m k p mode))
	 (sl (sl m k p mode))
	 (sh (sh m k p mode)))
    (implies (and (formatp f)
                  (rationalp z)
		  (< 0 z)
		  (< z 2)
		  (common-mode-p mode)
		  (natp k)
		  (>= k (+ p 2))
		  (natp m)
		  (>= (expo m) (- k p)))
	    (equal (drnd x mode f)
	           (* (expt 2 (+ (- 1 k))) s (rnd (m k z) mode (- (+ (expo m) 1 p) k))))))
  :hints (("Goal" :use (rdia-1 rdia-2))))

(local-defthmd rdia-4
  (let* ((p (prec f))
         (s (spn f))
  	 (x (* s z))
	 (m (m k z))
	 (sum (sum m k p mode))
	 (sl (sl m k p mode))
	 (sh (sh m k p mode)))
    (implies (and (formatp f)
                  (rationalp z)
		  (< 0 z)
		  (< z 2)
		  (common-mode-p mode)
		  (natp k)
		  (>= k (+ p 2))
		  (natp m)
		  (>= (expo m) (- k p)))
	    (equal (* (expt 2 (+ (- 1 k))) s (rnd (m k z) mode (- (+ (expo m) 1 p) k)))
	           (if (and (= mode 'rne) (= sl 0))
		       (* (expt 2 (- 2 p)) s (bits sh p 1))
		     (* (expt 2 (- 1 p)) s sh)))))
  :hints (("Goal" :use ((:instance m-lemma-a (p (prec f)) (m (m k z)))))))

(local-defthmd rdia-5
  (let* ((p (prec f))
         (s (spn f))
  	 (x (* s z))
	 (m (m k z))
	 (sum (sum m k p mode))
	 (sl (sl m k p mode))
	 (sh (sh m k p mode)))
    (implies (and (formatp f)
                  (rationalp z)
		  (< 0 z)
		  (< z 2)
		  (common-mode-p mode)
		  (natp k)
		  (>= k (+ p 2))
		  (natp m)
		  (>= (expo m) (- k p)))
	    (equal (drnd x mode f)
	           (if (and (= mode 'rne) (= sl 0))
		       (* (expt 2 (- 2 p)) s (bits sh p 1))
		     (* (expt 2 (- 1 p)) s sh)))))
  :hints (("Goal" :use (rdia-3 rdia-4))))

(local-defthmd rdia-6
  (let* ((p (prec f))
         (s (spn f))
  	 (x (* s z))
	 (m (m k z))
	 (sum (sum m k p mode))
	 (sl (sl m k p mode))
	 (sh (sh m k p mode)))
    (implies (and (formatp f)
                  (rationalp z)
		  (< 0 z)
		  (< z 2)
		  (common-mode-p mode)
		  (natp k)
		  (>= k (+ p 2))
		  (natp m)
		  (< (expo m) (- k p)))
	    (< x (* (expt 2 (- 1 (prec f))) s))))
  :hints (("Goal" :use ((:instance expo>= (x (m k z)) (n (- k (prec f)))))
                  :in-theory (enable m)
                  :nonlinearp t)))

(local-defthmd rdia-7
  (let* ((p (prec f))
         (s (spn f))
  	 (x (* s z))
	 (m (m k z))
	 (sum (sum m k p mode))
	 (sl (sl m k p mode))
	 (sh (sh m k p mode)))
    (implies (and (formatp f)
                  (rationalp z)
		  (< 0 z)
		  (< z 2)
		  (common-mode-p mode)
		  (natp k)
		  (>= k (+ p 2))
		  (natp m)
		  (< (expo m) (- k p)))
	    (< x (spd f))))
  :hints (("Goal" :use (rdia-6)
                  :in-theory (enable spn spd))))

(local-defthmd rdia-8
  (let* ((p (prec f))
         (s (spn f))
  	 (x (* s z))
	 (m (m k z))
	 (sum (sum m k p mode))
	 (sl (sl m k p mode))
	 (sh (sh m k p mode)))
    (implies (and (formatp f)
                  (rationalp z)
		  (< 0 z)
		  (< z 2)
		  (common-mode-p mode)
		  (natp k)
		  (>= k (+ p 2))
		  (natp m)
		  (< (expo m) (- k p))
		  (= mode 'rne))
	    (and (iff (= sl 0) (equal m (expt 2 (1- (- k p)))))
	         (iff (= sl 0) (equal sum (expt 2 (- k p)))))))
  :hints (("Goal" :use ((:instance expo>= (x (m k z)) (n (- k (prec f))))
                        (:instance mod-force (a 1) (n (expt 2 (- k (prec f)))) (m (sum (m k z) k p mode))))
	          :nonlinearp t
                  :in-theory (enable bits-mod c rnd-const sl sum))))

(local-defthmd rdia-9
  (let* ((p (prec f))
         (s (spn f))
  	 (x (* s z))
	 (m (m k z))
	 (sum (sum m k p mode))
	 (sl (sl m k p mode))
	 (sh (sh m k p mode)))
    (implies (and (formatp f)
                  (rationalp z)
		  (< 0 z)
		  (< z 2)
		  (common-mode-p mode)
		  (natp k)
		  (>= k (+ p 2))
		  (natp m)
		  (< (expo m) (- k p))
		  (= mode 'rne)
		  (= sl 0))
	    (equal x (/ (spd f) 2))))
  :hints (("Goal" :use (rdia-8)
	          :nonlinearp t
                  :in-theory (enable m spn spd))))

(local-defthmd rdia-10
  (let* ((p (prec f))
         (s (spn f))
  	 (x (* s z))
	 (m (m k z))
	 (sum (sum m k p mode))
	 (sl (sl m k p mode))
	 (sh (sh m k p mode)))
    (implies (and (formatp f)
                  (rationalp z)
		  (< 0 z)
		  (< z 2)
		  (common-mode-p mode)
		  (natp k)
		  (>= k (+ p 2))
		  (natp m)
		  (< (expo m) (- k p))
		  (= mode 'rne)
		  (= sl 0))
	    (equal (drnd x mode f) 0)))
  :hints (("Goal" :use (rdia-9 drnd-tiny-b))))

(local-defthmd rdia-11
  (let* ((p (prec f))
         (s (spn f))
  	 (x (* s z))
	 (m (m k z))
	 (sum (sum m k p mode))
	 (sl (sl m k p mode))
	 (sh (sh m k p mode)))
    (implies (and (formatp f)
                  (rationalp z)
		  (< 0 z)
		  (< z 2)
		  (common-mode-p mode)
		  (natp k)
		  (>= k (+ p 2))
		  (natp m)
		  (< (expo m) (- k p))
		  (= mode 'rne)
		  (= sl 0))
	    (equal (bits sh p 1) 0)))
  :hints (("Goal" :use (rdia-8
                        (:instance bits-shift-up-1 (x 1) (k (- k (prec f))) (i k) (j (1+ (- k (prec f)))))
			(:instance bits-plus-bitn (x 1) (n (prec f)) (m 0)))
		  :in-theory (enable sh bits-bits))))

(local-defthmd rdia-12
  (let* ((p (prec f))
         (s (spn f))
  	 (x (* s z))
	 (m (m k z))
	 (sum (sum m k p mode))
	 (sl (sl m k p mode))
	 (sh (sh m k p mode)))
    (implies (and (formatp f)
                  (rationalp z)
		  (< 0 z)
		  (< z 2)
		  (common-mode-p mode)
		  (natp k)
		  (>= k (+ p 2))
		  (natp m)
		  (< (expo m) (- k p))
		  (= mode 'rne)
		  (= sl 0))
	    (equal (drnd x mode f)
	           (* (expt 2 (- 2 p)) S (bits sh p 1)))))
  :hints (("Goal" :use (rdia-10 rdia-11))))

(local-defthmd rdia-13
  (let* ((p (prec f))
         (s (spn f))
  	 (x (* s z))
	 (m (m k z))
	 (sum (sum m k p mode))
	 (sl (sl m k p mode))
	 (sh (sh m k p mode)))
    (implies (and (formatp f)
                  (rationalp z)
		  (< 0 z)
		  (< z 2)
		  (common-mode-p mode)
		  (natp k)
		  (>= k (+ p 2))
		  (natp m)
		  (< (expo m) (- k p)))
	    (< sum (expt 2 (1+ (- k p))))))
  :hints (("Goal" :use ((:instance dia-2 (p (prec f)))
                        (:instance expo>= (x (m k z)) (n (- k (prec f)))))
		  :nonlinearp t
                  :in-theory (enable bvecp sum))))

(local-defthmd rdia-14
  (let* ((p (prec f))
         (s (spn f))
  	 (x (* s z))
	 (m (m k z))
	 (sum (sum m k p mode))
	 (sl (sl m k p mode))
	 (sh (sh m k p mode)))
    (implies (and (formatp f)
                  (rationalp z)
		  (< 0 z)
		  (< z 2)
		  (common-mode-p mode)
		  (natp k)
		  (>= k (+ p 2))
		  (natp m))
	    (> sum 0)))
  :hints (("Goal" :use ((:instance dia-2 (p (prec f))))
		  :nonlinearp t
                  :in-theory (enable bvecp sum m))))

(local-defthmd rdia-15
  (let* ((p (prec f))
         (s (spn f))
  	 (x (* s z))
	 (m (m k z))
	 (sum (sum m k p mode))
	 (sl (sl m k p mode))
	 (sh (sh m k p mode)))
    (implies (and (formatp f)
                  (rationalp z)
		  (< 0 z)
		  (< z 2)
		  (common-mode-p mode)
		  (natp k)
		  (>= k (+ p 2))
		  (natp m)
		  (< (expo m) (- k p)))
	    (equal sh (mod (fl (/ sum (expt 2 (- k p))))
	                   (expt 2 (1+ p))))))
  :hints (("Goal" :in-theory (enable sh)
                  :use ((:instance bits-mod-fl (x (sum (m k z) k (prec f) mode)) (i (1+ k)) (j (- k (prec f))))))))

(local-defthmd hack-3
  (implies (and (rationalp x) (posp n) (> x 0) (< x (* 2 n)))
           (equal (fl (/ x n)) (if (>= x n) 1 0)))
  :hints (("Goal" :nonlinearp t)))

(local-defthmd rdia-16
  (let* ((p (prec f))
         (s (spn f))
  	 (x (* s z))
	 (m (m k z))
	 (sum (sum m k p mode))
	 (sl (sl m k p mode))
	 (sh (sh m k p mode)))
    (implies (and (formatp f)
                  (rationalp z)
		  (< 0 z)
		  (< z 2)
		  (common-mode-p mode)
		  (natp k)
		  (>= k (+ p 2))
		  (natp m)
		  (< (expo m) (- k p)))
	    (equal (fl (/ sum (expt 2 (- k p))))
		   (if (>= sum (expt 2 (- k p))) 1 0))))
  :hints (("Goal" :use (rdia-13 rdia-14 (:instance hack-3 (x (sum (m k z) k (prec f) mode)) (n (expt 2 (- k (prec f)))))))))

(local-defthmd rdia-17
  (let* ((p (prec f))
         (s (spn f))
  	 (x (* s z))
	 (m (m k z))
	 (sum (sum m k p mode))
	 (sl (sl m k p mode))
	 (sh (sh m k p mode)))
    (implies (and (formatp f)
                  (rationalp z)
		  (< 0 z)
		  (< z 2)
		  (common-mode-p mode)
		  (natp k)
		  (>= k (+ p 2))
		  (natp m)
		  (< (expo m) (- k p)))
	    (equal sh (if (>= sum (expt 2 (- k p))) 1 0))))
  :hints (("Goal" :use (rdia-15 rdia-16))))

(local-defthmd rdia-18
  (let* ((p (prec f))
         (s (spn f))
  	 (x (* s z))
	 (m (m k z))
	 (sum (sum m k p mode))
	 (sl (sl m k p mode))
	 (sh (sh m k p mode)))
    (implies (and (formatp f)
                  (rationalp z)
		  (< 0 z)
		  (< z 2)
		  (common-mode-p mode)
		  (natp k)
		  (>= k (+ p 2))
		  (natp m)
		  (< (expo m) (- k p)))
	    (and (iff (< m (expt 2 (1- (- k p)))) (< x (/ (spd f) 2)))
	         (iff (> m (expt 2 (1- (- k p)))) (> x (/ (spd f) 2))))))
  :hints (("Goal" :in-theory (enable spn spd m) :nonlinearp t)))

(local-defthmd rdia-19
  (let* ((p (prec f))
         (s (spn f))
  	 (x (* s z))
	 (m (m k z))
	 (sum (sum m k p mode))
	 (sl (sl m k p mode))
	 (sh (sh m k p mode)))
    (implies (and (formatp f)
                  (rationalp z)
		  (< 0 z)
		  (< z 2)
		  (common-mode-p mode)
		  (natp k)
		  (>= k (+ p 2))
		  (natp m)
		  (< (expo m) (- k p))
		  (= mode 'rne)
		  (not (= sl 0)))
	    (equal (drnd x mode f)
	           (* (spd f) sh))))
  :hints (("Goal" :use (rdia-18 drnd-tiny-b rdia-17 rdia-8 rdia-7
                        (:instance expo>= (x (m k z)) (n (- k (prec f))))
                        (:instance drnd-tiny-a (x (* (spn f) z)))
			(:instance drnd-tiny-c (x (* (spn f) z))))
		  :nonlinearp t
		  :in-theory (enable c rnd-const sum))))

(local-defthmd rdia-20
  (let* ((p (prec f))
         (s (spn f))
  	 (x (* s z))
	 (m (m k z))
	 (sum (sum m k p mode))
	 (sl (sl m k p mode))
	 (sh (sh m k p mode)))
    (implies (and (formatp f)
                  (rationalp z)
		  (< 0 z)
		  (< z 2)
		  (common-mode-p mode)
		  (natp k)
		  (>= k (+ p 2))
		  (natp m)
		  (< (expo m) (- k p))
		  (= mode 'rna))
	    (equal (drnd x mode f)
	           (* (spd f) sh))))
  :hints (("Goal" :use (rdia-18 drnd-tiny-b rdia-7 rdia-17 rdia-8
                        (:instance drnd-tiny-a (x (* (spn f) z)))
			(:instance drnd-tiny-c (x (* (spn f) z))))
		  :nonlinearp t
		  :in-theory (enable c rnd-const sum))))

(local-defthmd rdia-21
  (let* ((p (prec f))
         (s (spn f))
  	 (x (* s z))
	 (m (m k z))
	 (sum (sum m k p mode))
	 (sl (sl m k p mode))
	 (sh (sh m k p mode)))
    (implies (and (formatp f)
                  (rationalp z)
		  (< 0 z)
		  (< z 2)
		  (common-mode-p mode)
		  (natp k)
		  (>= k (+ p 2))
		  (natp m)
		  (< (expo m) (- k p))
		  (or (= mode 'rtz) (= mode 'rdn)))
	    (equal (drnd x mode f)
	           (* (spd f) sh))))
  :hints (("Goal" :use (rdia-18 drnd-tiny-b rdia-7 rdia-17 rdia-8
                        (:instance expo>= (x (m k z)) (n (- k (prec f))))
			(:instance drnd-tiny-a (x (* (spn f) z)))
			(:instance drnd-tiny-c (x (* (spn f) z))))
		  :nonlinearp t
		  :in-theory (enable c rnd-const sum))))

(local-defthmd rdia-22
  (let* ((p (prec f))
         (s (spn f))
  	 (x (* s z))
	 (m (m k z))
	 (sum (sum m k p mode))
	 (sl (sl m k p mode))
	 (sh (sh m k p mode)))
    (implies (and (formatp f)
                  (rationalp z)
		  (< 0 z)
		  (< z 2)
		  (common-mode-p mode)
		  (natp k)
		  (>= k (+ p 2))
		  (natp m)
		  (< (expo m) (- k p)))
	    (> m 0)))
  :hints (("Goal" :in-theory (enable m))))

(local-defthmd rdia-23
  (let* ((p (prec f))
         (s (spn f))
  	 (x (* s z))
	 (m (m k z))
	 (sum (sum m k p mode))
	 (sl (sl m k p mode))
	 (sh (sh m k p mode)))
    (implies (and (formatp f)
                  (rationalp z)
		  (< 0 z)
		  (< z 2)
		  (common-mode-p mode)
		  (natp k)
		  (>= k (+ p 2))
		  (natp m)
		  (< (expo m) (- k p))
		  (or (= mode 'raz) (= mode 'rup)))
	    (equal (drnd x mode f)
	           (* (spd f) sh))))
  :hints (("Goal" :use (rdia-18 drnd-tiny-b rdia-7 rdia-17 rdia-8 rdia-22
                        (:instance expo>= (x (m k z)) (n (- k (prec f))))
			(:instance drnd-tiny-a (x (* (spn f) z)))
			(:instance drnd-tiny-c (x (* (spn f) z))))
		  :nonlinearp t
		  :in-theory (enable c rnd-const sum))))

(local-defthmd rdia-24
  (let* ((p (prec f))
         (s (spn f))
  	 (x (* s z))
	 (m (m k z))
	 (sum (sum m k p mode))
	 (sl (sl m k p mode))
	 (sh (sh m k p mode)))
    (implies (and (formatp f)
                  (rationalp z)
		  (< 0 z)
		  (< z 2)
		  (common-mode-p mode)
		  (natp k)
		  (>= k (+ p 2))
		  (natp m)
		  (< (expo m) (- k p))
		  (not (and (= mode 'rne) (= sl 0))))
	    (equal (drnd x mode f)
	           (* (spd f) sh))))
  :hints (("Goal" :use (rdia-19 rdia-20 rdia-21 rdia-23)
                  :in-theory (enable common-mode-p ieee-rounding-mode-p))))

(local-defthmd rdia-25
  (let* ((p (prec f))
         (s (spn f))
  	 (x (* s z))
	 (m (m k z))
	 (sum (sum m k p mode))
	 (sl (sl m k p mode))
	 (sh (sh m k p mode)))
    (implies (and (formatp f)
                  (rationalp z)
		  (< 0 z)
		  (< z 2)
		  (common-mode-p mode)
		  (natp k)
		  (>= k (+ p 2))
		  (natp m))
	    (equal (drnd x mode f)
	           (if (and (= mode 'rne) (= sl 0))
		       (* (expt 2 (- 2 p)) s (bits sh p 1))
		     (* (expt 2 (- 1 p)) s sh)))))
  :hints (("Goal" :use (rdia-5 rdia-12 rdia-19 rdia-24)
                  :in-theory (enable spn spd))))

(defthmd rnd-drnd-inject-a
  (let* ((p (prec f))
         (s (spn f))
  	 (x (* s z))
	 (m (* (expt 2 (1- k)) z))
	 (sum (+ m (rnd-const (1- k) mode p)))
	 (sl (bits sum (1- (- k p)) 0))
	 (sh (bits sum k (- k p))))
    (implies (and (formatp f)
                  (rationalp z)
		  (< 0 z)
		  (< z 2)
		  (common-mode-p mode)
		  (natp k)
		  (>= k (+ p 2))
		  (natp m)
		  (< (expo m) (- k p)))
	    (equal (drnd x mode f)
	           (if (and (= mode 'rne) (= sl 0))
		       (* (expt 2 (- 2 p)) s (bits sh p 1))
		     (* (expt 2 (- 1 p)) s sh)))))
  :hints (("Goal" :use (rdia-25)
                  :in-theory (enable m c sum sl sh))))

;;---------------------------------------------------------------------------------------------------------

(local-defthmd rdib-1
  (let* ((p (prec f))
         (s (spn f))
  	 (x (* s z))
	 (m (m k z))
	 (sum (sum m k p mode))
	 (sl (sl m k p mode))
	 (sh (sh m k p mode)))
    (implies (and (formatp f)
                  (rationalp z)
		  (< 0 z)
		  (< z 2)
		  (common-mode-p mode)
		  (natp k)
		  (>= k (+ p 2))
		  (natp m)
		  (>= (expo m) (- k p)))
	    (iff (= (sl m k p mode) (c k p mode))
	         (= (* (expt 2 (+ (- 1 k))) s m)
		    (drnd x mode f)))))
  :hints (("Goal" :use (rdia-3 (:instance m-lemma-b (p (prec f)) (m (m k z)))))))

(local-defthmd rdib-2
  (let* ((p (prec f))
         (s (spn f))
  	 (x (* s z))
	 (m (m k z))
	 (sum (sum m k p mode))
	 (sl (sl m k p mode))
	 (sh (sh m k p mode)))
    (implies (and (formatp f)
                  (rationalp z)
		  (< 0 z)
		  (< z 2)
		  (common-mode-p mode)
		  (natp k)
		  (>= k (+ p 2))
		  (natp m)
		  (>= (expo m) (- k p)))
	    (iff (= (sl m k p mode) (c k p mode))
	         (= x (drnd x mode f)))))
  :hints (("Goal" :in-theory (enable m) :use (rdib-1))))

(local-defthmd rdib-3
  (let* ((p (prec f))
         (s (spn f))
  	 (x (* s z))
	 (m (m k z))
	 (sum (sum m k p mode))
	 (sl (sl m k p mode))
	 (sh (sh m k p mode)))
    (implies (and (formatp f)
                  (rationalp z)
		  (< 0 z)
		  (< z 2)
		  (common-mode-p mode)
		  (natp k)
		  (>= k (+ p 2))
		  (natp m)
		  (< (expo m) (- k p)))
	    (not (equal (drnd x mode f) x))))
  :hints (("Goal" :use (rdia-7 drnd-tiny-b
			(:instance drnd-tiny-a (x (* (spn f) z)))
			(:instance drnd-tiny-c (x (* (spn f) z)))))))

(local-defthmd rdib-4
  (let* ((p (prec f))
         (s (spn f))
  	 (x (* s z))
	 (m (m k z))
	 (sum (sum m k p mode))
	 (sl (sl m k p mode))
	 (sh (sh m k p mode)))
    (implies (and (formatp f)
                  (rationalp z)
		  (< 0 z)
		  (< z 2)
		  (common-mode-p mode)
		  (natp k)
		  (>= k (+ p 2))
		  (natp m)
		  (< (expo m) (- k p)))
	    (not (equal (sl m k p mode) (c k p mode)))))
  :hints (("Goal" :use (rdia-22
                        (:instance dia-48 (p (prec f)) (m (m k z)))
                        (:instance expo>= (x (m k z)) (n (- k (prec f))))))))

(local-defthmd rdib-5
  (let* ((p (prec f))
         (s (spn f))
  	 (x (* s z))
	 (m (m k z))
	 (sum (sum m k p mode))
	 (sl (sl m k p mode))
	 (sh (sh m k p mode)))
    (implies (and (formatp f)
                  (rationalp z)
		  (< 0 z)
		  (< z 2)
		  (common-mode-p mode)
		  (natp k)
		  (>= k (+ p 2))
		  (natp m))
	    (iff (= (sl m k p mode) (c k p mode))
	         (= x (drnd x mode f)))))
  :hints (("Goal" :use (rdib-2 rdib-3 rdib-4))))

(defthmd rnd-drnd-inject-b
  (let* ((p (prec f))
         (s (spn f))
  	 (x (* s z))
	 (m (* (expt 2 (1- k)) z))
	 (c (rnd-const (1- k) mode p))
	 (sum (+ m c))
	 (sl (bits sum (1- (- k p)) 0)))
    (implies (and (formatp f)
                  (rationalp z)
		  (< 0 z)
		  (< z 2)
		  (common-mode-p mode)
		  (natp k)
		  (>= k (+ p 2))
		  (natp m))
	    (iff (= x (drnd x mode f))
	         (= sl c))))
  :hints (("Goal" :use (rdib-5)
                  :in-theory (enable m c sum sl sh))))


;;---------------------------------------------------------------------------------------------------------

(local-defthmd rdic-1
  (let* ((p (prec f))
         (s (spn f))
  	 (x (* s z))
	 (m (m k z))
	 (sum (sum m k p mode))
	 (sh (sh m k p mode)))
    (implies (and (formatp f)
                  (rationalp z)
		  (< 0 z)
		  (< z 2)
		  (common-mode-p mode)
		  (natp k)
		  (>= k (+ p 2))
		  (natp m)
		  (> sh (expt 2 (1- p))))
	    (>= (* sh (expt 2 (- k p)))
	        (+ (expt 2 (1- k)) (expt 2 (- k p))))))
  :hints (("Goal" :nonlinearp t)))

(local-defthmd rdic-2
  (let* ((p (prec f))
         (s (spn f))
  	 (x (* s z))
	 (m (m k z))
	 (sum (sum m k p mode))
	 (sh (sh m k p mode)))
    (implies (and (formatp f)
                  (rationalp z)
		  (< 0 z)
		  (< z 2)
		  (common-mode-p mode)
		  (natp k)
		  (>= k (+ p 2))
		  (natp m)
		  (< sum (expt 2 (1+ k))))
	    (>= sum (* sh (expt 2 (- k p))))))
  :hints (("Goal" :use (rdic-1 rdia-14 (:instance bits-plus-bits (x (sum (m k z) k (prec f) mode)) (n k) (p (- k (prec f))) (m 0)))
		  :cases ((< (sum (m k z) k (prec f) mode) (expt 2 (1+ k))))
		  :in-theory (enable sh bvecp))))

(local-defthmd rdic-3
  (let* ((p (prec f))
         (s (spn f))
  	 (x (* s z))
	 (m (m k z))
	 (sum (sum m k p mode))
	 (sh (sh m k p mode)))
    (implies (and (formatp f)
                  (rationalp z)
		  (< 0 z)
		  (< z 2)
		  (common-mode-p mode)
		  (natp k)
		  (>= k (+ p 2))
		  (natp m))
             (>= (expt 2 (1+ k))
	         (+ (expt 2 (1- k)) (expt 2 (- k p))))))
  :hints (("Goal" :nonlinearp t)))

(local-defthmd rdic-4
  (let* ((p (prec f))
         (s (spn f))
  	 (x (* s z))
	 (m (m k z))
	 (sum (sum m k p mode))
	 (sh (sh m k p mode)))
    (implies (and (formatp f)
                  (rationalp z)
		  (< 0 z)
		  (< z 2)
		  (common-mode-p mode)
		  (natp k)
		  (>= k (+ p 2))
		  (natp m)
		  (> sh (expt 2 (1- p))))
	    (>= sum (+ (expt 2 (1- k)) (expt 2 (- k p))))))
  :hints (("Goal" :use (rdic-1 rdic-2 rdic-3) :in-theory (theory 'minimal-theory))))

(local-defthmd rdic-5
  (let* ((p (prec f))
         (s (spn f))
  	 (x (* s z))
	 (m (m k z))
	 (sum (sum m k p mode))
	 (sh (sh m k p mode)))
    (implies (and (formatp f)
                  (rationalp z)
		  (< 0 z)
		  (< z 2)
		  (common-mode-p mode)
		  (natp k)
		  (>= k (+ p 2))
		  (natp m)
		  (> sh (expt 2 (1- p))))
	    (> m (expt 2 (1- k)))))
  :hints (("Goal" :use (rdic-4 (:instance dia-2 (p (prec f))))
                  :in-theory (enable bvecp sum))))

(local-defthmd rdic-6
  (let* ((p (prec f))
         (s (spn f))
  	 (x (* s z))
	 (m (m k z))
	 (sum (sum m k p mode))
	 (sh (sh m k p mode)))
    (implies (and (formatp f)
                  (rationalp z)
		  (< 0 z)
		  (< z 2)
		  (common-mode-p mode)
		  (natp k)
		  (>= k (+ p 2))
		  (natp m)
		  (> sh (expt 2 (1- p))))
	    (> x s)))
  :hints (("Goal" :use (rdic-5)
                  :in-theory (enable m))))

(local-defthmd rdic-7
  (Let* ((p (prec f))
         (s (spn f))
  	 (x (* s z))
	 (m (m k z))
	 (sum (sum m k p mode))
	 (sh (sh m k p mode)))
    (implies (and (formatp f)
                  (rationalp z)
		  (< 0 z)
		  (< z 2)
		  (common-mode-p mode)
		  (natp k)
		  (>= k (+ p 2))
		  (natp m)
		  (> sh (expt 2 (1- p))))
	    (>= (rnd x mode p) s)))
  :hints (("Goal" :use (rdic-6 (:instance rnd-exactp-d (x (* (spn f) z)) (a (spn f)) (n (prec f))))
                  :in-theory (enable exactp-2**n spn))))

(defthmd rnd-drnd-inject-c
  (let* ((p (prec f))
         (s (spn f))
  	 (x (* s z))
	 (m (* (expt 2 (1- k)) z))
	 (c (rnd-const (1- k) mode p))
	 (sum (+ m c))
	 (sh (bits sum k (- k p))))
    (implies (and (formatp f)
                  (rationalp z)
		  (< 0 z)
		  (< z 2)
		  (common-mode-p mode)
		  (natp k)
		  (>= k (+ p 2))
		  (natp m)
		  (> sh (expt 2 (1- p))))
	    (and (>= x s)
	         (>= (rnd x mode p) s))))
  :hints (("Goal" :use (rdic-6 rdic-7)
                  :in-theory (enable m c sum sh))))

;;---------------------------------------------------------------------------------------------------------

(local-defthmd rdid-1
  (let* ((p (prec f))
         (s (spn f))
  	 (x (* s z))
	 (m (m k z))
	 (sum (sum m k p mode))
	 (sh (sh m k p mode)))
    (implies (and (formatp f)
                  (rationalp z)
		  (< 0 z)
		  (< z 2)
		  (common-mode-p mode)
		  (natp k)
		  (>= k (+ p 2))
		  (natp m)
		  (< sh (expt 2 (1- p))))
	    (<= (* sh (expt 2 (- k p)))
	        (- (expt 2 (1- k)) (expt 2 (- k p))))))
  :hints (("Goal" :nonlinearp t)))

(local-defthmd rdid-2
  (let* ((p (prec f))
         (s (spn f))
  	 (x (* s z))
	 (m (m k z))
	 (sum (sum m k p mode))
	 (sh (sh m k p mode)))
    (implies (and (formatp f)
                  (rationalp z)
		  (< 0 z)
		  (< z 2)
		  (common-mode-p mode)
		  (natp k)
		  (>= k (+ p 2))
		  (natp m)
		  (< sum (expt 2 (1+ k))))
	    (<= sum (+ (* sh (expt 2 (- k p)))
	               (bits sum (1- (- k p)) 0)))))
  :hints (("Goal" :use (rdid-1 rdia-14 (:instance bits-plus-bits (x (sum (m k z) k (prec f) mode)) (n k) (p (- k (prec f))) (m 0)))
		  :in-theory (enable sh bvecp))))

(local-defthmd rdid-3
  (let* ((p (prec f))
         (s (spn f))
  	 (x (* s z))
	 (m (m k z))
	 (sum (sum m k p mode))
	 (sh (sh m k p mode)))
    (implies (and (formatp f)
                  (rationalp z)
		  (< 0 z)
		  (< z 2)
		  (common-mode-p mode)
		  (natp k)
		  (>= k (+ p 2))
		  (natp m)
		  (< sum (expt 2 (1+ k))))
	    (< sum (+ (* sh (expt 2 (- k p)))
	              (expt 2 (- k p))))))
  :hints (("Goal" :use (rdid-2
                        (:instance bits-bounds (x (sum (m k z) k (prec f) mode)) (i (1- (- k (prec f)))) (j 0))))))

(local-defthmd rdid-4
  (let* ((p (prec f))
         (s (spn f))
  	 (x (* s z))
	 (m (m k z))
	 (sum (sum m k p mode))
	 (sh (sh m k p mode)))
    (implies (and (formatp f)
                  (rationalp z)
		  (< 0 z)
		  (< z 2)
		  (common-mode-p mode)
		  (natp k)
		  (>= k (+ p 2))
		  (natp m))
	    (< sum (expt 2 (1+ k)))))
  :hints (("Goal" :use (bvecp-m (:instance dia-2 (p (prec f))))
                  :nonlinearp t
		  :in-theory (e/d (sum bvecp) (bvecp-m)))))

(local-defthmd rdid-5
  (let* ((p (prec f))
         (s (spn f))
  	 (x (* s z))
	 (m (m k z))
	 (sum (sum m k p mode))
	 (sh (sh m k p mode)))
    (implies (and (formatp f)
                  (rationalp z)
		  (< 0 z)
		  (< z 2)
		  (common-mode-p mode)
		  (natp k)
		  (>= k (+ p 2))
		  (natp m)
		  (< sh (expt 2 (1- p))))
	    (< sum (+ (* sh (expt 2 (- k p)))
	              (expt 2 (- k p))))))
  :hints (("Goal" :use (rdid-3 rdid-4))))

(local-defthmd rdid-6
  (let* ((p (prec f))
         (s (spn f))
  	 (x (* s z))
	 (m (m k z))
	 (sum (sum m k p mode))
	 (sh (sh m k p mode)))
    (implies (and (formatp f)
                  (rationalp z)
		  (< 0 z)
		  (< z 2)
		  (common-mode-p mode)
		  (natp k)
		  (>= k (+ p 2))
		  (natp m)
		  (< sh (expt 2 (1- p))))
	    (< sum (+ (- (expt 2 (1- k)) (expt 2 (- k p)))
	              (expt 2 (- k p))))))
  :hints (("Goal" :use (rdid-1 rdid-5) :in-theory (theory 'minimal-theory))))

(local-defthmd rdid-7
  (let* ((p (prec f))
         (s (spn f))
  	 (x (* s z))
	 (m (m k z))
	 (sum (sum m k p mode))
	 (sh (sh m k p mode)))
    (implies (and (formatp f)
                  (rationalp z)
		  (< 0 z)
		  (< z 2)
		  (common-mode-p mode)
		  (natp k)
		  (>= k (+ p 2))
		  (natp m)
		  (< sh (expt 2 (1- p))))
	    (< sum (expt 2 (1- k)))))
  :hints (("Goal" :use (rdid-6))))

(local-defthmd rdid-8
  (let* ((p (prec f))
         (s (spn f))
  	 (x (* s z))
	 (m (m k z))
	 (sum (sum m k p mode))
	 (sh (sh m k p mode)))
    (implies (and (formatp f)
                  (rationalp z)
		  (< 0 z)
		  (< z 2)
		  (common-mode-p mode)
		  (natp k)
		  (>= k (+ p 2))
		  (natp m))
	    (<= (drnd x mode f)
	        (* (expt 2 (- 1 k)) s sum))))
  :hints (("Goal" :use (rdia-25 rdid-4 rdia-14
                        (:instance bits-plus-bits (x (sum (m k z) k (prec f) mode)) (n k) (p (- k (prec f))) (m 0))
			(:instance bits-plus-bits (x (sum (m k z) k (prec f) mode)) (n k) (p (1+ (- k (prec f)))) (m 0)))
		  :in-theory (enable bvecp bits-bits sh))))

(local-defthmd rdid-9
  (let* ((p (prec f))
         (s (spn f))
  	 (x (* s z))
	 (m (m k z))
	 (sum (sum m k p mode))
	 (sh (sh m k p mode)))
    (implies (and (formatp f)
                  (rationalp z)
		  (< 0 z)
		  (< z 2)
		  (common-mode-p mode)
		  (natp k)
		  (>= k (+ p 2))
		  (natp m)
		  (< sh (expt 2 (1- p))))
	    (< (drnd x mode f) s)))
  :hints (("Goal" :use (rdid-7 rdid-8) :nonlinearp t)))

(local-defthmd rdid-10
  (let* ((p (prec f))
         (s (spn f))
  	 (x (* s z))
	 (m (m k z))
	 (sum (sum m k p mode))
	 (sh (sh m k p mode)))
    (implies (and (formatp f)
                  (rationalp z)
		  (< 0 z)
		  (< z 2)
		  (common-mode-p mode)
		  (natp k)
		  (>= k (+ p 2))
		  (natp m)
		  (< sh (expt 2 (1- p))))
	    (< m (expt 2 (1- k)))))
  :hints (("Goal" :use (rdid-7 (:instance dia-2 (p (prec f)))) :in-theory (enable sum bvecp))))

(local-defthmd rdid-11
  (let* ((p (prec f))
         (s (spn f))
  	 (x (* s z))
	 (m (m k z))
	 (sum (sum m k p mode))
	 (sh (sh m k p mode)))
    (implies (and (formatp f)
                  (rationalp z)
		  (< 0 z)
		  (< z 2)
		  (common-mode-p mode)
		  (natp k)
		  (>= k (+ p 2))
		  (natp m)
		  (< sh (expt 2 (1- p))))
	    (< x s)))
  :hints (("Goal" :use (rdid-10) :in-theory (enable m) :nonlinearp t)))

(defthmd drnd-pos
  (implies (and (formatp f)
                (rationalp x)
                (>= x 0)
	        (<= x (spn f))
	        (common-mode-p mode))
	   (>= (drnd x mode f) 0))
  :hints (("Goal" :in-theory (enable sgn spn exactp-2**n drnd-rewrite)
                  :use ((:instance rnd-monotone (x (spn f)) (y (+ x (spn f))) (n (prec f)))
		        (:instance rnd-exactp-b (x (spn f)) (n (prec f)))))))

(local-defthmd rdid-12
  (let* ((p (prec f))
         (s (spn f))
  	 (x (* s z))
	 (m (m k z))
	 (sum (sum m k p mode))
	 (sh (sh m k p mode)))
    (implies (and (formatp f)
                  (rationalp z)
		  (< 0 z)
		  (< z 2)
		  (common-mode-p mode)
		  (natp k)
		  (>= k (+ p 2))
		  (natp m)
		  (< sh (expt 2 (1- p))))
	    (not (equal (rnd x mode p) s))))
  :hints (("Goal" :nonlinearp t :use (rdid-9 rdid-11
                                      (:instance drnd-pos (x (* (spn f) z)))
                                      (:instance rnd-drnd-up (x (* (spn f) z)))))))

(local-defthmd rdid-13
  (let* ((p (prec f))
         (s (spn f))
  	 (x (* s z))
	 (m (m k z))
	 (sum (sum m k p mode))
	 (sh (sh m k p mode)))
    (implies (and (formatp f)
                  (rationalp z)
		  (< 0 z)
		  (< z 2)
		  (common-mode-p mode)
		  (natp k)
		  (>= k (+ p 2))
		  (natp m)
		  (< sh (expt 2 (1- p))))
	    (< (rnd x mode p) s)))
  :hints (("Goal" :in-theory (enable spn exactp-2**n)
                  :use (rdid-12 rdid-11
                        (:instance rnd-exactp-b (x (spn f)) (n (prec f)))
			(:instance rnd-exactp-c (x (* (spn f) z)) (a (spn f)) (n (prec f)))))))

(defthmd rnd-drnd-inject-d
  (let* ((p (prec f))
         (s (spn f))
  	 (x (* s z))
	 (m (* (expt 2 (1- k)) z))
	 (c (rnd-const (1- k) mode p))
	 (sum (+ m c))
	 (sh (bits sum k (- k p))))
    (implies (and (formatp f)
                  (rationalp z)
		  (< 0 z)
		  (< z 2)
		  (common-mode-p mode)
		  (natp k)
		  (>= k (+ p 2))
		  (natp m)
		  (< sh (expt 2 (1- p))))
	    (and (< x s)
	         (< (rnd x mode p) s))))
  :hints (("Goal" :use (rdid-11 rdid-13)
                  :in-theory (enable m c sum sh))))

;;---------------------------------------------------------------------------------------------------------

(local-defthmd rdie-1
  (let* ((p (prec f))
         (s (spn f))
  	 (x (* s z))
	 (m (m k z))
	 (sum (sum m k p mode))
	 (sh (sh m k p mode)))
    (implies (and (formatp f)
                  (rationalp z)
		  (< 0 z)
		  (< z 2)
		  (common-mode-p mode)
		  (natp k)
		  (>= k (+ p 2))
		  (natp m)
		  (= sh (expt 2 (1- p))))
	    (>= (* sh (expt 2 (- k p)))
	        (expt 2 (1- k)))))
  :hints (("Goal" :nonlinearp t)))

(local-defthmd rdie-2
  (let* ((p (prec f))
         (s (spn f))
  	 (x (* s z))
	 (m (m k z))
	 (sum (sum m k p mode))
	 (sh (sh m k p mode)))
    (implies (and (formatp f)
                  (rationalp z)
		  (< 0 z)
		  (< z 2)
		  (common-mode-p mode)
		  (natp k)
		  (>= k (+ p 2))
		  (natp m)
		  (= sh (expt 2 (1- p))))
	    (>= sum (* sh (expt 2 (- k p))))))
  :hints (("Goal" :use (rdie-1 rdid-4 rdia-14 (:instance bits-plus-bits (x (sum (m k z) k (prec f) mode)) (n k) (p (- k (prec f))) (m 0)))
		  :in-theory (enable sh bvecp))))

(local-defthmd rdie-3
  (let* ((p (prec f))
         (s (spn f))
  	 (x (* s z))
	 (m (m k z))
	 (sum (sum m k p mode))
	 (sh (sh m k p mode)))
    (implies (and (formatp f)
                  (rationalp z)
		  (< 0 z)
		  (< z 2)
		  (common-mode-p mode)
		  (natp k)
		  (>= k (+ p 2))
		  (natp m)
		  (= sh (expt 2 (1- p))))
	    (>= sum (expt 2 (1- k)))))
  :hints (("Goal" :use (rdie-1 rdie-2))))

(local-defthmd rdie-4
  (let* ((p (prec f))
         (s (spn f))
  	 (x (* s z))
	 (m (m k z))
	 (sum (sum m k p mode))
	 (sh (sh m k p mode)))
    (implies (and (formatp f)
                  (rationalp z)
		  (< 0 z)
		  (< z 2)
		  (common-mode-p mode)
		  (natp k)
		  (>= k (+ p 2))
		  (natp m)
		  (= sh (expt 2 (1- p)))
		  (or (= mode 'rtz) (= mode 'rdn)))
	    (>= m (expt 2 (1- k)))))
  :hints (("Goal" :in-theory (enable sum c rnd-const)
                  :use (rdie-3))))

(local-defthmd rdie-5
  (let* ((p (prec f))
         (s (spn f))
  	 (x (* s z))
	 (m (m k z))
	 (sum (sum m k p mode))
	 (sh (sh m k p mode)))
    (implies (and (formatp f)
                  (rationalp z)
		  (< 0 z)
		  (< z 2)
		  (common-mode-p mode)
		  (natp k)
		  (>= k (+ p 2))
		  (natp m)
		  (= sh (expt 2 (1- p)))
		  (or (= mode 'rtz) (= mode 'rdn)))
	    (>= x s)))
  :hints (("Goal" :use (rdie-4):nonlinearp t
                  :in-theory (enable m))))

(local-defthmd rdie-6
  (Let* ((p (prec f))
         (s (spn f))
  	 (x (* s z))
	 (m (m k z))
	 (sum (sum m k p mode))
	 (sh (sh m k p mode)))
    (implies (and (formatp f)
                  (rationalp z)
		  (< 0 z)
		  (< z 2)
		  (common-mode-p mode)
		  (natp k)
		  (>= k (+ p 2))
		  (natp m)
		  (= sh (expt 2 (1- p)))
		  (or (= mode 'rtz) (= mode 'rdn)))
	    (>= (rnd x mode p) s)))
  :hints (("Goal" :use (rdie-5 (:instance rnd-exactp-d (x (* (spn f) z)) (a (spn f)) (n (prec f))))
                  :in-theory (enable exactp-2**n spn))))

(defthmd rnd-drnd-inject-e
  (let* ((p (prec f))
         (s (spn f))
  	 (x (* s z))
	 (m (* (expt 2 (1- k)) z))
	 (c (rnd-const (1- k) mode p))
	 (sum (+ m c))
	 (sh (bits sum k (- k p))))
    (implies (and (formatp f)
                  (rationalp z)
		  (< 0 z)
		  (< z 2)
		  (common-mode-p mode)
		  (natp k)
		  (>= k (+ p 2))
		  (natp m)
		  (= sh (expt 2 (1- p)))
		  (or (= mode 'rtz) (= mode 'rdn)))
	    (and (>= x s)
	         (>= (rnd x mode p) s))))
  :hints (("Goal" :use (rdie-6 rdie-5)
                  :in-theory (enable m c sum sh))))

;;---------------------------------------------------------------------------------------------------------

(local-defthmd rdif-1
  (let* ((p (prec f))
         (s (spn f))
  	 (x (* s z))
	 (m (m k z))
	 (sum (sum m k p mode))
	 (sl (sl m k p mode))
	 (sh (sh m k p mode)))
    (implies (and (formatp f)
                  (rationalp z)
		  (< 0 z)
		  (< z 2)
		  (common-mode-p mode)
		  (natp k)
		  (>= k (+ p 2))
		  (natp m)
		  (= sh (expt 2 (1- p))))
	    (equal sh (mod (fl (/ sum (expt 2 (- k p))))
	                   (expt 2 (1+ p))))))
  :hints (("Goal" :in-theory (enable sh)
                  :use ((:instance bits-mod-fl (x (sum (m k z) k (prec f) mode)) (i (1+ k)) (j (- k (prec f))))))))

(local-defthmd rdif-2
  (let* ((p (prec f))
         (s (spn f))
  	 (x (* s z))
	 (m (m k z))
	 (sum (sum m k p mode))
	 (sl (sl m k p mode))
	 (sh (sh m k p mode)))
    (implies (and (formatp f)
                  (rationalp z)
		  (< 0 z)
		  (< z 2)
		  (common-mode-p mode)
		  (natp k)
		  (>= k (+ p 2))
		  (natp m)
		  (= sh (expt 2 (1- p))))
	    (< (/ sum (expt 2 (- k p)))
	       (expt 2 (1+ p)))))
  :hints (("Goal" :use (rdid-4) :nonlinearp t)))

(local-defthmd rdif-3
  (let* ((p (prec f))
         (s (spn f))
  	 (x (* s z))
	 (m (m k z))
	 (sum (sum m k p mode))
	 (sl (sl m k p mode))
	 (sh (sh m k p mode)))
    (implies (and (formatp f)
                  (rationalp z)
		  (< 0 z)
		  (< z 2)
		  (common-mode-p mode)
		  (natp k)
		  (>= k (+ p 2))
		  (natp m)
		  (= sh (expt 2 (1- p))))
	    (equal sh (fl (/ sum (expt 2 (- k p)))))))
  :hints (("Goal" :nonlinearp t :use (rdia-14 rdif-2 rdif-1))))

(local-defthmd rdif-4
  (let* ((p (prec f))
         (s (spn f))
  	 (x (* s z))
	 (m (m k z))
	 (sum (sum m k p mode))
	 (sl (sl m k p mode))
	 (sh (sh m k p mode)))
    (implies (and (formatp f)
                  (rationalp z)
		  (< 0 z)
		  (< z 2)
		  (common-mode-p mode)
		  (natp k)
		  (>= k (+ p 2))
		  (natp m)
		  (= sh (expt 2 (1- p))))
	    (equal (fl (/ sum (expt 2 (- k p))))
	           (+ (bits m (1- k) (- k p))
		      (fl (/ (+ (bits m (1- (- k p)) 0)
		                (c k p mode))
			     (expt 2 (- k p))))))))
  :hints (("Goal" :in-theory (enable sum)
                  :use ((:instance bits-plus-bits (x (m k z)) (n (1- k)) (p (- k (prec f))) (m 0))))))

(local-defthmd rdif-5
  (let* ((p (prec f))
         (s (spn f))
  	 (x (* s z))
	 (m (m k z))
	 (sum (sum m k p mode))
	 (sl (sl m k p mode))
	 (sh (sh m k p mode)))
    (implies (and (formatp f)
                  (rationalp z)
		  (< 0 z)
		  (< z 2)
		  (common-mode-p mode)
		  (natp k)
		  (>= k (+ p 2))
		  (natp m)
		  (= sh (expt 2 (1- p))))
	    (and (< (+ (bits m (1- (- k p)) 0) (c k p mode))
		    (* 2 (expt 2 (- k p))))
	         (>= (+ (bits m (1- (- k p)) 0) (c k p mode))
		     0))))
  :hints (("Goal" :use ((:instance dia-2 (p (prec f))) (:instance bits-bounds (x (m k z)) (i (1- (- k (prec f)))) (j 0)))
                  :nonlinearp t
		  :in-theory (e/d (bvecp) (dia-2)))))

(local-defthmd hack-4
  (implies (and (rationalp x) (posp n) (>= x 0) (< x (* 2 n)))
           (equal (fl (/ x n))
	          (if (>= x n) 1 0)))
  :hints (("Goal" :nonlinearp t)))

(local-defthmd rdif-6
  (let* ((p (prec f))
         (s (spn f))
  	 (x (* s z))
	 (m (m k z))
	 (sum (sum m k p mode))
	 (sl (sl m k p mode))
	 (sh (sh m k p mode)))
    (implies (and (formatp f)
                  (rationalp z)
		  (rationalp (+ (bits m (1- (- k p)) 0) (c k p mode)))
		  (posp (expt 2 (- k p)))
		  (< 0 z)
		  (< z 2)
		  (common-mode-p mode)
		  (natp k)
		  (>= k (+ p 2))
		  (natp m)
		  (= sh (expt 2 (1- p))))
	    (equal (fl (/ (+ (bits m (1- (- k p)) 0) (c k p mode))
		          (expt 2 (- k p))))
	           (if (>= (+ (bits m (1- (- k p)) 0) (c k p mode))
		           (expt 2 (- k p)))
		       1 0))))
  :hints (("Goal" :in-theory (theory 'minimal-theory)
                  :use (rdif-5 (:instance hack-4 (x (+ (bits (m k z) (1- (- k (prec f))) 0) (c k (prec f) mode))) (n (expt 2 (- k (prec f)))))))))

(local-defthmd rdif-7
  (let* ((p (prec f))
         (s (spn f))
  	 (x (* s z))
	 (m (m k z))
	 (sum (sum m k p mode))
	 (sl (sl m k p mode))
	 (sh (sh m k p mode)))
    (implies (and (formatp f)
                  (rationalp z)
		  (< 0 z)
		  (< z 2)
		  (common-mode-p mode)
		  (natp k)
		  (>= k (+ p 2))
		  (natp m)
		  (= sh (expt 2 (1- p))))
	    (equal (fl (/ (+ (bits m (1- (- k p)) 0) (c k p mode))
		          (expt 2 (- k p))))
	           (if (>= (+ (bits m (1- (- k p)) 0) (c k p mode))
		           (expt 2 (- k p)))
		       1 0))))
  :hints (("Goal" :use (rdif-6))))

(local-defthmd rdif-8
  (let* ((p (prec f))
         (s (spn f))
  	 (x (* s z))
	 (m (m k z))
	 (sum (sum m k p mode))
	 (sl (sl m k p mode))
	 (sh (sh m k p mode)))
    (implies (and (formatp f)
                  (rationalp z)
		  (< 0 z)
		  (< z 2)
		  (common-mode-p mode)
		  (natp k)
		  (>= k (+ p 2))
		  (natp m)
		  (= sh (expt 2 (1- p))))
	    (equal (expt 2 (1- p))
	           (+ (bits m (1- k) (- k p))
		      (if (>= (+ (bits m (1- (- k p)) 0) (c k p mode))
		              (expt 2 (- k p)))
		          1 0)))))
  :hints (("Goal" :use (rdif-3 rdif-4 rdif-7))))

(local-defthmd hack-5
  (implies (and (natp a) (posp b) (natp c) (posp p) (< c b) (= m (+ (* a b) c)) (= k (* p b)))
           (iff (< m k)
	        (< a p)))
  :hints (("Goal" :nonlinearp t)))

(local-defthmd rdif-9
  (let* ((p (prec f))
         (s (spn f))
  	 (x (* s z))
	 (m (m k z))
	 (sum (sum m k p mode))
	 (sl (sl m k p mode))
	 (sh (sh m k p mode)))
    (implies (and (formatp f)
                  (rationalp z)
		  (< 0 z)
		  (< z 2)
		  (common-mode-p mode)
		  (natp k)
		  (>= k (+ p 2))
		  (natp m)
		  (= sh (expt 2 (1- p)))
		  (natp (bits m (1- k) (- k p)))
		  (posp (expt 2 (- k p)))
		  (natp (bits m (1- (- k p)) 0))
		  (< (bits m (1- (- k p)) 0) (expt 2 (- k p)))
		  (posp (expt 2 (1- p)))
		  (= m (+ (* (bits m (1- k) (- k p)) (expt 2 (- k p))) (bits m (1- (- k p)) 0)))
		  (= (expt 2 (1- k)) (* (expt 2 (1- p)) (expt 2 (- k p)))))
	    (iff (< m (expt 2 (1- k)))
	         (< (bits m (1- k) (- k p)) (expt 2 (1- p))))))
  :hints (("Goal" :use ((:instance hack-5 (a (bits (m k z) (1- k) (- k (prec f))))
                                          (b (expt 2 (- k (prec f))))
					  (c (bits (m k z) (1- (- k (prec f))) 0))
					  (m (m k z))
					  (k (expt 2 (1- k)))
					  (p (expt 2 (1- (prec f))))))
		  :in-theory (theory 'minimal-theory))))

(local-defthmd rdif-10
  (let* ((p (prec f))
         (s (spn f))
  	 (x (* s z))
	 (m (m k z))
	 (sum (sum m k p mode))
	 (sl (sl m k p mode))
	 (sh (sh m k p mode)))
    (implies (and (formatp f)
                  (rationalp z)
		  (< 0 z)
		  (< z 2)
		  (common-mode-p mode)
		  (natp k)
		  (>= k (+ p 2))
		  (natp m)
		  (= sh (expt 2 (1- p))))
             (and (natp (bits m (1- k) (- k p)))
		  (posp (expt 2 (- k p)))
		  (natp (bits m (1- (- k p)) 0))
		  (< (bits m (1- (- k p)) 0) (expt 2 (- k p)))
		  (posp (expt 2 (1- p)))
		  (= m (+ (* (bits m (1- k) (- k p)) (expt 2 (- k p))) (bits m (1- (- k p)) 0)))
		  (= (expt 2 (1- k)) (* (expt 2 (1- p)) (expt 2 (- k p)))))))
  :hints (("Goal" :use ((:instance bits-plus-bits (x (m k z)) (n (1- k)) (p (- k (prec f))) (m 0))
                        (:instance bits-bounds (x (m k z)) (i (1- (- k (prec f)))) (j 0))))))

(local-defthmd rdif-11
  (let* ((p (prec f))
         (s (spn f))
  	 (x (* s z))
	 (m (m k z))
	 (sum (sum m k p mode))
	 (sl (sl m k p mode))
	 (sh (sh m k p mode)))
    (implies (and (formatp f)
                  (rationalp z)
		  (< 0 z)
		  (< z 2)
		  (common-mode-p mode)
		  (natp k)
		  (>= k (+ p 2))
		  (natp m)
		  (= sh (expt 2 (1- p))))
	    (iff (< m (expt 2 (1- k)))
	         (< (bits m (1- k) (- k p)) (expt 2 (1- p))))))
  :hints (("Goal" :use (rdif-9 rdif-10)
		  :in-theory (theory 'minimal-theory))))

(local-defthmd rdif-12
  (let* ((p (prec f))
         (s (spn f))
  	 (x (* s z))
	 (m (m k z))
	 (sum (sum m k p mode))
	 (sl (sl m k p mode))
	 (sh (sh m k p mode)))
    (implies (and (formatp f)
                  (rationalp z)
		  (< 0 z)
		  (< z 2)
		  (common-mode-p mode)
		  (natp k)
		  (>= k (+ p 2))
		  (natp m)
		  (= sh (expt 2 (1- p)))
		  (or (= mode 'raz) (= mode 'rup)))
	    (iff (< m (expt 2 (1- k)))
	         (> (bits m (1- (- k p)) 0) 0))))
  :hints (("Goal" :in-theory (enable c rnd-const) :use (rdif-11 rdif-8))))

(local-defthmd rdif-13
  (let* ((p (prec f))
         (s (spn f))
  	 (x (* s z))
	 (m (m k z))
	 (sum (sum m k p mode))
	 (sl (sl m k p mode))
	 (sh (sh m k p mode)))
    (implies (and (formatp f)
                  (rationalp z)
		  (< 0 z)
		  (< z 2)
		  (common-mode-p mode)
		  (natp k)
		  (>= k (+ p 2))
		  (natp m)
		  (= sh (expt 2 (1- p)))
		  (or (= mode 'raz) (= mode 'rup)))
	    (iff (< m (expt 2 (1- k)))
	         (not (= sl (c k p mode))))))
  :hints (("Goal" :in-theory (enable bits-mod) :use (rdif-12 (:instance dia-48 (p (prec f)) (m (m k z)))))))

(local-defthmd rdif-14
  (let* ((p (prec f))
         (s (spn f))
  	 (x (* s z))
	 (m (m k z))
	 (sum (sum m k p mode))
	 (sl (sl m k p mode))
	 (sh (sh m k p mode)))
    (implies (and (formatp f)
                  (rationalp z)
		  (< 0 z)
		  (< z 2)
		  (common-mode-p mode)
		  (natp k)
		  (>= k (+ p 2))
		  (natp m)
		  (= sh (expt 2 (1- p))))
	    (iff (< m (expt 2 (1- k)))
	         (< x s))))
  :hints (("Goal" :in-theory (enable m) :nonlinearp t)))

(defthmd rnd-drnd-inject-f-i
  (let* ((p (prec f))
         (s (spn f))
  	 (x (* s z))
	 (m (* (expt 2 (1- k)) z))
	 (c (rnd-const (1- k) mode p))
	 (sum (+ m c))
	 (sh (bits sum k (- k p)))
	 (sl (bits sum (1- (- k p)) 0)))
    (implies (and (formatp f)
                  (rationalp z)
		  (< 0 z)
		  (< z 2)
		  (common-mode-p mode)
		  (natp k)
		  (>= k (+ p 2))
		  (natp m)
		  (= sh (expt 2 (1- p)))
		  (or (= mode 'raz) (= mode 'rup)))
	    (iff (< x s)
	         (not (= sl c)))))
  :hints (("Goal" :use (rdif-13 rdif-14)
                  :in-theory (enable m c sum sh sl))))

;;---------------------------------------------------------------------------------------------------------

(local-defthmd rdif-15
  (let* ((p (prec f))
         (s (spn f))
  	 (x (* s z))
	 (m (m k z))
	 (sum (sum m k p mode))
	 (sl (sl m k p mode))
	 (sh (sh m k p mode))
	 (g (bitn sum (1- (- k p)))))
    (implies (and (formatp f)
                  (rationalp z)
		  (< 0 z)
		  (< z 2)
		  (common-mode-p mode)
		  (natp k)
		  (>= k (+ p 2))
		  (natp m)
		  (= sh (expt 2 (1- p)))
		  (or (= mode 'raz) (= mode 'rup))
		  (>= x s))
	    (equal g (bitn (1- (expt 2 (- k p))) (1- (- k p))))))
  :hints (("Goal" :in-theory (enable c rnd-const sl)
                  :use (rdif-14 rdif-13
                        (:instance bitn-bits (x (sum (m k z) k (prec f) mode)) (i (1- (- k (prec f)))) (j 0) (k (1- (- k (prec f)))))))))

(local-defthmd rdif-16
  (let* ((p (prec f))
         (s (spn f))
  	 (x (* s z))
	 (m (m k z))
	 (sum (sum m k p mode))
	 (sl (sl m k p mode))
	 (sh (sh m k p mode))
	 (g (bitn sum (1- (- k p)))))
    (implies (and (formatp f)
                  (rationalp z)
		  (< 0 z)
		  (< z 2)
		  (common-mode-p mode)
		  (natp k)
		  (>= k (+ p 2))
		  (natp m)
		  (= sh (expt 2 (1- p)))
		  (or (= mode 'raz) (= mode 'rup))
		  (>= x s))
	    (equal (bitn (1- (expt 2 (- k p))) (1- (- k p)))
	           1)))
  :hints (("Goal" :in-theory (enable bvecp)
                  :nonlinearp t
                  :use ((:instance bvecp-bitn-1 (x (1- (expt 2 (- k (prec f))))) (n (1- (- k (prec f)))))))))

(local-defthmd rdif-17
  (let* ((p (prec f))
         (s (spn f))
  	 (x (* s z))
	 (m (m k z))
	 (sum (sum m k p mode))
	 (sl (sl m k p mode))
	 (sh (sh m k p mode))
	 (g (bitn sum (1- (- k p)))))
    (implies (and (formatp f)
                  (rationalp z)
		  (< 0 z)
		  (< z 2)
		  (common-mode-p mode)
		  (natp k)
		  (>= k (+ p 2))
		  (natp m)
		  (= sh (expt 2 (1- p)))
		  (or (= mode 'raz) (= mode 'rup))
		  (>= x s))
	    (equal g 1)))
  :hints (("Goal" :use (rdif-15 rdif-16))))

(local-defthmd rdif-18
  (let* ((p (prec f))
         (s (spn f))
  	 (x (* s z))
	 (m (m k z))
	 (sum (sum m k p mode))
	 (sl (sl m k p mode))
	 (sh (sh m k p mode))
	 (g (bitn sum (1- (- k p)))))
    (implies (and (formatp f)
                  (rationalp z)
		  (< 0 z)
		  (< z 2)
		  (common-mode-p mode)
		  (natp k)
		  (>= k (+ p 2))
		  (natp m)
		  (= sh (expt 2 (1- p)))
		  (or (= mode 'raz) (= mode 'rup))
		  (< x s))
	    (and (equal (bits m (1- k) (- k p))
	                (1- (expt 2 (1- p))))
		 (> (bits m (1- (- k p)) 0) 0))))
  :hints (("Goal" :use (rdif-14 rdif-12 rdif-11 rdif-8))))

(local-defthmd rdif-19
  (let* ((p (prec f))
         (s (spn f))
  	 (x (* s z))
	 (m (m k z))
	 (sum (sum m k p mode))
	 (sl (sl m k p mode))
	 (sh (sh m k p mode))
	 (g (bitn sum (1- (- k p)))))
    (implies (and (formatp f)
                  (rationalp z)
		  (< 0 z)
		  (< z 2)
		  (common-mode-p mode)
		  (natp k)
		  (>= k (+ p 2))
		  (natp m)
		  (= sh (expt 2 (1- p)))
		  (or (= mode 'raz) (= mode 'rup))
		  (< x s))
	    (equal m (+ (- (expt 2 (1- k)) (expt 2 (- k p))) (bits m (1- (- k p)) 0)))))
  :hints (("Goal" :use (rdif-18 (:instance bits-plus-bits (x (m k z)) (n (1- k)) (p (- k (prec f))) (m 0))))))

(local-defthmd rdif-20
  (let* ((p (prec f))
         (s (spn f))
  	 (x (* s z))
	 (m (m k z))
	 (sum (sum m k p mode))
	 (sl (sl m k p mode))
	 (sh (sh m k p mode))
	 (g (bitn sum (1- (- k p))))
	 (e (- (expt 2 (1- k)) (expt 2 (1- (- k p))))))
    (implies (and (formatp f)
                  (rationalp z)
		  (< 0 z)
		  (< z 2)
		  (common-mode-p mode)
		  (natp k)
		  (>= k (+ p 2))
		  (natp m)
		  (= sh (expt 2 (1- p)))
		  (< x s))
	    (equal (expo e) (- k 2))))
  :hints (("Goal" :use ((:instance expo>= (x (- (expt 2 (1- k)) (expt 2 (1- (- k (prec f)))))) (n (- k 2)))
                        (:instance expo<= (x (- (expt 2 (1- k)) (expt 2 (1- (- k (prec f)))))) (n (- k 2))))
		  :nonlinearp t)))

(local-defthmd rdif-21
  (let* ((p (prec f))
         (s (spn f))
  	 (x (* s z))
	 (m (m k z))
	 (sum (sum m k p mode))
	 (sl (sl m k p mode))
	 (sh (sh m k p mode))
	 (g (bitn sum (1- (- k p))))
	 (e (- (expt 2 (1- k)) (expt 2 (1- (- k p))))))
    (implies (and (formatp f)
                  (rationalp z)
		  (< 0 z)
		  (< z 2)
		  (common-mode-p mode)
		  (natp k)
		  (>= k (+ p 2))
		  (natp m)
		  (= sh (expt 2 (1- p)))
		  (< x s))
	    (exactp e p)))
  :hints (("Goal" :use (rdif-20) :in-theory (enable exactp2))))

(local-defthmd rdif-22
  (let* ((p (prec f))
         (s (spn f))
  	 (x (* s z))
	 (m (m k z))
	 (sum (sum m k p mode))
	 (sl (sl m k p mode))
	 (sh (sh m k p mode))
	 (g (bitn sum (1- (- k p))))
	 (e (- (expt 2 (1- k)) (expt 2 (1- (- k p))))))
    (implies (and (formatp f)
                  (rationalp z)
		  (< 0 z)
		  (< z 2)
		  (common-mode-p mode)
		  (natp k)
		  (>= k (+ p 2))
		  (natp m)
		  (= sh (expt 2 (1- p)))
		  (< x s))
	    (equal (fp+ e p) (expt 2 (1- k)))))
  :hints (("Goal" :use (rdif-20))))

(local-defthmd rdif-23
  (let* ((p (prec f))
         (s (spn f))
  	 (x (* s z))
	 (m (m k z))
	 (sum (sum m k p mode))
	 (sl (sl m k p mode))
	 (sh (sh m k p mode))
	 (g (bitn sum (1- (- k p))))
	 (e (- (expt 2 (1- k)) (expt 2 (1- (- k p))))))
    (implies (and (formatp f)
                  (rationalp z)
		  (< 0 z)
		  (< z 2)
		  (common-mode-p mode)
		  (natp k)
		  (>= k (+ p 2))
		  (natp m)
		  (= sh (expt 2 (1- p)))
		  (< x s))
	    (iff (<= m e)
	         (< (raz m p) (expt 2 (1- k))))))
  :hints (("Goal" :in-theory (disable fp+)
                  :use (rdif-22 rdif-21
                        (:instance raz-exactp-c (x (m k z)) (n (prec f)) (a (- (expt 2 (1- k)) (expt 2 (1- (- k (prec f)))))))
			(:instance raz-lower-pos (x (m k z)) (n (prec f)))
			(:instance raz-exactp-a (x (m k z)) (n (prec f)))
			(:instance fp+2 (n (prec f)) (x (- (expt 2 (1- k)) (expt 2 (1- (- k (prec f)))))) (y (raz (m k z) (prec f))))))))

(local-defthmd rdif-24
  (let* ((p (prec f))
         (s (spn f))
  	 (x (* s z))
	 (m (m k z))
	 (sum (sum m k p mode))
	 (sl (sl m k p mode))
	 (sh (sh m k p mode))
	 (g (bitn sum (1- (- k p))))
	 (e (- (expt 2 (1- k)) (expt 2 (1- (- k p))))))
    (implies (and (formatp f)
                  (rationalp z)
		  (< 0 z)
		  (< z 2)
		  (common-mode-p mode)
		  (natp k)
		  (>= k (+ p 2))
		  (natp m))
	    (iff (< (rnd x mode p) s)
	         (< (rnd m mode p) (expt 2 (1- k))))))
  :hints (("Goal" :in-theory (enable m) :nonlinearp t
                  :use (expo-s (:instance rnd-shift (x (m k z)) (k (+ (- 1 k) (expo (spn f)))) (n (prec f)))))))

(local-defthmd rdif-25
  (let* ((p (prec f))
         (s (spn f))
  	 (x (* s z))
	 (m (m k z))
	 (sum (sum m k p mode))
	 (sl (sl m k p mode))
	 (sh (sh m k p mode))
	 (g (bitn sum (1- (- k p))))
	 (e (- (expt 2 (1- k)) (expt 2 (1- (- k p))))))
    (implies (and (formatp f)
                  (rationalp z)
		  (< 0 z)
		  (< z 2)
		  (common-mode-p mode)
		  (natp k)
		  (>= k (+ p 2))
		  (natp m)
		  (= sh (expt 2 (1- p)))
		  (or (= mode 'raz) (= mode 'rup))
		  (< x s)
		  (= (bitn m (1- (- k p))) 0))
	    ( < m e)))
  :hints (("Goal" :nonlinearp t
                  :use (rdif-19
                        (:instance bits-bounds (x (m k z)) (i (- k (+ 2 (prec f)))) (j 0))
                        (:instance bitn-plus-bits (x (m k z)) (n (1- (- k (prec f)))) (m 0))))))

(local-defthmd rdif-26
  (let* ((p (prec f))
         (s (spn f))
  	 (x (* s z))
	 (m (m k z))
	 (sum (sum m k p mode))
	 (sl (sl m k p mode))
	 (sh (sh m k p mode)))
    (implies (and (formatp f)
                  (rationalp z)
		  (< 0 z)
		  (< z 2)
		  (common-mode-p mode)
		  (natp k)
		  (>= k (+ p 2))
		  (natp m)
		  (= sh (expt 2 (1- p))))
	    (iff (< x s)
                 (>= (+ (bits m (1- (- k p)) 0) (c k p mode))
	             (expt 2 (- k p))))))
  :hints (("Goal" :in-theory (enable c rnd-const) :use (rdif-14 rdif-11 rdif-8))))

(local-defthmd rdif-27
  (let* ((p (prec f))
         (s (spn f))
  	 (x (* s z))
	 (m (m k z))
	 (sum (sum m k p mode))
	 (sl (sl m k p mode))
	 (sh (sh m k p mode)))
    (implies (and (formatp f)
                  (rationalp z)
		  (< 0 z)
		  (< z 2)
		  (common-mode-p mode)
		  (natp k)
		  (>= k (+ p 2))
		  (natp m)
		  (= sh (expt 2 (1- p)))
		  (or (= mode 'raz) (= mode 'rup))
	          (< x s)
		  (= (bitn m (1- (- k p))) 0))
             (<  (+ (bits m (1- (- k p)) 0) (c k p mode))
	         (+ (expt 2 (- k p))
		    (expt 2 (1- (- k p)))))))
  :hints (("Goal" :in-theory (enable c rnd-const)
                  :use (rdif-26
                        (:instance bits-bounds (x (m k z)) (i (- k (+ 2 (prec f)))) (j 0))
                        (:instance bitn-plus-bits (x (m k z)) (n (1- (- k (prec f)))) (m 0))))))

(local-defthmd rdif-28
  (let* ((p (prec f))
         (s (spn f))
  	 (x (* s z))
	 (m (m k z))
	 (sum (sum m k p mode))
	 (sl (sl m k p mode))
	 (sh (sh m k p mode)))
    (implies (and (formatp f)
                  (rationalp z)
		  (< 0 z)
		  (< z 2)
		  (common-mode-p mode)
		  (natp k)
		  (>= k (+ p 2))
		  (natp m)
		  (= sh (expt 2 (1- p)))
		  (or (= mode 'raz) (= mode 'rup))
	          (< x s)
		  (= (bitn m (1- (- k p))) 0))
             (<  (mod (+ (mod m (expt 2 (- k p)))
	                 (c k p mode))
		      (expt 2 (- k p)))
	         (expt 2 (1- (- k p))))))
  :hints (("Goal" :use (rdif-26 rdif-27
                        (:instance mod-bnd-3 (m (+ (mod (m k z) (expt 2 (- k (prec f)))) (c k (prec f) mode)))
			                     (n (expt 2 (- k (prec f))))
					     (a 1)
					     (r (expt 2 (1- (- k (prec f)))))))
		  :in-theory (enable bits-mod))))

(local-defthmd rdif-29
  (let* ((p (prec f))
         (s (spn f))
  	 (x (* s z))
	 (m (m k z))
	 (sum (sum m k p mode))
	 (sl (sl m k p mode))
	 (sh (sh m k p mode)))
    (implies (and (formatp f)
                  (rationalp z)
		  (< 0 z)
		  (< z 2)
		  (common-mode-p mode)
		  (natp k)
		  (>= k (+ p 2))
		  (natp m)
		  (= sh (expt 2 (1- p)))
		  (or (= mode 'raz) (= mode 'rup))
	          (< x s))
             (equal (mod (+ (mod m (expt 2 (- k p)))
	                    (c k p mode))
		         (expt 2 (- k p)))
	            (bits sum (1- (- k p)) 0))))
  :hints (("Goal" :use ((:instance mod-sum (a (c k (prec f) mode)) (b (m k z)) (n (expt 2 (- k (prec f))))))
		  :in-theory (enable c rnd-const sum bits-mod))))

(local-defthmd rdif-30
  (let* ((p (prec f))
         (s (spn f))
  	 (x (* s z))
	 (m (m k z))
	 (sum (sum m k p mode))
	 (sl (sl m k p mode))
	 (sh (sh m k p mode)))
    (implies (and (formatp f)
                  (rationalp z)
		  (< 0 z)
		  (< z 2)
		  (common-mode-p mode)
		  (natp k)
		  (>= k (+ p 2))
		  (natp m)
		  (= sh (expt 2 (1- p)))
		  (or (= mode 'raz) (= mode 'rup))
	          (< x s)
		  (= (bitn m (1- (- k p))) 0))
             (<  (bits sum (1- (- k p)) 0)
	         (expt 2 (1- (- k p))))))
  :hints (("Goal" :use (rdif-28 rdif-29))))

(local-defthmd rdif-31
  (let* ((p (prec f))
         (s (spn f))
  	 (x (* s z))
	 (m (m k z))
	 (sum (sum m k p mode))
	 (sl (sl m k p mode))
	 (sh (sh m k p mode))
	 (g (bitn sum (1- (- k p)))))
    (implies (and (formatp f)
                  (rationalp z)
		  (< 0 z)
		  (< z 2)
		  (common-mode-p mode)
		  (natp k)
		  (>= k (+ p 2))
		  (natp m)
		  (= sh (expt 2 (1- p)))
		  (or (= mode 'raz) (= mode 'rup))
	          (< x s)
		  (= (bitn m (1- (- k p))) 0))
             (equal g 0)))
  :hints (("Goal" :use (rdif-30
                        (:instance bits-bounds (x (sum (m k z) k (prec f) mode)) (i (- k (+ 2 (prec f)))) (j 0))
                        (:instance bitn-plus-bits (x (sum (m k z) k (prec f) mode)) (n (1- (- k (prec f)))) (m 0))))))

(local-defthmd rdif-32
  (let* ((p (prec f))
         (s (spn f))
  	 (x (* s z))
	 (m (m k z))
	 (sum (sum m k p mode))
	 (sl (sl m k p mode))
	 (sh (sh m k p mode))
	 (g (bitn sum (1- (- k p))))
	 (e (- (expt 2 (1- k)) (expt 2 (1- (- k p))))))
    (implies (and (formatp f)
                  (rationalp z)
		  (< 0 z)
		  (< z 2)
		  (common-mode-p mode)
		  (natp k)
		  (>= k (+ p 2))
		  (natp m)
		  (= sh (expt 2 (1- p)))
		  (or (= mode 'raz) (= mode 'rup))
	          (< x s)
		  (= (bitn m (1- (- k p))) 1))
             (equal m (+ e (bits m (- (- k p) 2) 0)))))
  :hints (("Goal" :nonlinearp t
                  :use (rdif-19
                        (:instance bitn-plus-bits (x (m k z)) (n (1- (- k (prec f)))) (m 0))))))

(local-defthmd rdif-33
  (let* ((p (prec f))
         (s (spn f))
  	 (x (* s z))
	 (m (m k z))
	 (sum (sum m k p mode))
	 (sl (sl m k p mode))
	 (sh (sh m k p mode)))
    (implies (and (formatp f)
                  (rationalp z)
		  (< 0 z)
		  (< z 2)
		  (common-mode-p mode)
		  (natp k)
		  (>= k (+ p 2))
		  (natp m)
		  (= sh (expt 2 (1- p)))
		  (or (= mode 'raz) (= mode 'rup))
	          (< x s)
		  (= (bitn m (1- (- k p))) 1))
             (<  (+ (bits m (1- (- k p)) 0) (c k p mode))
	         (* 2 (expt 2 (- k p))))))
  :hints (("Goal" :in-theory (enable c rnd-const)
                  :nonlinearp t
                  :use ((:instance bits-bounds (x (m k z)) (i (1- (- k (prec f)))) (j 0))))))

(local-defthmd rdif-34
  (let* ((p (prec f))
         (s (spn f))
  	 (x (* s z))
	 (m (m k z))
	 (sum (sum m k p mode))
	 (sl (sl m k p mode))
	 (sh (sh m k p mode)))
    (implies (and (formatp f)
                  (rationalp z)
		  (< 0 z)
		  (< z 2)
		  (common-mode-p mode)
		  (natp k)
		  (>= k (+ p 2))
		  (natp m)
		  (= sh (expt 2 (1- p)))
		  (or (= mode 'raz) (= mode 'rup))
	          (< x s)
		  (= (bitn m (1- (- k p))) 1))
             (equal (mod (+ (mod m (expt 2 (- k p)))
	                    (c k p mode))
		         (expt 2 (- k p)))
	            (1- (+ (expt 2 (1- (- k p))) (bits m (- (- k p)2) 0))))))
  :hints (("Goal" :use (rdif-26 rdif-33
                        (:instance bitn-plus-bits (x (m k z)) (n (1- (- k (prec f)))) (m 0))
                        (:instance mod-force (m (+ (bits (m k z) (1- (- k (prec f))) 0) (c k (prec f) mode)))
                                             (a 1)
					     (n (expt 2 (- k (prec f))))))
		  :in-theory (enable bits-mod c rnd-const))))

(local-defthmd rdif-35
  (let* ((p (prec f))
         (s (spn f))
  	 (x (* s z))
	 (m (m k z))
	 (sum (sum m k p mode))
	 (sl (sl m k p mode))
	 (sh (sh m k p mode)))
    (implies (and (formatp f)
                  (rationalp z)
		  (< 0 z)
		  (< z 2)
		  (common-mode-p mode)
		  (natp k)
		  (>= k (+ p 2))
		  (natp m)
		  (= sh (expt 2 (1- p)))
		  (or (= mode 'raz) (= mode 'rup))
	          (< x s)
		  (= (bitn m (1- (- k p))) 1))
             (equal (bits sum (1- (- k p)) 0)
	            (1- (+ (expt 2 (1- (- k p))) (bits m (- (- k p)2) 0))))))
  :hints (("Goal" :use (rdif-34 rdif-29) :in-theory (theory 'minimal-theory))))

(local-defthmd rdif-36
  (let* ((p (prec f))
         (s (spn f))
  	 (x (* s z))
	 (m (m k z))
	 (sum (sum m k p mode))
	 (sl (sl m k p mode))
	 (sh (sh m k p mode))
	 (g (bitn sum (1- (- k p)))))
    (implies (and (formatp f)
                  (rationalp z)
		  (< 0 z)
		  (< z 2)
		  (common-mode-p mode)
		  (natp k)
		  (>= k (+ p 2))
		  (natp m)
		  (= sh (expt 2 (1- p)))
		  (or (= mode 'raz) (= mode 'rup))
	          (< x s)
		  (= (bitn m (1- (- k p))) 1))
             (iff (= g 1) (> (bits m (- (- k p) 2) 0) 0))))
  :hints (("Goal" :use (rdif-35
                        (:instance bitn-plus-bits (x (sum (m k z) k (prec f) mode)) (n (1- (- k (prec f)))) (m 0))
                        (:instance bits-bounds (x (sum (m k z) k (prec f) mode)) (i (- (- k (prec f)) 2)) (j 0))))))

(local-defthmd rdif-37
  (let* ((p (prec f))
         (s (spn f))
  	 (x (* s z))
	 (m (m k z))
	 (sum (sum m k p mode))
	 (sl (sl m k p mode))
	 (sh (sh m k p mode))
	 (g (bitn sum (1- (- k p))))
	 (e (- (expt 2 (1- k)) (expt 2 (1- (- k p))))))
    (implies (and (formatp f)
                  (rationalp z)
		  (< 0 z)
		  (< z 2)
		  (common-mode-p mode)
		  (natp k)
		  (>= k (+ p 2))
		  (natp m)
		  (= sh (expt 2 (1- p)))
		  (or (= mode 'raz) (= mode 'rup))
	          (< x s))
             (iff (= g 1) (>= (raz m p) (expt 2 (1- k))))))
  :hints (("Goal" :use (rdif-23 rdif-25 rdif-31 rdif-36 rdif-32))))

(local-defthmd rdif-38
  (let* ((p (prec f))
         (s (spn f))
  	 (x (* s z))
	 (m (m k z))
	 (sum (sum m k p mode))
	 (sl (sl m k p mode))
	 (sh (sh m k p mode))
	 (g (bitn sum (1- (- k p))))
	 (e (- (expt 2 (1- k)) (expt 2 (1- (- k p))))))
    (implies (and (formatp f)
                  (rationalp z)
		  (< 0 z)
		  (< z 2)
		  (common-mode-p mode)
		  (natp k)
		  (>= k (+ p 2))
		  (natp m)
		  (= sh (expt 2 (1- p)))
		  (or (= mode 'raz) (= mode 'rup))
	          (< x s))
             (iff (= g 1) (>= (rnd m mode p) (expt 2 (1- k))))))
  :hints (("Goal" :use (rdif-37) :in-theory (enable rnd rup))))

(local-defthmd rdif-39
  (let* ((p (prec f))
         (s (spn f))
  	 (x (* s z))
	 (m (m k z))
	 (sum (sum m k p mode))
	 (sl (sl m k p mode))
	 (sh (sh m k p mode))
	 (g (bitn sum (1- (- k p))))
	 (e (- (expt 2 (1- k)) (expt 2 (1- (- k p))))))
    (implies (and (formatp f)
                  (rationalp z)
		  (< 0 z)
		  (< z 2)
		  (common-mode-p mode)
		  (natp k)
		  (>= k (+ p 2))
		  (natp m)
		  (= sh (expt 2 (1- p)))
		  (or (= mode 'raz) (= mode 'rup))
	          (< x s))
             (iff (= g 0) (< (rnd x mode p) s))))
  :hints (("Goal" :use (rdif-38 rdif-24))))

(local-defthmd rdif-40
  (let* ((p (prec f))
         (s (spn f))
  	 (x (* s z)))
    (implies (and (formatp f)
                  (rationalp z)
		  (common-mode-p mode)
		  (>= x s))
	    (>= (rnd x mode p) s)))
  :hints (("Goal" :use ((:instance rnd-exactp-d (a (spn f)) (x (* (spn f) z)) (n (prec f))))
                  :in-theory (enable spn exactp-2**n))))

(defthmd rnd-drnd-inject-f-ii
  (let* ((p (prec f))
         (s (spn f))
  	 (x (* s z))
	 (m (* (expt 2 (1- k)) z))
	 (c (rnd-const (1- k) mode p))
	 (sum (+ m c))
	 (sh (bits sum k (- k p)))
	 (g (bitn sum (1- (- k p)))))
    (implies (and (formatp f)
                  (rationalp z)
		  (< 0 z)
		  (< z 2)
		  (common-mode-p mode)
		  (natp k)
		  (>= k (+ p 2))
		  (natp m)
		  (= sh (expt 2 (1- p)))
		  (or (= mode 'raz) (= mode 'rup)))
	    (iff (< (rnd x mode p) s)
	         (= g 0))))
  :hints (("Goal" :use (rdif-17 rdif-39 rdif-40)
                  :in-theory (enable m c sum sh sl))))

;;---------------------------------------------------------------------------------------------------------

(local-defthmd rdig-1
  (let* ((p (prec f))
         (s (spn f))
  	 (x (* s z))
	 (m (m k z))
	 (sum (sum m k p mode))
	 (sl (sl m k p mode))
	 (sh (sh m k p mode))
	 (g (bitn sum (1- (- k p)))))
    (implies (and (formatp f)
                  (rationalp z)
		  (< 0 z)
		  (< z 2)
		  (common-mode-p mode)
		  (natp k)
		  (>= k (+ p 2))
		  (natp m)
		  (= sh (expt 2 (1- p)))
		  (or (= mode 'rne) (= mode 'rna)))
	    (and (iff (< x s) (= (bitn m (1- (- k p))) 1))
	         (iff (< x s) (>= (+ (bits m (1- (- k p)) 0) (c k p mode)) (expt 2 (- k p))))
	         (iff (< x s) (= g 0)))))
  :hints (("Goal" :use (rdif-26
                        (:instance bitn-0-1 (x (sum (m k z) k (prec f) mode)) (n (1- (- k (prec f)))))
                        (:instance bitn-0-1 (x (m k z)) (n (1- (- k (prec f)))))
                        (:instance bitn-plus-bits (x (m k z)) (n (1- (- k (prec f)))) (m 0))
			(:instance bits-bounds (x (m k z)) (i (- (- k (prec f)) 2)) (j 0))
			(:instance bitn-plus-expt-1 (x (m k z)) (n (1- (- k (prec f))))))
	          :nonlinearp t
                  :in-theory (enable c rnd-const sum))))

(defthmd rnd-drnd-inject-g-i
  (let* ((p (prec f))
         (s (spn f))
  	 (x (* s z))
	 (m (* (expt 2 (1- k)) z))
	 (c (rnd-const (1- k) mode p))
	 (sum (+ m c))
	 (sh (bits sum k (- k p)))
	 (g (bitn sum (1- (- k p)))))
    (implies (and (formatp f)
                  (rationalp z)
		  (< 0 z)
		  (< z 2)
		  (common-mode-p mode)
		  (natp k)
		  (>= k (+ p 2))
		  (natp m)
		  (= sh (expt 2 (1- p)))
		  (or (= mode 'rne) (= mode 'rna)))
	    (iff (< x s)
                 (= g 0))))
  :hints (("Goal" :use (rdig-1)
                  :in-theory (enable m c sum sh sl))))

;;---------------------------------------------------------------------------------------------------------

(local-defthmd rdig-2
  (let* ((p (prec f))
         (s (spn f))
  	 (x (* s z))
	 (m (m k z))
	 (sum (sum m k p mode))
	 (sl (sl m k p mode))
	 (sh (sh m k p mode))
	 (g (bitn sum (1- (- k p)))))
    (implies (and (formatp f)
                  (rationalp z)
		  (< 0 z)
		  (< z 2)
		  (common-mode-p mode)
		  (natp k)
		  (>= k (+ p 2))
		  (natp m)
		  (= sh (expt 2 (1- p)))
		  (or (= mode 'rne) (= mode 'rna))
	          (>= x s))
	    (and (not (= g 0)) (>= (rnd x mode p) s))))
  :hints (("Goal" :use (rdig-1 rdif-40))))

(local-defthmd rdig-3
  (let* ((p (prec f))
         (s (spn f))
  	 (x (* s z))
	 (m (m k z))
	 (sum (sum m k p mode))
	 (sl (sl m k p mode))
	 (sh (sh m k p mode))
	 (g (bitn sum (1- (- k p)))))
    (implies (and (formatp f)
                  (rationalp z)
		  (< 0 z)
		  (< z 2)
		  (common-mode-p mode)
		  (natp k)
		  (>= k (+ p 2))
		  (natp m)
		  (= sh (expt 2 (1- p)))
		  (or (= mode 'rne) (= mode 'rna))
	          (< x s)
		  (= g 0))
	    (< m (expt 2 (1- k)))))
  :hints (("Goal" :use (rdig-1 rdif-11 rdif-8))))

(local-defthmd rdig-4
  (let* ((p (prec f))
         (s (spn f))
  	 (x (* s z))
	 (m (m k z))
	 (sum (sum m k p mode))
	 (sl (sl m k p mode))
	 (sh (sh m k p mode))
	 (g (bitn sum (1- (- k p)))))
    (implies (and (formatp f)
                  (rationalp z)
		  (< 0 z)
		  (< z 2)
		  (common-mode-p mode)
		  (natp k)
		  (>= k (+ p 2))
		  (natp m)
		  (= sh (expt 2 (1- p)))
		  (or (= mode 'rne) (= mode 'rna))
		  (< x s)
		  (= g 0))
	    (equal (bits m (1- k) (- k p))
	           (1- (expt 2 (1- p))))))
  :hints (("Goal" :use (rdif-14 rdig-3 rdif-11 rdif-8))))

(local-defthmd rdig-5
  (let* ((p (prec f))
         (s (spn f))
  	 (x (* s z))
	 (m (m k z))
	 (sum (sum m k p mode))
	 (sl (sl m k p mode))
	 (sh (sh m k p mode))
	 (g (bitn sum (1- (- k p))))
	 (r (bitn sum (- (- k p) 2)))
	 (e (- (expt 2 (1- k)) (expt 2 (1- (- k p))))))
    (implies (and (formatp f)
                  (rationalp z)
		  (< 0 z)
		  (< z 2)
		  (natp k)
		  (>= k (+ p 2))
		  (or (= mode 'rne) (= mode 'rna)))
	    (equal r (bitn m (- (- k p) 2)))))
  :hints (("Goal" :use ((:instance bitn-plus-mult (x (m k z)) (k 1) (m (1- (- k (prec f)))) (n (- (- k (prec f)) 2))))
                  :in-theory (enable sum c rnd-const))))

(local-defthmd rdig-6
  (let* ((p (prec f))
         (s (spn f))
  	 (x (* s z))
	 (m (m k z))
	 (sum (sum m k p mode))
	 (sl (sl m k p mode))
	 (sh (sh m k p mode))
	 (g (bitn sum (1- (- k p))))
	 (r (bitn sum (- (- k p) 2)))
	 (e (- (expt 2 (1- k)) (expt 2 (1- (- k p))))))
    (implies (and (formatp f)
                  (rationalp z)
		  (< 0 z)
		  (< z 2)
		  (common-mode-p mode)
		  (natp k)
		  (>= k (+ p 2))
		  (natp m)
		  (= sh (expt 2 (1- p)))
		  (or (= mode 'rne) (= mode 'rna))
		  (< x s)
		  (= g 0))
	    (equal m (+ (- (expt 2 (1- k))
	                   (- (expt 2 (- k p)) (expt 2 (1- (- k p)))))
			(* (expt 2 (- (- k p) 2)) r)
			(bits m (- (- k p) 3) 0)))))
  :hints (("Goal" :use (rdig-1 rdig-4 rdig-5
                        (:instance bits-plus-bits (x (m k z)) (n (1- k)) (p (- k (prec f))) (m 0))
			(:instance bitn-plus-bits (x (m k z)) (n (1- (- k (prec f)))) (m 0))
		        (:instance bitn-plus-bits (x (m k z)) (n (- (- k (prec f)) 2)) (m 0))))))

(local-defthmd rdig-7
  (let* ((p (prec f))
         (s (spn f))
  	 (x (* s z))
	 (m (m k z))
	 (sum (sum m k p mode))
	 (sl (sl m k p mode))
	 (sh (sh m k p mode))
	 (g (bitn sum (1- (- k p))))
	 (r (bitn sum (- (- k p) 2)))
	 (e (- (expt 2 (1- k)) (expt 2 (1- (- k p))))))
    (implies (and (formatp f)
                  (rationalp z)
		  (< 0 z)
		  (< z 2)
		  (common-mode-p mode)
		  (natp k)
		  (>= k (+ p 2))
		  (natp m)
		  (= sh (expt 2 (1- p)))
		  (or (= mode 'rne) (= mode 'rna))
		  (< x s)
		  (= g 0))
	    (equal (- (expt 2 (- k p)) (expt 2 (1- (- k p))))
	           (expt 2 (1- (- k p))))))
  :hints (("Goal" :nonlinearp t)))

(local-defthmd rdig-8
  (let* ((p (prec f))
         (s (spn f))
  	 (x (* s z))
	 (m (m k z))
	 (sum (sum m k p mode))
	 (sl (sl m k p mode))
	 (sh (sh m k p mode))
	 (g (bitn sum (1- (- k p))))
	 (r (bitn sum (- (- k p) 2)))
	 (e (- (expt 2 (1- k)) (expt 2 (1- (- k p))))))
    (implies (and (formatp f)
                  (rationalp z)
		  (< 0 z)
		  (< z 2)
		  (common-mode-p mode)
		  (natp k)
		  (>= k (+ p 2))
		  (natp m)
		  (= sh (expt 2 (1- p)))
		  (or (= mode 'rne) (= mode 'rna))
		  (< x s)
		  (= g 0))
	    (equal m (+ e
			(* (expt 2 (- (- k p) 2)) r)
			(bits m (- (- k p) 3) 0)))))
  :hints (("Goal" :use (rdig-6 rdig-7) :in-theory (theory 'minimal-theory))))

(local-defthmd rdig-9
  (let* ((p (prec f))
         (s (spn f))
  	 (x (* s z))
	 (m (m k z))
	 (sum (sum m k p mode))
	 (sl (sl m k p mode))
	 (sh (sh m k p mode))
	 (g (bitn sum (1- (- k p))))
	 (r (bitn sum (- (- k p) 2)))
	 (e (- (expt 2 (1- k)) (expt 2 (1- (- k p))))))
    (implies (and (formatp f)
                  (rationalp z)
		  (< 0 z)
		  (< z 2)
		  (common-mode-p mode)
		  (natp k)
		  (>= k (+ p 2))
		  (natp m)
		  (= sh (expt 2 (1- p)))
		  (or (= mode 'rne) (= mode 'rna))
		  (< x s)
		  (= g 0))
	    (>= m (expt 2 (- k 2)))))
  :hints (("Goal" :nonlinearp t :use (rdig-6))))

(local-defthmd rdig-10
  (let* ((p (prec f))
         (s (spn f))
  	 (x (* s z))
	 (m (m k z))
	 (sum (sum m k p mode))
	 (sl (sl m k p mode))
	 (sh (sh m k p mode))
	 (g (bitn sum (1- (- k p)))))
    (implies (and (formatp f)
                  (rationalp z)
		  (< 0 z)
		  (< z 2)
		  (common-mode-p mode)
		  (natp k)
		  (>= k (+ p 2))
		  (natp m)
		  (= sh (expt 2 (1- p)))
		  (or (= mode 'rne) (= mode 'rna))
	          (< x s)
		  (= g 0))
	    (< m (expt 2 (1- k)))))
  :hints (("Goal" :use (rdig-1 rdif-11 rdif-8))))

(local-defthmd rdig-11
  (let* ((p (prec f))
         (s (spn f))
  	 (x (* s z))
	 (m (m k z))
	 (sum (sum m k p mode))
	 (sl (sl m k p mode))
	 (sh (sh m k p mode))
	 (g (bitn sum (1- (- k p))))
	 (r (bitn sum (- (- k p) 2)))
	 (e (- (expt 2 (1- k)) (expt 2 (1- (- k p))))))
    (implies (and (formatp f)
                  (rationalp z)
		  (< 0 z)
		  (< z 2)
		  (common-mode-p mode)
		  (natp k)
		  (>= k (+ p 2))
		  (natp m)
		  (= sh (expt 2 (1- p)))
		  (or (= mode 'rne) (= mode 'rna))
		  (< x s)
		  (= g 0))
	    (equal (expo m) (- k 2))))
  :hints (("Goal" :use (rdig-10 rdig-9
                        (:instance expo<= (x (m k z)) (n (- k 2)))
                        (:instance expo>= (x (m k z)) (n (- k 2)))))))

(local-defthmd rdig-12
  (let* ((p (prec f))
         (s (spn f))
  	 (x (* s z))
	 (m (m k z))
	 (sum (sum m k p mode))
	 (sl (sl m k p mode))
	 (sh (sh m k p mode))
	 (g (bitn sum (1- (- k p))))
	 (r (bitn sum (- (- k p) 2)))
	 (e (- (expt 2 (1- k)) (expt 2 (1- (- k p))))))
    (implies (and (formatp f)
                  (rationalp z)
		  (< 0 z)
		  (< z 2)
		  (common-mode-p mode)
		  (natp k)
		  (>= k (+ p 2))
		  (natp m)
		  (= sh (expt 2 (1- p)))
		  (or (= mode 'rne) (= mode 'rna))
		  (< x s)
		  (= g 0))
	    (iff (= r 0)
	         (< m (+ e (expt 2 (- (expo m) p)))))))
  :hints (("Goal" :use (rdig-8 rdig-11
                        (:instance bits-bounds (x (m k z)) (i (- (- k (prec f)) 3)) (j 0))
			(:instance bitn-0-1 (x (m k z)) (n (- (- k (prec f)) 2)))))))

(local-defthmd rdig-13
  (let* ((p (prec f))
         (s (spn f))
  	 (x (* s z))
	 (m (m k z))
	 (sum (sum m k p mode))
	 (sl (sl m k p mode))
	 (sh (sh m k p mode))
	 (g (bitn sum (1- (- k p))))
	 (r (bitn sum (- (- k p) 2)))
	 (e (- (expt 2 (1- k)) (expt 2 (1- (- k p))))))
    (implies (and (formatp f)
                  (rationalp z)
		  (< 0 z)
		  (< z 2)
		  (common-mode-p mode)
		  (natp k)
		  (>= k (+ p 2))
		  (natp m)
		  (= sh (expt 2 (1- p)))
		  (or (= mode 'rne) (= mode 'rna))
		  (< x s)
		  (= g 0)
		  (= r 0))
	    (>= m e)))
  :hints (("Goal" :use (rdig-8) :nonlinearp t)))

(local-defthmd rdig-14
  (let* ((p (prec f))
         (s (spn f))
  	 (x (* s z))
	 (m (m k z))
	 (sum (sum m k p mode))
	 (sl (sl m k p mode))
	 (sh (sh m k p mode))
	 (g (bitn sum (1- (- k p))))
	 (r (bitn sum (- (- k p) 2)))
	 (e (- (expt 2 (1- k)) (expt 2 (1- (- k p))))))
    (implies (and (formatp f)
                  (rationalp z)
		  (< 0 z)
		  (< z 2)
		  (common-mode-p mode)
		  (natp k)
		  (>= k (+ p 2))
		  (natp m)
		  (= sh (expt 2 (1- p)))
		  (or (= mode 'rne) (= mode 'rna))
		  (< x s)
		  (= g 0)
		  (= r 0))
	    (< (abs (- m e)) (expt 2 (- (expo m) p)))))
  :hints (("Goal" :use (rdig-13 rdig-12))))

(local-defthmd rdig-15
  (let* ((p (prec f))
         (s (spn f))
  	 (x (* s z))
	 (m (m k z))
	 (sum (sum m k p mode))
	 (sl (sl m k p mode))
	 (sh (sh m k p mode))
	 (g (bitn sum (1- (- k p))))
	 (r (bitn sum (- (- k p) 2)))
	 (e (- (expt 2 (1- k)) (expt 2 (1- (- k p))))))
    (implies (and (formatp f)
                  (rationalp z)
		  (< 0 z)
		  (< z 2)
		  (common-mode-p mode)
		  (natp k)
		  (>= k (+ p 2))
		  (natp m)
		  (= sh (expt 2 (1- p)))
		  (or (= mode 'rne) (= mode 'rna)))
	    (> e 0)))
  :hints (("Goal" :nonlinearp t)))

(local-defthmd rdig-16
  (let* ((p (prec f))
         (s (spn f))
  	 (x (* s z))
	 (m (m k z))
	 (sum (sum m k p mode))
	 (sl (sl m k p mode))
	 (sh (sh m k p mode))
	 (g (bitn sum (1- (- k p))))
	 (r (bitn sum (- (- k p) 2)))
	 (e (- (expt 2 (1- k)) (expt 2 (1- (- k p))))))
    (implies (and (formatp f)
                  (rationalp z)
		  (< 0 z)
		  (< z 2)
		  (common-mode-p mode)
		  (natp k)
		  (>= k (+ p 2))
		  (natp m)
		  (= sh (expt 2 (1- p)))
		  (or (= mode 'rne) (= mode 'rna))
		  (< x s)
		  (= g 0)
		  (= r 0))
	    (equal (rnd m mode p) e)))
  :hints (("Goal" :use (rdif-20 rdif-21 rdig-14 rdig-13 rdig-15
                        (:instance rne-force (n (prec f)) (x (m k z)) (y (- (expt 2 (1- k)) (expt 2 (1- (- k (prec f)))))))
                        (:instance rna-force (n (prec f)) (x (m k z)) (y (- (expt 2 (1- k)) (expt 2 (1- (- k (prec f))))))))
		  :in-theory (enable rnd))))

(local-defthmd rdig-20
  (let* ((p (prec f))
         (s (spn f))
  	 (x (* s z))
	 (m (m k z))
	 (sum (sum m k p mode))
	 (sl (sl m k p mode))
	 (sh (sh m k p mode))
	 (g (bitn sum (1- (- k p))))
	 (r (bitn sum (- (- k p) 2)))
	 (e (- (expt 2 (1- k)) (expt 2 (1- (- k p))))))
    (implies (and (formatp f)
                  (rationalp z)
		  (< 0 z)
		  (< z 2)
		  (common-mode-p mode)
		  (natp k)
		  (>= k (+ p 2))
		  (natp m)
		  (= sh (expt 2 (1- p)))
		  (or (= mode 'rne) (= mode 'rna))
		  (< x s)
		  (= g 0)
		  (= r 0))
	    (< (rnd m mode p) (expt 2 (1- k)))))
  :hints (("Goal" :use (rdig-16)
		  :in-theory (enable rnd))))

(local-defthmd rdig-21
  (let* ((p (prec f))
         (s (spn f))
  	 (x (* s z))
	 (m (m k z))
	 (sum (sum m k p mode))
	 (sl (sl m k p mode))
	 (sh (sh m k p mode))
	 (g (bitn sum (1- (- k p))))
	 (r (bitn sum (- (- k p) 2)))
	 (e (- (expt 2 (1- k)) (expt 2 (1- (- k p))))))
    (implies (and (formatp f)
                  (rationalp z)
		  (< 0 z)
		  (< z 2)
		  (common-mode-p mode)
		  (natp k)
		  (>= k (+ p 2))
		  (natp m)
		  (= sh (expt 2 (1- p)))
		  (or (= mode 'rne) (= mode 'rna))
		  (< x s)
		  (= g 0)
		  (= r 1))
	     (>= m (+ e (expt 2 (- (- k p) 2))))))
  :hints (("Goal" :use (rdig-12 rdig-11))))

(local-defthmd rdig-22
  (let* ((p (prec f))
         (s (spn f))
  	 (x (* s z))
	 (m (m k z))
	 (sum (sum m k p mode))
	 (sl (sl m k p mode))
	 (sh (sh m k p mode))
	 (g (bitn sum (1- (- k p))))
	 (r (bitn sum (- (- k p) 2)))
	 (e (- (expt 2 (1- k)) (expt 2 (1- (- k p))))))
    (implies (and (formatp f)
                  (rationalp z)
		  (< 0 z)
		  (< z 2)
		  (common-mode-p mode)
		  (natp k)
		  (>= k (+ p 2))
		  (natp m)
		  (= sh (expt 2 (1- p)))
		  (or (= mode 'rne) (= mode 'rna))
		  (< x s)
		  (= g 0)
		  (= r 1))
	     (>= m (- (expt 2 (1- k)) (expt 2 (- (- k p) 2))))))
  :hints (("Goal" :use (rdig-21) :nonlinearp t)))

(local-defthmd rdig-23
  (let* ((p (prec f))
         (s (spn f))
  	 (x (* s z))
	 (m (m k z))
	 (sum (sum m k p mode))
	 (sl (sl m k p mode))
	 (sh (sh m k p mode))
	 (g (bitn sum (1- (- k p))))
	 (r (bitn sum (- (- k p) 2)))
	 (e (- (expt 2 (1- k)) (expt 2 (1- (- k p))))))
    (implies (and (formatp f)
                  (rationalp z)
		  (< 0 z)
		  (< z 2)
		  (common-mode-p mode)
		  (natp k)
		  (>= k (+ p 2))
		  (natp m)
		  (= sh (expt 2 (1- p)))
		  (or (= mode 'rne) (= mode 'rna))
		  (< x s)
		  (= g 0)
		  (= r 1))
	     (>= m (- (expt 2 (1- k)) (expt 2 (- (expo m) p))))))
  :hints (("Goal" :use (rdig-22 rdig-11))))

(local-defthmd rdig-24
  (let* ((p (prec f))
         (s (spn f))
  	 (x (* s z))
	 (m (m k z))
	 (sum (sum m k p mode))
	 (sl (sl m k p mode))
	 (sh (sh m k p mode))
	 (g (bitn sum (1- (- k p))))
	 (r (bitn sum (- (- k p) 2)))
	 (e (- (expt 2 (1- k)) (expt 2 (1- (- k p))))))
    (implies (and (formatp f)
                  (rationalp z)
		  (< 0 z)
		  (< z 2)
		  (common-mode-p mode)
		  (natp k)
		  (>= k (+ p 2))
		  (natp m)
		  (= sh (expt 2 (1- p)))
		  (or (= mode 'rne) (= mode 'rna))
		  (< x s)
		  (= g 0)
		  (= r 1))
	     (equal (rnd m mode p)
	            (expt 2 (1- k)))))
  :hints (("Goal" :use (rdig-11 rdig-23
                        (:instance rne-pow-2 (x (m k z)) (n (prec f)))
                        (:instance rna-pow-2 (x (m k z)) (n (prec f))))
		  :in-theory (enable rnd))))

(local-defthmd rdig-25
  (let* ((p (prec f))
         (s (spn f))
  	 (x (* s z))
	 (m (m k z))
	 (sum (sum m k p mode))
	 (sl (sl m k p mode))
	 (sh (sh m k p mode))
	 (g (bitn sum (1- (- k p))))
	 (r (bitn sum (- (- k p) 2)))
	 (e (- (expt 2 (1- k)) (expt 2 (1- (- k p))))))
    (implies (and (formatp f)
                  (rationalp z)
		  (< 0 z)
		  (< z 2)
		  (common-mode-p mode)
		  (natp k)
		  (>= k (+ p 2))
		  (natp m)
		  (= sh (expt 2 (1- p)))
		  (or (= mode 'rne) (= mode 'rna))
		  (< x s)
		  (= g 0))
	    (iff (= r 0)
	         (< (rnd x mode p) s))))
  :hints (("Goal" :use (rdif-24 rdig-20 rdig-24))))

(local-defthmd rdig-26
  (let* ((p (prec f))
         (s (spn f))
  	 (x (* s z))
	 (m (m k z))
	 (sum (sum m k p mode))
	 (sl (sl m k p mode))
	 (sh (sh m k p mode))
	 (g (bitn sum (1- (- k p))))
	 (r (bitn sum (- (- k p) 2)))
	 (e (- (expt 2 (1- k)) (expt 2 (1- (- k p))))))
    (implies (and (formatp f)
                  (rationalp z)
		  (< 0 z)
		  (< z 2)
		  (common-mode-p mode)
		  (natp k)
		  (>= k (+ p 2))
		  (natp m)
		  (= sh (expt 2 (1- p)))
		  (or (= mode 'rne) (= mode 'rna)))
	    (iff (< (rnd x mode p) s)
	         (and (= g 0) (= r 0)))))
  :hints (("Goal" :use (rdig-25 rdig-1 rdig-2))))

(defthmd rnd-drnd-inject-g-ii
  (let* ((p (prec f))
         (s (spn f))
  	 (x (* s z))
	 (m (* (expt 2 (1- k)) z))
	 (c (rnd-const (1- k) mode p))
	 (sum (+ m c))
	 (sh (bits sum k (- k p)))
	 (g (bitn sum (1- (- k p))))
	 (r (bitn sum (- (- k p) 2))))
    (implies (and (formatp f)
                  (rationalp z)
		  (< 0 z)
		  (< z 2)
		  (common-mode-p mode)
		  (natp k)
		  (>= k (+ p 2))
		  (natp m)
		  (= sh (expt 2 (1- p)))
		  (or (= mode 'rne) (= mode 'rna)))
	    (iff (< (rnd x mode p) s)
	         (and (= g 0) (= r 0)))))
  :hints (("Goal" :use (rdig-26)
                  :in-theory (enable m c sum sh))))

;;---------------------------------------------------------------------------------------------------------

(defthmd rnd-drnd-inject-a
  (let* ((p (prec f))
         (s (spn f))
  	 (x (* s z))
	 (m (* (expt 2 (1- k)) z))
	 (sum (+ m (rnd-const (1- k) mode p)))
	 (sl (bits sum (1- (- k p)) 0))
	 (sh (bits sum k (- k p))))
    (implies (and (formatp f)
                  (rationalp z)
		  (< 0 z)
		  (< z 2)
		  (common-mode-p mode)
		  (natp k)
		  (>= k (+ p 2))
		  (natp m)
		  (< (expo m) (- k p)))
	    (equal (drnd x mode f)
	           (if (and (= mode 'rne) (= sl 0))
		       (* (expt 2 (- 2 p)) s (bits sh p 1))
		     (* (expt 2 (- 1 p)) s sh))))))

(defthmd rnd-drnd-inject-b
  (let* ((p (prec f))
         (s (spn f))
  	 (x (* s z))
	 (m (* (expt 2 (1- k)) z))
	 (c (rnd-const (1- k) mode p))
	 (sum (+ m c))
	 (sl (bits sum (1- (- k p)) 0)))
    (implies (and (formatp f)
                  (rationalp z)
		  (< 0 z)
		  (< z 2)
		  (common-mode-p mode)
		  (natp k)
		  (>= k (+ p 2))
		  (natp m))
	    (iff (= x (drnd x mode f))
	         (= sl c)))))

(defthmd rnd-drnd-inject-c
  (let* ((p (prec f))
         (s (spn f))
  	 (x (* s z))
	 (m (* (expt 2 (1- k)) z))
	 (c (rnd-const (1- k) mode p))
	 (sum (+ m c))
	 (sh (bits sum k (- k p))))
    (implies (and (formatp f)
                  (rationalp z)
		  (< 0 z)
		  (< z 2)
		  (common-mode-p mode)
		  (natp k)
		  (>= k (+ p 2))
		  (natp m)
		  (> sh (expt 2 (1- p))))
	    (and (>= x s)
	         (>= (rnd x mode p) s)))))

(defthmd rnd-drnd-inject-d
  (let* ((p (prec f))
         (s (spn f))
  	 (x (* s z))
	 (m (* (expt 2 (1- k)) z))
	 (c (rnd-const (1- k) mode p))
	 (sum (+ m c))
	 (sh (bits sum k (- k p))))
    (implies (and (formatp f)
                  (rationalp z)
		  (< 0 z)
		  (< z 2)
		  (common-mode-p mode)
		  (natp k)
		  (>= k (+ p 2))
		  (natp m)
		  (< sh (expt 2 (1- p))))
	    (and (< x s)
	         (< (rnd x mode p) s)))))

(defthmd rnd-drnd-inject-e
  (let* ((p (prec f))
         (s (spn f))
  	 (x (* s z))
	 (m (* (expt 2 (1- k)) z))
	 (c (rnd-const (1- k) mode p))
	 (sum (+ m c))
	 (sh (bits sum k (- k p))))
    (implies (and (formatp f)
                  (rationalp z)
		  (< 0 z)
		  (< z 2)
		  (common-mode-p mode)
		  (natp k)
		  (>= k (+ p 2))
		  (natp m)
		  (= sh (expt 2 (1- p)))
		  (or (= mode 'rtz) (= mode 'rdn)))
	    (and (>= x s)
	         (>= (rnd x mode p) s)))))

(defthmd rnd-drnd-inject-f-i
  (let* ((p (prec f))
         (s (spn f))
  	 (x (* s z))
	 (m (* (expt 2 (1- k)) z))
	 (c (rnd-const (1- k) mode p))
	 (sum (+ m c))
	 (sh (bits sum k (- k p)))
	 (sl (bits sum (1- (- k p)) 0)))
    (implies (and (formatp f)
                  (rationalp z)
		  (< 0 z)
		  (< z 2)
		  (common-mode-p mode)
		  (natp k)
		  (>= k (+ p 2))
		  (natp m)
		  (= sh (expt 2 (1- p)))
		  (or (= mode 'raz) (= mode 'rup)))
	    (iff (< x s)
	         (not (= sl c))))))

(defthmd rnd-drnd-inject-f-ii
  (let* ((p (prec f))
         (s (spn f))
  	 (x (* s z))
	 (m (* (expt 2 (1- k)) z))
	 (c (rnd-const (1- k) mode p))
	 (sum (+ m c))
	 (sh (bits sum k (- k p)))
	 (g (bitn sum (1- (- k p)))))
    (implies (and (formatp f)
                  (rationalp z)
		  (< 0 z)
		  (< z 2)
		  (common-mode-p mode)
		  (natp k)
		  (>= k (+ p 2))
		  (natp m)
		  (= sh (expt 2 (1- p)))
		  (or (= mode 'raz) (= mode 'rup)))
	    (iff (< (rnd x mode p) s)
	         (= g 0)))))

(defthmd rnd-drnd-inject-g-i
  (let* ((p (prec f))
         (s (spn f))
  	 (x (* s z))
	 (m (* (expt 2 (1- k)) z))
	 (c (rnd-const (1- k) mode p))
	 (sum (+ m c))
	 (sh (bits sum k (- k p)))
	 (g (bitn sum (1- (- k p)))))
    (implies (and (formatp f)
                  (rationalp z)
		  (< 0 z)
		  (< z 2)
		  (common-mode-p mode)
		  (natp k)
		  (>= k (+ p 2))
		  (natp m)
		  (= sh (expt 2 (1- p)))
		  (or (= mode 'rne) (= mode 'rna)))
	    (iff (< x s)
                 (= g 0)))))

(defthmd rnd-drnd-inject-g-ii
  (let* ((p (prec f))
         (s (spn f))
  	 (x (* s z))
	 (m (* (expt 2 (1- k)) z))
	 (c (rnd-const (1- k) mode p))
	 (sum (+ m c))
	 (sh (bits sum k (- k p)))
	 (g (bitn sum (1- (- k p))))
	 (r (bitn sum (- (- k p) 2))))
    (implies (and (formatp f)
                  (rationalp z)
		  (< 0 z)
		  (< z 2)
		  (common-mode-p mode)
		  (natp k)
		  (>= k (+ p 2))
		  (natp m)
		  (= sh (expt 2 (1- p)))
		  (or (= mode 'rne) (= mode 'rna)))
	    (iff (< (rnd x mode p) s)
	         (and (= g 0) (= r 0))))))
