/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | Copyright (C) 1991-2010 OpenCFD Ltd.
     \\/     M anipulation  |
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

Class
    Foam::fileFormats::AC3DsurfaceFormat

Description
    Provide a means of reading/writing AC3D format.

    http://www.inivis.com/ac3d/man/ac3dfileformat.html

Note
    On input, the faces are already organized as zones.
    The output is always sorted by zones.
    In the absence of zones, a single zone will be assigned.

SourceFiles
    AC3DsurfaceFormat.C

\*---------------------------------------------------------------------------*/

#ifndef AC3DsurfaceFormat_H
#define AC3DsurfaceFormat_H

#include <surfMesh/MeshedSurface.H>
#include <surfMesh/MeshedSurfaceProxy.H>
#include <surfMesh/UnsortedMeshedSurface.H>
#include "AC3DsurfaceFormatCore.H"

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

namespace Foam
{
namespace fileFormats
{

/*---------------------------------------------------------------------------*\
                      Class AC3DsurfaceFormat Declaration
\*---------------------------------------------------------------------------*/

template<class Face>
class AC3DsurfaceFormat
:
    public MeshedSurface<Face>,
    public AC3DsurfaceFormatCore
{
    // Private Member Functions

        //- Disallow default bitwise copy construct
        AC3DsurfaceFormat(const AC3DsurfaceFormat<Face>&);

        //- Disallow default bitwise assignment
        void operator=(const AC3DsurfaceFormat<Face>&);


public:

    // Constructors

        //- Construct from file name
        AC3DsurfaceFormat(const fileName&);


    // Selectors

        //- Read file and return surface
        static autoPtr<MeshedSurface<Face> > New(const fileName& name)
        {
            return autoPtr<MeshedSurface<Face> >
            (
                new AC3DsurfaceFormat<Face>(name)
            );
        }


    // Destructor

        virtual ~AC3DsurfaceFormat()
        {}


    // Member Functions

        //- Write surface mesh components by proxy
        static void write(const fileName&, const MeshedSurfaceProxy<Face>&);

        //- Write UnsortedMeshedSurface, the output is always sorted by zones.
        static void write(const fileName&, const UnsortedMeshedSurface<Face>&);

        //- Read from file
        virtual bool read(const fileName&);

        //- Write object
        virtual void write(const fileName& name) const
        {
            write(name, MeshedSurfaceProxy<Face>(*this));
        }
};


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

} // End namespace fileFormats
} // End namespace Foam

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#ifdef NoRepository
#   include "AC3DsurfaceFormat.C"
#endif

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#endif

// ************************ vim: set sw=4 sts=4 et: ************************ //
