/*
   Copyright (C) 1998-2004 Scott Dattalo

This file is part of the libgpsim library of gpsim

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, see
<http://www.gnu.org/licenses/lgpl-2.1.html>.
*/

#include "ValueCollections.h"
#include "symbol.h"
#include "registers.h"

#include <string.h>
#include <sstream>

IIndexedCollection::IIndexedCollection(const char *pName,
                                       const char *pDesc,
                                       int iAddressRadix)
  : Value(pName, pDesc)
{
  SetAddressRadix(iAddressRadix);
}


IIndexedCollection::IIndexedCollection(int iAddressRadix)
{
  SetAddressRadix(iAddressRadix);
}


void IIndexedCollection::SetAddressRadix(int iRadix)
{
  m_iAddressRadix = iRadix;

  if (iRadix == 16) {
    strcpy(m_szPrefix, "$");

  } else {
    m_szPrefix[0] = 0;
  }
}


void IIndexedCollection::Set(Value * pValue)
{
  unsigned int  uUpper = GetUpperBound() + 1;

  for (unsigned int uIndex = GetLowerBound(); uIndex < uUpper; uIndex++) {
    SetAt(uIndex, pValue);
  }
}


void IIndexedCollection::SetAt(ExprList_t* pIndexers, Expression *pExpr)
{
  ExprList_t::iterator it;
  ExprList_t::iterator itEnd = pIndexers->end();
  Value * pValue = pExpr->evaluate();

  //    _CT * pCTValue = dynamic_cast<_CT*>(pValue);
  //    if(pCTValue != NULL) {
  for (it = pIndexers->begin(); it != itEnd; ++it) {
    Value * pIndex = (*it)->evaluate();
    Integer *pIntIndex = dynamic_cast<Integer*>(pIndex);

    if (pIntIndex != nullptr) {
      SetAt(int(*pIntIndex), pValue);

    } else {
      AbstractRange *pRange = dynamic_cast<AbstractRange*>(pIndex);

      if (pRange) {
        unsigned int uEnd = pRange->get_rightVal() + 1;

        for (unsigned int uIndex = pRange->get_leftVal(); uIndex < uEnd; uIndex++) {
          SetAt(uIndex, pValue);
        }

      } else {
        /*
        register_symbol *pReg = dynamic_cast<register_symbol*>(pIndex);
        if(pReg) {
          SetAt(pReg->getReg()->address, pValue);
        }
        else {
          throw Error("indexer not valid");
        }
        */
        Register *pReg = dynamic_cast<Register*>(pIndex);

        if (pReg) {
          SetAt(pReg->getAddress(), pValue);

        } else {
          throw Error("indexer not valid");
        }
      }
    }

    delete pIndex;
  }

  delete pValue;
}


char *IIndexedCollection::toString(char *pBuffer, int len)
{
  return strncpy(pBuffer, toString().c_str(), len);
}


std::string IIndexedCollection::toString()
{
  int iColumnWidth = 0;
  std::vector<std::string> asIndexes;
  std::vector<std::string> asValue;
  ConsolidateValues(iColumnWidth, asIndexes, asValue);
  return toString(iColumnWidth, asIndexes, asValue);
}


std::string IIndexedCollection::toString(ExprList_t* pIndexerExprs)
{
  try {
    std::ostringstream sOut;

    if (!pIndexerExprs)  {
      sOut << toString() << std::ends;
      return sOut.str();

    } else {
      ExprList_t::iterator it = pIndexerExprs->begin();
      ExprList_t::iterator itEnd = pIndexerExprs->end();

      for (; it != itEnd; ++it) {
        Value * pIndex = (*it)->evaluate();
        AbstractRange *pRange = dynamic_cast<AbstractRange*>(pIndex);

        if (pRange) {
          unsigned uEnd = pRange->get_rightVal() + 1;

          for (unsigned int uIndex = pRange->get_leftVal(); uIndex < uEnd; uIndex++) {
            Value &Value = GetAt(uIndex);
            sOut << Value.name() << " = " << Value.toString() << '\n';
          }

          continue;
        }

        String *pName = dynamic_cast<String*>(pIndex);
        Integer *pInt = pName ?
                        globalSymbolTable().findInteger(pName->getVal()) :
                        dynamic_cast<Integer*>(pIndex);
        Integer temp(0);

        if (pInt == nullptr) {
          // This is a temp workaround. I (JR) would expect a register symbol
          // evaluate to an Integer object containing the value of the
          // register. It currently returns an object that is a copy
          // of the register_symbol object.
          /*
          register_symbol *pReg = dynamic_cast<register_symbol*>(pIndex);
          if(pReg) {
            gint64 i;
            pReg->get(i);
            temp.set(i);
            pInt = &temp;
          }
          */
          Register *pReg = dynamic_cast<Register*>(pIndex);

          if (pReg) {
            gint64 i = pReg->get_value();
            temp.set(i);
            pInt = &temp;
          }
        }

        if (pInt) {
          unsigned int uIndex = (unsigned int)pInt->getVal();

          if (bIsIndexInRange(uIndex)) {
            Value &Value = GetAt(uIndex);
            sOut << Value.name() << " = " << Value.toString() << '\n';

          } else {
            sOut << "Error: Index " << uIndex << " is out of range" << '\n';
          }

        } else {
          sOut << "Error: The index specified for '"
               << name() << "' does not contain a valid index.\n";
        }

        delete pIndex;
      }
    }

    sOut << std::ends;
    return sOut.str();

  } catch (Error &e) {
    return e.what();
  }
}


void IIndexedCollection::PushValue(int iFirstIndex, int iCurrentIndex,
                                   Value *pValue,
                                   std::vector<std::string> &asIndexes,
                                   std::vector<std::string> &asValue)
{
  std::ostringstream sIndex;

  if (m_iAddressRadix == 16) {
    sIndex << std::hex;
  }

  sIndex << Value::name() << "[" << m_szPrefix << iFirstIndex;

  if (iFirstIndex != iCurrentIndex) {
    sIndex << ".." << m_szPrefix << iCurrentIndex;
  }

  sIndex << "]" << std::ends;
  asIndexes.push_back(std::string(sIndex.str()));
  asValue.push_back(pValue->toString());
}


std::string IIndexedCollection::ElementIndexedName(unsigned int iIndex)
{
  std::ostringstream sIndex;

  if (m_iAddressRadix == 16) {
    sIndex << std::hex;
  }

  sIndex << Value::name() << "[" << m_szPrefix << iIndex;
  sIndex << "]" << std::ends;
  return sIndex.str();
}


std::string IIndexedCollection::toString(int iColumnWidth,
    std::vector<std::string> &asIndexes,
    std::vector<std::string> &asValue)
{
  std::ostringstream sOut;
  std::vector<std::string>::iterator itValue = asValue.begin();
  std::vector<std::string>::iterator itElement = asIndexes.begin();
  std::vector<std::string>::iterator itElementEnd = asIndexes.end();

  // Dump the consolidated element list
  for (; itElement != itElementEnd; ++itElement, ++itValue) {
    sOut.width(iColumnWidth);
    sOut.setf(std::ios_base::left);
    sOut << (*itElement) << " = " << (*itValue);

    if (itElement + 1 != itElementEnd) {
      sOut << std::endl;
    }
  }

  sOut << std::ends;
  return sOut.str();
}


Integer * IIndexedCollection::FindInteger(const char *s)
{
  return globalSymbolTable().findInteger(s);
}

