/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

package org.netbeans.modules.apisupport.project.ui.branding;

import org.netbeans.modules.apisupport.project.spi.BrandingModel;
import java.awt.Dialog;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import org.openide.DialogDescriptor;
import org.openide.DialogDisplayer;
import org.openide.util.ImageUtilities;
import org.openide.util.NbBundle;

/**
 *
 * @author S. Aubrecht
 */
public class BrandingEditorPanel extends javax.swing.JPanel {

    private final String title;
    private final BrandingModel model;
    private final AbstractBrandingPanel[] panels;
    private boolean isModified = false;
    private boolean isValid = true;
    private DialogDescriptor descriptor;
    private boolean brandingEnabled = true;

    public BrandingEditorPanel(String title, BrandingModel model) {
        initComponents();
        this.title = title;
        this.model = model;
        panels = new AbstractBrandingPanel[] {
            new BasicBrandingPanel(model),
            new SplashBrandingPanel(model),
            new WindowSystemBrandingPanel(model),
            new ResourceBundleBrandingPanel(model),
            new InternationalizationResourceBundleBrandingPanel(model)
        };
        for( AbstractBrandingPanel panel : panels ) {
            panel.init(this);
            tabbedPane.add(panel.getDisplayName(), panel);
        }
        lblError.setIcon(ImageUtilities.loadImageIcon("org/netbeans/modules/apisupport/project/ui/resources/error.gif", true)); //NOI18N
        lblError.setVisible(false);
        //TODO restrict the functionality of generic resource bundle editor when platform context isn't available
    }

    /** This method is called from within the constructor to
     * initialize the form.
     * WARNING: Do NOT modify this code. The content of this method is
     * always regenerated by the Form Editor.
     */
    @SuppressWarnings("unchecked")
    // <editor-fold defaultstate="collapsed" desc="Generated Code">//GEN-BEGIN:initComponents
    private void initComponents() {

        tabbedPane = new javax.swing.JTabbedPane();
        lblError = new javax.swing.JLabel();

        setBorder(javax.swing.BorderFactory.createEmptyBorder(10, 10, 10, 10));
        setLayout(new java.awt.BorderLayout(0, 5));
        add(tabbedPane, java.awt.BorderLayout.CENTER);
        tabbedPane.getAccessibleContext().setAccessibleName(NbBundle.getMessage(BrandingEditorPanel.class, "ACS_BrandingTabs")); // NOI18N
        tabbedPane.getAccessibleContext().setAccessibleDescription(NbBundle.getMessage(BrandingEditorPanel.class, "ACS_BrandingTabs")); // NOI18N

        add(lblError, java.awt.BorderLayout.SOUTH);
    }// </editor-fold>//GEN-END:initComponents


    // Variables declaration - do not modify//GEN-BEGIN:variables
    private javax.swing.JLabel lblError;
    private javax.swing.JTabbedPane tabbedPane;
    // End of variables declaration//GEN-END:variables

    public Dialog open() {
        descriptor = new DialogDescriptor(this, title, false, DialogDescriptor.OK_CANCEL_OPTION, null, new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                if( descriptor.getValue() == DialogDescriptor.OK_OPTION ) {
                    doSave();
                }
            }
        });
        Dialog dlg = DialogDisplayer.getDefault().createDialog(descriptor);
        dlg.getAccessibleContext().setAccessibleName(NbBundle.getMessage(BrandingEditorPanel.class, "ACS_BrandingCustomizer")); //NOI18N
        dlg.getAccessibleContext().setAccessibleDescription(NbBundle.getMessage(BrandingEditorPanel.class, "ACS_BrandingCustomizer")); //NOI18N
        dlg.setVisible(true);
        //it's possible to open project properties window while branding editor is opened
        //and switch from application suite to module suite which doesn't support branding
        //so let's prevent saving when in 'module suite' mode
        dlg.addWindowListener(new WindowAdapter() {
            @Override
            public void windowActivated(WindowEvent e) {
                refreshBrandingEnabled();
            }

            @Override
            public void windowGainedFocus(WindowEvent e) {
                refreshBrandingEnabled();
            }
        });
        return dlg;
    }

    void setModified() {
        isModified = true;
    }

    boolean isModified() {
        return isModified;
    }

    private void doSave() {
        for( AbstractBrandingPanel panel : panels ) {
            panel.store();
        }
        model.doSave();
        isModified = false;
    }


    void onBrandingValidation() {
        isValid = true;
        String errMessage = null;
        for( AbstractBrandingPanel panel : panels ) {
            isValid &= panel.isBrandingValid();
            String msg = panel.getErrorMessage();
            if( null != msg && null == errMessage )
                errMessage = msg;
        }
        lblError.setText(errMessage);
        lblError.setVisible(!isValid);
        if( null != descriptor )
            descriptor.setValid(isValid && brandingEnabled);
    }

    private void refreshBrandingEnabled() {
        model.reloadProperties();
        model.brandingEnabledRefresh();
        brandingEnabled = model.isBrandingEnabled();
        if( null != descriptor ) {
            descriptor.setValid( isValid && brandingEnabled );
        }
    }
}
