from abc import ABC, abstractmethod
from typing import FrozenSet, NamedTuple, Optional, Tuple


__all__ = [  # pylint: disable=unused-variable
    "Content",
    "EncryptedKeyMaterial",
    "KeyExchange",
    "Message",
    "PlainKeyMaterial"
]


class Content(ABC):
    """
    The encrypted content of an OMEMO-encrypted message. Contains for example the ciphertext, but can contain
    other backend-specific data that is shared between all recipients.
    """

    @property
    @abstractmethod
    def empty(self) -> bool:
        """
        Returns:
            Whether this instance corresponds to an empty OMEMO message purely used for protocol stability
            reasons.
        """


class PlainKeyMaterial(ABC):
    """
    Key material which be used to decrypt the content. Defails are backend-specific.
    """


class EncryptedKeyMaterial(ABC):
    """
    Encrypted key material. When decrypted, the key material can in turn be used to decrypt the content. One
    collection of key material is included in an OMEMO-encrypted message per recipient. Defails are
    backend-specific.
    """

    @property
    @abstractmethod
    def bare_jid(self) -> str:
        # pylint: disable=missing-function-docstring
        pass

    @property
    @abstractmethod
    def device_id(self) -> int:
        # pylint: disable=missing-function-docstring
        pass


class KeyExchange(ABC):
    """
    Key exchange information, generated by the active part of the session building process, then transferred
    to and consumed by the passive part of the session building process. Details are backend-specific.
    """

    @property
    @abstractmethod
    def identity_key(self) -> bytes:
        # pylint: disable=missing-function-docstring
        pass

    @abstractmethod
    def builds_same_session(self, other: "KeyExchange") -> bool:
        """
        Args:
            other: The other key exchange instance to compare to this instance.

        Returns:
            Whether the key exchange information stored in this instance and the key exchange information
            stored in the other instance would build the same session.
        """


class Message(NamedTuple):
    # pylint: disable=invalid-name
    """
    Simple structure representing an OMEMO-encrypted message.
    """

    namespace: str
    bare_jid: str
    device_id: int
    content: Content
    keys: FrozenSet[Tuple[EncryptedKeyMaterial, Optional[KeyExchange]]]
