#' Generate your own Weighted C++ Roll Function
#'
#' Using this interface, you can define a function that you would like to
#' be called on each sub-vector you are rolling over. The generated code is
#' compiled and exposed via \code{sourceCpp}.
#'
#' By default, we include \code{<RcppArmadillo.h>} in each file; however, you can
#' include your own libraries with the \code{includes} call.
#'
#' @param fun A character string defining the function call. The function must be in
#' terms of \code{x}. The function will be applied individually
#' to each element being 'roll'ed over, unless \code{vector} is \code{TRUE}.
#' @param vector boolean; if \code{TRUE}, then \code{fun} is a scalar-valued
#' function of a vector, rather than a function to apply to each element
#' individually.
#' @param const_vars Constant variables that will live within
#' the sourced C++ function. Format is a named \code{list}; e.g, you
#' could pass \code{list(e=exp(1))} to have \code{e} as a constant variable
#' available in the function being called. Note that the variable \code{N}
#' is fixed to be the number of non-zero weights passed, to facilitate the use
#' of \code{0} weights in terms of skipping elements.
#' @param combine character; typically one of \code{"+", "-", "*", "/"}, but
#' any operator usable as a C++ compound assignment operator is accepted.
#' This specifies how each element should be combined in the rolling function.
#' Only used when \code{vector} is \code{FALSE}.
#' @param final_trans A final transformation to perform after either 'rolling'
#' over each element in the vector \code{x} (with \code{vector=FALSE}),
#' or after applying a scalar-valued function of a vector (with \code{vector=TRUE}).
#' Must be in terms of \code{x}.
#' @param includes Other C++ libraries to include. For example, to include
#' \code{boost/math.hpp}, you would pass
#' \code{c("<boost/math.hpp>")}.
#' @param depends Other libraries to link to. Linking is done through
#' Rcpp attributes.
#' @param inline boolean; mark the function generated as \code{inline}?
#' This may or may not increase execution speed.
#' @param name string; a name to internally assign to your generated C++ functions.
#' @param ... Additional arguments passed to \code{sourceCpp}.
#' @return A wrapper \R function that calls compiled C++ files, as generated through
#' \code{sourceCpp}. See \code{\link{rollit_example}} for more information on the
#' functions generated by \code{rollit}.
#' @export
#' @seealso \code{\link{rollit_example}} for an example of the function signature
#' for functions generated with \code{rollit},
#' \code{\link{sourceCpp}} for information on how Rcpp
#' compiles your functions, \code{\link{get_rollit_source}} for
#' inspection of the generated C++ code, and \code{\link{rollit_raw}} for
#' wrapping in your own C++ code.
#' @note All functions generated use Rcpp's \code{NumericVector} and
#' \code{NumericMatrix} to interface with \R vectors and matrices.
#' Elements within these vectors are
#' translated as \code{double}s so any function that receives a \code{double}
#' will work fine.
#'
#' If you want to just write your own C++ function to wrap into a 'rolling'
#' interface, see \code{\link{rollit_raw}}.
#'
#' @examples \dontrun{
#' x <- matrix(1:16, nrow=4)
#'
#' ## the squared rolling sum -- we square the sum of our rolled results
#' rolling_sqsum <- rollit( final_trans="x*x" )
#'
#' rolling_sqsum( x, 4 )
#' rolling_sqsum( x, 4, by.column=FALSE )
#' cbind( as.vector(rolling_sqsum(x, 4)), apply(x, 2, function(x) { sum(x)^2 } ) )
#'
#' ## implement your own variance function
#' ## we can use the sugar function 'mean' to get
#' ## the mean of x
#'
#' const_vars <- list(m = "mean(x)")
#' var_fun <- "( (x-m) * (x-m) )/(N-1)"
#' rolling_var <- rollit( var_fun, const_vars=const_vars )
#'
#' x <- c(1, 5, 10, 15)
#' cbind( rolling_var(x, 2), roll_var(x, 2) )
#'
#' ## use a function from cmath
#'
#' rolling_log10 <- rollit( "log10(x)" )
#' rolling_log10( 10^(1:5), 2 )
#'
#' ## rolling product
#' rolling_prod <- rollit( combine="*" )
#' rolling_prod( 1:10, 2 )
#'
#' ## using weights to specify something like a 'by' argument
#' rolling_prod( 1:10, 3, weights=c(1,0,1) )
#'
#' ## a benchmark
#'
#' if( require("microbenchmark") && require("zoo") ) {
#'   x <- rnorm(1E4)
#'   microbenchmark(
#'     rolling_var(x, 100),
#'     roll_var(x, 100),
#'     rollapply(x, 100, var),
#'     times=10
#'     )
#'   }}
#' @importFrom Rcpp sourceCpp
rollit <- function(fun = "x",
                   vector = FALSE,
                   const_vars = NULL,
                   combine = "+",
                   final_trans = NULL,
                   includes = NULL,
                   depends = NULL,
                   inline = TRUE,
                   name = NULL,
                   ...) {

  .Deprecated()

  ## error checks
  if (!is.null(const_vars)) {
    if (!is.list(const_vars) ||
        is.list(const_vars) && is.null(names(const_vars))) {
      stop("'const_vars' must be a named list")
    }
  }

  if (length(combine) > 1 ||
      !(combine %in% c("+", "-", "*", "/", "&", "|", "^", "<<", ">>"))) {
    stop("combine must be one of '+', '-', '*', '/', '&', '|', '^', '<<', '>>'")
  }

  funky_regex <- "([^a-zA-Z_])(x)(?=[^x])|(\\Ax)|(x\\z)"
  if (length(grep(funky_regex, fun, perl = TRUE)) < 1) {
    stop("'fun' must be in terms of a variable 'x'")
  }

  ## random name if null
  if (is.null(name)) {
    random_string <- sample(c(letters, LETTERS, 0:9), 20, TRUE)
    name <- paste(sep = "", collapse = "", c("z", random_string))
  }

  ## environment for cppSource generated files
  cpp_env <- new.env()

  outFile <- paste(sep = "", tempfile(), ".cpp")
  conn <- file(outFile, open = "w")
  on.exit(close(conn))

  w <- function(...) {
    cat(paste0(..., "\n"), file = conn)
  }

  w1 <- function(...) {
    cat(paste0("\t", ..., "\n"), file = conn)
  }

  w2 <- function(...) {
    cat(paste0("\t\t", ..., "\n"), file = conn)
  }

  w3 <- function(...) {
    cat(paste0("\t\t\t", ..., "\n"), file = conn)
  }

  w4 <- function(...) {
    cat(paste0("\t\t\t\t", ..., "\n"), file = conn)
  }

  ## depends
  if (is.null(depends)) {
    w("// [[Rcpp::depends(RcppArmadillo)]]")
  } else {
    w("// [[Rcpp::depends(RcppArmadillo, ",
      paste(depends, collapse = ", "),
      ")")
  }

  w("#include <RcppArmadillo.h>")
  if (!is.null(includes)) {
    for (include in includes) {
      w(paste0("#include ", include))
      w()
    }
  }

  ## namespace
  w("using namespace Rcpp;")
  w()
  w("typedef NumericVector::iterator iter;")
  w("typedef NumericVector NV;")
  w("typedef NumericMatrix NM;")
  w()

  ## wrap the function provided by the user

  if (inline) w("inline")
  w("double ", name, "(NV& x, NV& weights, const int& n, const int& N, const int& ind) {")
  if (combine %in% c("+", "-")) {
    w1("double out_ = 0;")
  } else {
    w1("double out_ = 1;")
  }

  ## constant variables
  ## be sure to parse any functions of x within the constant variables
  if (!is.null(const_vars)) {
    for(i in seq_along(const_vars)) {
      tmp <-
        gsub(funky_regex, "\\1\\2\\3\\4\\5[ seq(ind, ind+n-1) ]", const_vars[i], perl =
               TRUE)
      w1("const double ", names(const_vars)[i], " = ", tmp, ";")
    }
  }


  ## funky parser
  (parsed_fun <-
     gsub(funky_regex, "\\1\\2\\3\\4\\5[i+ind]", fun, perl = TRUE))

  ## apply function as vector
  if (vector) {
    w1(
      "out_ = ", gsub(
        funky_regex, "\\1\\2\\3\\4\\5[ seq(ind, ind+n-1) ] * weights", fun, perl =
          TRUE
      ), ";"
    )
  } else {
    ## apply function elementwise
    w1("for( int i=0; i < n; i++ ) {")
    w2("if( weights[i] != 0 ) {")
    w3("out_ ", combine, "= weights[i] * ", parsed_fun, ";")
    w2("}")
    w1("}")
  }

  if (!is.null(final_trans)) {
    w1("out_ = ", gsub(funky_regex, "\\1out_", final_trans, perl = TRUE), ";")
  }
  w1("return out_;")
  w("}")
  w()

  ## numericvector call
  w("// [[Rcpp::export]]")
  w("NumericVector ", name, "_numeric( NumericVector x, int n, NumericVector weights ) {")
  w1()
  w1("int len = x.size();")
  w1("int len_out = len - n + 1;")
  w1("int N = sum( sign( weights*weights ) );")
  w1()
  w1("NV out = no_init( len_out );")
  w1()
  w1("for( int ind=0; ind < len_out; ind++ ) {")
  w2()
  w2("out[ind] = ", name, "(x, weights, n, N, ind );")
  w1("}")
  w1()
  w1("return out;")
  w1()
  w("}")
  w()

  ## function definition -- matrix

  w("// [[Rcpp::export]]")
  w(
    "NumericMatrix ", name, "_matrix( NumericMatrix A, int n, bool by_column, NumericVector weights ) {"
  )
  w1()
  w1("int nRow = A.nrow();")
  w1("int nCol = A.ncol();")
  w1("int N = sum( sign( weights*weights ) );")

  ## by column
  w1("if( by_column ) {")
  w2()
  w2("NumericMatrix B( nRow - n + 1, nCol );")
  w2()
  w2("for( int j=0; j < nCol; j++ ) {")
  w3()
  w3("NumericVector tmp = A(_, j);")
  w3("for( int ind=0; ind < nRow - n + 1; ind++ ) {")
  w4()
  w4("B(ind, j) = ", name, "( tmp, weights, n, N, ind );")
  w3("}")
  w2("}")
  w2()
  w1("return B;")
  w1()

  ## by row
  w1("} else {")
  w2()
  w2("NumericMatrix B( nRow, nCol - n + 1 );")
  w2()
  w2("for( int i=0; i < nRow; i++ ) {")
  w3()
  w3("NumericVector tmp = A(i, _);")
  w3("for( int ind=0; ind < nCol - n + 1; ind++ ) {")
  w4()
  w4("B(i, ind) = ", name, "( tmp, weights, n, N, ind );")
  w3("}")
  w2("}")
  w2()
  w1("return B;")
  w1()
  w1("}")
  w()
  w("}")


  cat("C++ source file written to", outFile, ".\n")
  cat("Compiling...\n")
  sourceCpp(outFile, env = cpp_env, ...)
  cat("Done!\n")

  return(function(x, n, by.column = TRUE, weights = rep(1,n), normalize =
                    FALSE) {
    force(combine)
    force(outFile)

    if (length(weights) != n) {
      stop("length of weights must equal n")
    }

    if (normalize) {
      weights <- weights * length(weights) / sum(weights)
    }

    if (is.matrix(x)) {
      if (n > nrow(x)) {
        stop("n cannot be greater than nrow(x)")
      }
      call <- call(
        paste(sep = "", name, "_matrix"),
        x,
        as.integer(n),
        as.logical(by.column),
        as.numeric(weights)
      )
      return(eval(call, envir = cpp_env))
    }

    if (is.vector(x)) {
      if (n > length(x)) {
        stop("n cannot be greater than length(x)")
      }
      call <- call(paste(sep = "", name, "_numeric"),
                   x,
                   as.integer(n),
                   as.numeric(weights))
      return(as.numeric(eval(call, envir = cpp_env)))
    }

    stop("the x supplied is neither a vector nor a matrix")

  })

}
