dnl Copyright 2006, Klaus Triendl
dnl 
dnl This library is free software; you can redistribute it and/or 
dnl modify it under the terms of the GNU Lesser General Public 
dnl License as published by the Free Software Foundation; either 
dnl version 2.1 of the License, or (at your option) any later version. 
dnl 
dnl This library is distributed in the hope that it will be useful, 
dnl but WITHOUT ANY WARRANTY; without even the implied warranty of 
dnl MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU 
dnl Lesser General Public License for more details. 
dnl 
dnl You should have received a copy of the GNU Library General Public
dnl License along with this library; if not, write to the Free 
dnl Software Foundation, 51 Franklin Street, Fifth Floor, 
dnl Boston, MA 02110-1301, USA.
dnl
divert(-1)

include(template.macros.m4)

define([REQUEST_F],[dnl
/**	@short Asynchronous request functor for a sigx::threadable.
 *	
 *	It saves you writing request methods that have to call a handler method
 *	for the request through a tunnel, like:
 *	
 *	@code
 *	class IPResolverThread: public sigx::threadable
 *	{
 *	public:
 *		void resolve(in_addr_t nIP);
 *		void stop_resolving();
 *	};
 *	
 *	void IPResolverThread::resolve(in_addr_t nIP)
 *	{
 *		sigx::open_tunnel(sigc::mem_fun(this, &IPResolverThread::on_resolve))(nIP);
 *	}
 *	
 *	void IPResolverThread::stop_resolving()
 *	{
 *		sigx::open_tunnel(sigc::mem_fun(this, &IPResolverThread::on_stop_resolving))();
 *	}
 *	@endcode
 *	
 *	Instead, delegate it to the request functor:
 *	
 *	@code
 *	class IPResolverThread: public sigx::threadable
 *	{
 *	public:
 *		sigx::request_f<in_addr_t> resolve;
 *		sigx::request_f<> stop_resolving;
 *	};
 *	
 *	IPResolverThread::IPResolverThread(): 
 *		resolve(sigc::mem_fun(this, &IPResolverThread::on_resolve)), 
 *		stop_resolving(sigc::mem_fun(this, &IPResolverThread::on_stop_resolving))
 *	{}
 *	@endcode
 *	
 *	It is derived from %sigc::slot because a slot provides already all the
 *	necessary functionalities: takes a functor and creates a untyped slot
 *	representation, has function invokation operator ().
 *	
 *	@attention Do not specify a return type as the first template parameter.
 *	As asynchronous tunnels actually do not have a return type, @e request_f 
 *	omits it, thus the return type is always `void".
 *	
 *	@note non-copyable, not constructible on the heap (with new) and can't be 
 *	pointer aliased (with operator &) to ensure that it is de-facto bound to  
 *	a wrapping object.
 *	
 *	@ingroup signals
 */
template<LOOP(typename T_arg%1 = sigc::nil, CALL_SIZE)>
class request_f: LIST(noncopyable, nonheapallocatable, nonpointeraliasing, protected sigc::slot<[LIST(void, [LOOP(T_arg%1, CALL_SIZE)]]))>
{
public:
	typedef sigc::slot<void, LOOP(T_arg%1, CALL_SIZE)> parent_type;
	
	// allow function operator to be used
	using parent_type::operator ();

	/**	@short Constructs the request functor.
	 *	
	 *	@note The passed in functor must not be a sigc::slot and must not be 
	 *	a tunnel functor.
	 *	The passed in functor gets tunneled automatically.
	 *	
	 *	@param _A_func A dispatchable functor, i.e. a functor on a dispatchable's
	 *	method or one explicitly created with dispatch_with().
	 */
	template<typename T_functor>
	explicit request_f(const T_functor& _A_func): 
		parent_type(tunnel_functor<ASYNC, T_functor>(_A_func))
	{
		// passed in functor must not be tunneled
		SIGX_STATIC_ASSERT((internal::is_functor_tunneled<T_functor>::value == false));

		// passed in functor must not be a slot or adapt a slot;
		// we have to apply this restriction because slots might have bound
		// trackables that can cause non-threadsafe access to the passed in slot
		// which will live in the context of the server thread
		SIGX_STATIC_ASSERT((sigx::internal::is_or_adapts_slot<T_functor>::value == false));
	}
};
])


divert(0)

__FIREWALL__

COPYRIGHT_NOTICE

#include <sigc++/slot.h>
#include <sigx/noncopyable.h>
#include <sigx/nonheapallocatable.h>
#include <sigx/nonpointeraliasing.h>
#include <sigx/internal_types.h>
#include <sigx/static_assert.h>


namespace sigx
{

PROT(REQUEST_F)

} // namespace sigx
