dnl Copyright 2006, Klaus Triendl
dnl 
dnl This library is free software; you can redistribute it and/or 
dnl modify it under the terms of the GNU Lesser General Public 
dnl License as published by the Free Software Foundation; either 
dnl version 2.1 of the License, or (at your option) any later version. 
dnl 
dnl This library is distributed in the hope that it will be useful, 
dnl but WITHOUT ANY WARRANTY; without even the implied warranty of 
dnl MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU 
dnl Lesser General Public License for more details. 
dnl 
dnl You should have received a copy of the GNU Library General Public
dnl License along with this library; if not, write to the Free 
dnl Software Foundation, 51 Franklin Street, Fifth Floor, 
dnl Boston, MA 02110-1301, USA.
dnl
divert(-1)

include(template.macros.m4)


define([COUNT_SIGNAL_ARGUMENTS_FOR_NUMBERED_SIGGROUP_SIGCS], [dnl
/**	@short counts the arguments of a sigc::signal$1
 */
template<LIST(typename T_return, LOOP(typename T_arg%1, $1), typename T_accumulator)>
struct count_signal_arguments<sigc::signal$1<LIST(T_return, LOOP(T_arg%1, $1), T_accumulator)> >
{
	static const int value = $1;
	// template specialization for argument count needed
	static const int tspec = value;
};

])

define([COUNT_SIGNAL_ARGUMENTS_FOR_NUMBERED_GLIB_SIGNALS], [dnl
/**	@short counts the arguments of a Glib::SignalProxy$1
 */
template<LIST(typename T_return, LOOP(typename T_arg%1, $1))>
struct count_signal_arguments<Glib::SignalProxy$1<LIST(T_return, LOOP(T_arg%1, $1))> >
{
	static const int value = $1;
	// template specialization for argument count not needed;
	// this allows us to group all SignProxyN signals together in one template
	// class
	static const int tspec = -1;
};

])


define([ARG_N_TYPE_], [dnl
	typedef typename slot_type::arg$1_type_ arg$1_type_;
])

	
define([SIGNAL], [dnl
/**	@short A threadsafe wrapper for any sigc signal with $1 argument(s).
 *	
 *	@ingroup signals
 */
template<typename T_signal>
class signal_wrapper<T_signal, internal::SIGGROUP_SIGC, $1>: public signal_wrapper_base
{
public:
	//BOOST_STATIC_ASSERT((internal::count_signal_arguments<T_signal>::value == $1));
	static const int argument_count = $1;
	static const internal::signal_group signal_group = internal::SIGGROUP_SIGC;
	typedef T_signal signal_type;
	typedef signal_wrapper<signal_type, signal_group, argument_count> this_type;
	typedef typename signal_type::slot_type slot_type;
	typedef typename signal_type::result_type result_type;
	FOR(1, $1, [[ARG_N_TYPE_(%1)]])
	typedef signal_type (*fp_sig_getter_type)(signal_source_ptr);
	
protected:
	typedef sigc::bound_const_mem_functor0<signal_source_base::hook, signal_source_base> make_slot_f1;
	typedef sigc::retype_return_functor<fp_sig_getter_type, make_slot_f1> make_slot_f2;
	typedef sigc::const_mem_functor0<signal_source_ptr, std::tr1::shared_ptr<signal_source_base> > make_slot_f3;
	typedef sigc::bind_functor<-1, make_slot_f3, std::tr1::shared_ptr<signal_source_base> > make_slot_f4;
	typedef sigc::compose1_functor<make_slot_f2, make_slot_f4> make_slot_composed1_functor_type;
	typedef sigc::const_mem_functor$1<LIST(result_type, signal_type, LOOP(arg%1_type_, $1))> make_slot_emit_functor_type;
	typedef sigc::compose1_functor<make_slot_emit_functor_type, make_slot_composed1_functor_type> make_slot_composed2_functor_type;
	typedef SIGX_DISPATCH_WITH_FUNCTOR(make_slot_composed2_functor_type) make_slot_functor_type;


public:
	/**	@short Constructs an empty signal_wrapper.
	 *	@throw	Might throw a std::bad_alloc exception (from dispatchable's ctor)
	 */
	signal_wrapper(): signal_wrapper_base()
	{}

	/**	@short Creates a signal_wrapper from a signal source.
	 *	@param _A_disp The dispatchable to operate on
	 *	@param _A_sigsource A shared pointer to the server 
	 *	thread's signal source.
	 */
	signal_wrapper(const shared_dispatchable& _A_disp, const std::tr1::shared_ptr<signal_source_base>& _A_sigsource) throw(): 
		signal_wrapper_base(_A_disp, _A_sigsource)
	{}
	
	/**	@short Connects a functor, tunnels it automatically if not yet 
	 *	tunneled and activates validity tracking for sigc::trackableS.
	 *	
	 *	auto tunneling is successful only if the passed in functor is a
	 *	"dispatchable" functor, i.e. a functor on a dispatchable's method
	 *	or one explicitly created with "dispatch_with".
	 *	You will get compiler errors if the dispatchable can't be deduced from the 
	 *	passed in functor.
	 *	
	 *	@note At the moment it is only possible to pass in a non-tunneled functor or
	 *	a toplevel tunneled functor due to the fact that the visit_each mechanism
	 *	is turned off for the tunnel functor (otherwise there would be the problem
	 *	of not threadsafe access to the sigc::trackable base of the of a 
	 *	dispatchable object..
	 *	
	 *	@note passed in functor must not be a slot or adapt a slot;
	 *	we have to apply this restriction because slots might have bound
	 *	trackables that can cause non-threadsafe access to the passed in slot
	 *	which will live in the context of the server thread.
	 *	
	 *	@attention	All sigc::trackableS and the original dispatchable contained 
	 *	in the passed functor must belong to the context of the calling thread.
	 *	
	 *	@return sigx::connection_wrapper A threadsafe connection wrapper
	 *	@note asynchronous
	 */
	template<typename T_functor>
	connection_wrapper connect(const T_functor& _A_func) const
	{
		return	signal_wrapper_base::connect(
					_A_func, 
					sigc::ptr_fun(&typed_connection_handler<signal_type, signal_group>::connect)
				);
	}

	/**	@short emits the signal on the other side of the tunnel.
	 */
	template<sync_type I_sync>
	result_type emit(LOOP(arg%1_type_ _A_a%1, $1)) const
	{
		return open_tunnel_with<I_sync>(
			// calls T_signal::*emit
			sigc::compose(
				sigc::mem_fun(&signal_type::emit), 
				// getter for the T_signal
				sigc::compose(
					sigc::retype_return<fp_sig_getter_type>(
						sigc::mem_fun(m_sigsource.get(), &signal_source_base::getter)
					), 
					// this makes a copy of the shared signal source and thus 
					// shares it within the tunnel functor ensuring the lifetime 
					// of the shared signal source
					sigc::bind(
						sigc::mem_fun(&std::tr1::shared_ptr<signal_source_base>::operator *), 
						m_sigsource
					)
				)
			), 
			m_disp
		)(LOOP(_A_a%1, $1));
	}

	/**	@short emits the signal asynchronously.
	 */
	result_type emit(LOOP(arg%1_type_ _A_a%1, $1)) const
	{
		return emit<ASYNC>(LOOP(_A_a%1, $1));
	}

	/**	@short emits the signal synchronously.
	 */
	result_type emit_sync(LOOP(arg%1_type_ _A_a%1, $1)) const
	{
		return emit<SYNC>(LOOP(_A_a%1, $1));
	}

	/**	see emit(LOOP(arg%1_type_, $1))
	 */
	result_type operator()(LOOP(arg%1_type_ _A_a%1, $1)) const
	{
		return emit<ASYNC>(LOOP(_A_a%1, $1));
	}


	/**	@short creates a tunnel_functor that emits the signal when invoked
	 */
	template<sync_type I_sync>
	tunnel_functor<I_sync, make_slot_functor_type> make_slot() const
	{
		typedef tunnel_functor<I_sync, make_slot_functor_type> tunnel_funtor_type;
		return tunnel_funtor_type(
			dispatch_with(
				// calls T_signal::*emit
				sigc::compose(
					&signal_type::emit, 
					// getter for the T_signal
					sigc::compose(
						sigc::retype_return<fp_sig_getter_type>(
							sigc::mem_fun(m_sigsource.get(), &signal_source_base::getter)
						), 
						sigc::bind(
							sigc::mem_fun(&std::tr1::shared_ptr<signal_source_base>::operator *), 
							m_sigsource
						)
					)
				), 
				m_disp
			));
	}

	/**	@short creates an asynchronous tunnel_functor that emits the signal 
	 *	when invoked
	 */
	tunnel_functor<ASYNC, make_slot_functor_type> make_slot() const
	{
		return make_slot<ASYNC>();
	}

	/**	@short creates a synchronous tunnel_functor that emits the signal 
	 *	when invoked
	 */
	tunnel_functor<SYNC, make_slot_functor_type> make_slot_sync() const
	{
		return make_slot<SYNC>();
	}
};

])


divert(0)

__FIREWALL__

COPYRIGHT_NOTICE

#include <tr1/memory> // std::tr1::shared_ptr, std::auto_ptr
#include <sigc++/sigc++.h>
#include <glibmm/main.h>	// glibmm signals
#include <sigx/types.h>
#include <sigx/internal_types.h>
#include <sigx/signal_traits.h>
#include <sigx/signal_wrapper_base.h>
#include <sigx/signal_source_base.h>
#include <sigx/tunnel_functor.h>
#include <sigx/connection_handler.h>
#include <sigx/connection_wrapper.h>


/**	@defgroup signals Signals
 *	Threadsafe signals on top of the @ref Functors and @ref Dispatching
 *	facilities
 */


namespace sigx
{

	namespace internal
	{
	
/**	@short Counts a signal's arguments, default class
 */
template<typename T_signal>
struct count_signal_arguments
{
	static const int value = -1;
	static const int tspec = -1;
};


/**	@short Counts the arguments of an unnumbered sigc::signal
 */
template<LIST(typename T_return, LOOP(typename T_arg%1, CALL_SIZE))>
struct count_signal_arguments<sigc::signal<LIST(T_return, LOOP(T_arg%1, CALL_SIZE))> >
{
	// forward to count_arguments and not to count_signal_arguments, otherwise
	// we would get a false count if there is a another signal as the first 
	// argument of a signal, like: sigc::signal<void, sigc::signal<void> >
	static const int value = count_arguments<LOOP(T_arg%1, CALL_SIZE)>::value;
	static const int tspec = value;
};

FOR(0, CALL_SIZE, [[COUNT_SIGNAL_ARGUMENTS_FOR_NUMBERED_SIGGROUP_SIGCS(%1)]])
FOR(0, 6, [[COUNT_SIGNAL_ARGUMENTS_FOR_NUMBERED_GLIB_SIGNALS(%1)]])

/**	@short Counts the arguments of a Glib::SignalIdle
 */
template<>
struct count_signal_arguments<Glib::SignalIdle>
{
	static const int value = 0;
	static const int tspec = value;
};

/**	@short Counts the arguments of a Glib::SignalTimeout
 */
template<>
struct count_signal_arguments<Glib::SignalTimeout>
{
	static const int value = 0;
	static const int tspec = value;
};

/**	@short Counts the arguments of a Glib::SignalIO
 */
template<>
struct count_signal_arguments<Glib::SignalIO>
{
	static const int value = 1;
	static const int tspec = value;
};

/**	@short Counts the arguments of a Glib::SignalChildWatch
 */
template<>
struct count_signal_arguments<Glib::SignalChildWatch>
{
	static const int value = 2;
	static const int tspec = value;
};


	} // namespace internal


/**	@short	A threadsafe wrapper for sigc signals, Glib
 *	signals or theoretically any other type of signal.
 *	
 *	sigx signals have a shared signal source that exists in the context of
 *	another thread. This signal source has access to the signal. Access (e.g. 
 *	connecting) is regulated by a dispatcher running in the context of the 
 *	thread owning that signal.
 */
template<typename T_signal, internal::signal_group I_oneof = internal::signal_type_trait<T_signal>::type, int I_arg_count = internal::count_signal_arguments<T_signal>::tspec>
class signal_wrapper;

FOR(0, CALL_SIZE, [[SIGNAL(%1)]])


/**	@short A threadsafe wrapper for a Glib::SignalNormalProxy derived signal.
 *	@ingroup signals
 */
template<typename T_signal>
// have to specialize the argument count explicitly because template arguments
// can't be involved as template parameters in further template arguments
class signal_wrapper<T_signal, internal::SIGGROUP_GLIB_PROXY, -1>: public signal_wrapper_base
{
public:
	static const int argument_count = internal::count_signal_arguments<T_signal>::value;
	static const internal::signal_group signal_group = internal::SIGGROUP_GLIB_PROXY;
	typedef T_signal signal_type;
	typedef signal_wrapper<signal_type, signal_group, argument_count> this_type;
	typedef typename signal_type::SlotType slot_type;
	typedef typename signal_type::VoidSlotType void_slot_type;
	typedef typename slot_type::result_type result_type;
	typedef signal_type (*fp_sig_getter_type)(signal_source_ptr);


	/**	@short Constructs an empty signal_wrapper.
	 *	@throw	Might throw a std::bad_alloc exception (from dispatchable's ctor)
	 */
	signal_wrapper(): signal_wrapper_base()
	{}

	/**	@short Creates a signal_wrapper from a signal source.
	 *	@param _A_disp The dispatchable to operate on
	 *	@param _A_sigsource A double pointer to the server 
	 *	thread's signal source.
	 */
	signal_wrapper(const shared_dispatchable& _A_disp, const std::tr1::shared_ptr<signal_source_base>& _A_sigsource): 
		signal_wrapper_base(_A_disp, _A_sigsource)
	{}
	
	/**	@short Connects a functor, tunnels it automatically if not yet 
	 *	tunneled and activates validity tracking for sigc::trackableS.
	 *	
	 *	auto tunneling is successful only if the passed in functor is a
	 *	"dispatchable" functor, i.e. a functor on a dispatchable's method
	 *	or one explicitly created with "dispatch_with".
	 *	You will get compiler errors if the dispatchable can't be deduced from the 
	 *	passed in functor.
	 *	
	 *	@note At the moment it is only possible to pass in a non-tunneled functor or
	 *	a toplevel tunneled functor due to the fact that the visit_each mechanism
	 *	is turned off for the tunnel functor (otherwise there would be the problem
	 *	of not threadsafe access to the sigc::trackable base of the of a 
	 *	dispatchable object..
	 *	
	 *	@note passed in functor must not be a slot or adapt a slot;
	 *	we have to apply this restriction because slots might have bound
	 *	trackables that can cause non-threadsafe access to the passed in slot
	 *	which will live in the context of the server thread
	 *	
	 *	@attention	All sigc::trackableS and the original dispatchable contained 
	 *	in the passed functor must belong to the context of the calling thread.
	 *	
	 *	@return sigx::connection_wrapper A threadsafe connection wrapper
	 *	@note asynchronous
	 */
	template<typename T_functor>
	connection_wrapper connect(const T_functor& _A_func, bool after = true) const
	{
		return	signal_wrapper_base::connect(
					_A_func, 
					sigc::bind(
						sigc::ptr_fun(&typed_connection_handler<signal_type, signal_group>::connect), 
						after
					)
				);
	}

	/**	
	 *	@return sigx::connection_wrapper A threadsafe connection wrapper
	 *	@note asynchronous
	 */
	template<typename T_functor>
	connection_wrapper connect_notify(const T_functor& _A_func, bool after = false) const
	{
		return	signal_wrapper_base::connect(
					_A_func, 
					sigc::bind(
						sigc::ptr_fun(&typed_connection_handler<signal_type, signal_group>::connect_notify), 
						after
					)
				);
	}
};


/**	@short A threadsafe wrapper for a Glib::SignalIdle.
 *	@ingroup signals
 */
template<>
class signal_wrapper<Glib::SignalIdle>: public signal_wrapper_base
{
public:
	static const int argument_count = internal::count_signal_arguments<Glib::SignalIdle>::value;
	static const internal::signal_group signal_group = internal::SIGGROUP_IRRELEVANT;
	typedef Glib::SignalIdle signal_type;
	typedef signal_wrapper<signal_type, signal_group, argument_count> this_type;
	typedef bool result_type;
	typedef sigc::slot<bool> slot_type;
	typedef signal_type (*fp_sig_getter_type)(signal_source_ptr);


	/**	@short Constructs an empty signal_wrapper.
	 *	@throw	Might throw a std::bad_alloc exception (from dispatchable's ctor)
	 */
	signal_wrapper(): signal_wrapper_base()
	{}

	/**	@short Creates a signal_wrapper from a signal source.
	 *	@param _A_disp The dispatchable to operate on
	 *	@param _A_sigsource A double pointer to the server 
	 *	thread's signal source.
	 */
	signal_wrapper(const shared_dispatchable& _A_disp, const std::tr1::shared_ptr<signal_source_base>& _A_sigsource): 
		signal_wrapper_base(_A_disp, _A_sigsource)
	{}
	
	/**	@short Connects a functor, tunnels it automatically if not yet 
	 *	tunneled and activates validity tracking for sigc::trackableS.
	 *	
	 *	auto tunneling is successful only if the passed in functor is a
	 *	"dispatchable" functor, i.e. a functor on a dispatchable's method
	 *	or one explicitly created with "dispatch_with".
	 *	You will get compiler errors if the dispatchable can't be deduced from the 
	 *	passed in functor.
	 *	
	 *	@note At the moment it is only possible to pass in a non-tunneled functor or
	 *	a toplevel tunneled functor due to the fact that the visit_each mechanism
	 *	is turned off for the tunnel functor (otherwise there would be the problem
	 *	of not threadsafe access to the sigc::trackable base of the of a 
	 *	dispatchable object..
	 *	
	 *	@note passed in functor must not be a slot or adapt a slot;
	 *	we have to apply this restriction because slots might have bound
	 *	trackables that can cause non-threadsafe access to the passed in slot
	 *	which will live in the context of the server thread
	 *	
	 *	@attention	All sigc::trackableS and the original dispatchable contained 
	 *	in the passed functor must belong to the context of the calling thread.
	 *	
	 *	@return sigx::connection_wrapper A threadsafe connection wrapper
	 *	@note asynchronous
	 */
	template<typename T_functor>
	connection_wrapper connect(const T_functor& _A_func, int priority = Glib::PRIORITY_DEFAULT_IDLE) const
	{
		return	signal_wrapper_base::connect(
					_A_func, 
					sigc::bind(
						sigc::ptr_fun(&typed_connection_handler<signal_type, signal_group>::connect), 
						priority
					)
				);
	}
};


/**	@short A threadsafe wrapper for a Glib::SignalTimeout.
 *	@ingroup signals
 */
template<>
class signal_wrapper<Glib::SignalTimeout>: public signal_wrapper_base
{
public:
	static const int argument_count = internal::count_signal_arguments<Glib::SignalTimeout>::value;
	static const internal::signal_group signal_group = internal::SIGGROUP_IRRELEVANT;
	typedef Glib::SignalTimeout signal_type;
	typedef signal_wrapper<signal_type, signal_group, argument_count> this_type;
	typedef bool result_type;
	typedef sigc::slot<bool> slot_type;
	typedef signal_type (*fp_sig_getter_type)(signal_source_ptr);


	/**	@short Constructs an empty signal_wrapper.
	 *	@throw	Might throw a std::bad_alloc exception (from dispatchable's ctor)
	 */
	signal_wrapper(): signal_wrapper_base()
	{}

	/**	@short Creates a signal_wrapper from a signal source.
	 *	@param _A_disp The dispatchable to operate on
	 *	@param _A_sigsource A double pointer to the server 
	 *	thread's signal source.
	 */
	signal_wrapper(const shared_dispatchable& _A_disp, const std::tr1::shared_ptr<signal_source_base>& _A_sigsource): 
		signal_wrapper_base(_A_disp, _A_sigsource)
	{}
	
	/**	@short Connects a functor, tunnels it automatically if not yet 
	 *	tunneled and activates validity tracking for sigc::trackableS.
	 *	
	 *	auto tunneling is successful only if the passed in functor is a
	 *	"dispatchable" functor, i.e. a functor on a dispatchable's method
	 *	or one explicitly created with "dispatch_with".
	 *	You will get compiler errors if the dispatchable can't be deduced from the 
	 *	passed in functor.
	 *	
	 *	@note At the moment it is only possible to pass in a non-tunneled functor or
	 *	a toplevel tunneled functor due to the fact that the visit_each mechanism
	 *	is turned off for the tunnel functor (otherwise there would be the problem
	 *	of not threadsafe access to the sigc::trackable base of the of a 
	 *	dispatchable object..
	 *	
	 *	@note passed in functor must not be a slot or adapt a slot;
	 *	we have to apply this restriction because slots might have bound
	 *	trackables that can cause non-threadsafe access to the passed in slot
	 *	which will live in the context of the server thread
	 *	
	 *	@attention	All sigc::trackableS and the original dispatchable contained 
	 *	in the passed functor must belong to the context of the calling thread.
	 *	
	 *	@return sigx::connection_wrapper A threadsafe connection wrapper
	 *	@note asynchronous
	 */
	template<typename T_functor>
	connection_wrapper connect(const T_functor& _A_func, unsigned int interval,
						int priority = Glib::PRIORITY_DEFAULT) const
	{
		return	signal_wrapper_base::connect(
					_A_func, 
					sigc::bind(
						sigc::ptr_fun(&typed_connection_handler<signal_type, signal_group>::connect), 
						interval, priority
					)
				);
	}
};


/**	@short A threadsafe wrapper for a Glib::SignalIO.
 *	@ingroup signals
 */
template<>
class signal_wrapper<Glib::SignalIO>: public signal_wrapper_base
{
public:
	static const int argument_count = internal::count_signal_arguments<Glib::SignalIO>::value;
	static const internal::signal_group signal_group = internal::SIGGROUP_IRRELEVANT;
	typedef Glib::SignalIO signal_type;
	typedef signal_wrapper<signal_type, signal_group, argument_count> this_type;
	typedef bool result_type;
	typedef sigc::slot<bool, Glib::IOCondition> slot_type;
	typedef signal_type (*fp_sig_getter_type)(signal_source_ptr);


	/**	@short Constructs an empty signal_wrapper.
	 *	@throw	Might throw a std::bad_alloc exception (from dispatchable's ctor)
	 */
	signal_wrapper(): signal_wrapper_base()
	{}

	/**	@short Creates a signal_wrapper from a signal source.
	 *	@param _A_disp The dispatchable to operate on
	 *	@param _A_sigsource A double pointer to the source of the server 
	 *	thread's signal.
	 */
	signal_wrapper(const shared_dispatchable& _A_disp, const std::tr1::shared_ptr<signal_source_base>& _A_sigsource): 
		signal_wrapper_base(_A_disp, _A_sigsource)
	{}
	
	/**	@short Connects a functor, tunnels it automatically if not yet 
	 *	tunneled and activates validity tracking for sigc::trackableS.
	 *	
	 *	auto tunneling is successful only if the passed in functor is a
	 *	"dispatchable" functor, i.e. a functor on a dispatchable's method
	 *	or one explicitly created with "dispatch_with".
	 *	You will get compiler errors if the dispatchable can't be deduced from the 
	 *	passed in functor.
	 *	
	 *	@note At the moment it is only possible to pass in a non-tunneled functor or
	 *	a toplevel tunneled functor due to the fact that the visit_each mechanism
	 *	is turned off for the tunnel functor (otherwise there would be the problem
	 *	of not threadsafe access to the sigc::trackable base of the of a 
	 *	dispatchable object..
	 *	
	 *	@note passed in functor must not be a slot or adapt a slot;
	 *	we have to apply this restriction because slots might have bound
	 *	trackables that can cause non-threadsafe access to the passed in slot
	 *	which will live in the context of the server thread
	 *	
	 *	@attention	All sigc::trackableS and the original dispatchable contained 
	 *	in the passed functor must belong to the context of the calling thread.
	 *	
	 *	@return sigx::connection_wrapper A threadsafe connection wrapper
	 *	@note asynchronous
	 */
	template<typename T_functor>
	connection_wrapper connect(const T_functor& _A_func, Glib::IOCondition condition,
						int priority = Glib::PRIORITY_DEFAULT) const
	{
		return	signal_wrapper_base::connect(
					_A_func, 
					sigc::bind(
						sigc::ptr_fun(&typed_connection_handler<signal_type, signal_group>::connect), 
						condition, priority
					)
				);
	}
};


/**	@short A threadsafe wrapper for a Glib::SignalChildWatch.
 *	@ingroup signals
 */
template<>
class signal_wrapper<Glib::SignalChildWatch>: public signal_wrapper_base
{
public:
	static const int argument_count = internal::count_signal_arguments<Glib::SignalChildWatch>::value;
	static const internal::signal_group signal_group = internal::SIGGROUP_IRRELEVANT;
	typedef Glib::SignalChildWatch signal_type;
	typedef signal_wrapper<signal_type, signal_group, argument_count> this_type;
	typedef bool result_type;
	typedef sigc::slot<bool, GPid, int> slot_type;
	typedef signal_type (*fp_sig_getter_type)(signal_source_ptr);


	/**	@short Constructs an empty signal_wrapper.
	 *	@throw	Might throw a std::bad_alloc exception (from dispatchable's ctor)
	 */
	signal_wrapper(): signal_wrapper_base()
	{}

	/**	@short Creates a signal_wrapper from a signal source.
	 *	@param _A_disp The dispatchable to operate on
	 *	@param _A_sigsource A double pointer to the server 
	 *	thread's signal source.
	 */
	signal_wrapper(const shared_dispatchable& _A_disp, const std::tr1::shared_ptr<signal_source_base>& _A_sigsource): 
		signal_wrapper_base(_A_disp, _A_sigsource)
	{}
	
	/**	@short Connects a functor, tunnels it automatically if not yet 
	 *	tunneled and activates validity tracking for sigc::trackableS.
	 *	
	 *	auto tunneling is successful only if the passed in functor is a
	 *	"dispatchable" functor, i.e. a functor on a dispatchable's method
	 *	or one explicitly created with "dispatch_with".
	 *	You will get compiler errors if the dispatchable can't be deduced from the 
	 *	passed in functor.
	 *	
	 *	@note At the moment it is only possible to pass in a non-tunneled functor or
	 *	a toplevel tunneled functor due to the fact that the visit_each mechanism
	 *	is turned off for the tunnel functor (otherwise there would be the problem
	 *	of not threadsafe access to the sigc::trackable base of the of a 
	 *	dispatchable object..
	 *	
	 *	@note passed in functor must not be a slot or adapt a slot;
	 *	we have to apply this restriction because slots might have bound
	 *	trackables that can cause non-threadsafe access to the passed in slot
	 *	which will live in the context of the server thread
	 *	
	 *	@attention	All sigc::trackableS and the original dispatchable contained 
	 *	in the passed functor must belong to the context of the calling thread.
	 *	
	 *	@return sigx::connection_wrapper A threadsafe connection wrapper
	 *	@note asynchronous
	 */
	template<typename T_functor>
	connection_wrapper connect(const T_functor& _A_func, GPid pid,
						int priority = Glib::PRIORITY_DEFAULT) const
	{
		return	signal_wrapper_base::connect(
					_A_func, 
					sigc::bind(
						sigc::ptr_fun(&typed_connection_handler<signal_type, signal_group>::connect), 
						pid, priority
					)
				);
	}
};


typedef signal_wrapper<Glib::SignalIdle> glib_signal_idle;
typedef signal_wrapper<Glib::SignalTimeout> glib_signal_timeout;
typedef signal_wrapper<Glib::SignalIO> glib_signal_io;
typedef signal_wrapper<Glib::SignalChildWatch> glib_ignal_childwatch;


} // namespace sigx


