/**
 * UGENE - Integrated Bioinformatics Tools.
 * Copyright (C) 2008-2011 UniPro <ugene@unipro.ru>
 * http://ugene.unipro.ru
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 */

#include <U2Core/AppContext.h>
#include <U2Core/Counter.h>
#include <U2Core/Version.h>
#include <U2Core/IOAdapter.h>
#include <U2Core/ProjectModel.h>
#include <U2Core/DNATranslation.h>
#include <U2Core/BaseDocumentFormats.h>
#include <U2Core/DocumentModel.h>
#include <U2Core/AddDocumentTask.h>
#include <U2Core/LoadDocumentTask.h>
#include <U2Core/SaveDocumentTask.h>
#include <U2Core/MultiTask.h>
#include <U2Core/FormatUtils.h>
#include <U2Core/GObjectUtils.h>
#include <U2Core/GObjectRelationRoles.h>
#include <U2Core/DNASequenceObject.h>
#include <U2Gui/OpenViewTask.h>

#include "FindEnzymesTask.h"
#include "CloningUtilTasks.h"

namespace U2 {

DigestSequenceTask::DigestSequenceTask(  const DNASequenceObject* so, AnnotationTableObject* source, 
                                       AnnotationTableObject* dest, const QList<SEnzymeData>& cutSites )
                                       :   Task("DigestSequenceTask", TaskFlags_FOSCOE | TaskFlag_ReportingIsSupported | TaskFlag_ReportingIsEnabled),
                                       searchForRestrictionSites(false), sourceObj(source), destObj(dest), dnaObj(so), enzymeData(cutSites)
{
    GCOUNTER(cvar,tvar,"DigestSequenceIntoFragments");

    assert(sourceObj != NULL);
    assert(destObj != NULL);
    assert(dnaObj != NULL);


}

DigestSequenceTask::DigestSequenceTask( const DNASequenceObject* so, AnnotationTableObject* aobj, 
                                       const QList<SEnzymeData>& cutSites )
:   Task("DigestSequenceTask", TaskFlags_FOSCOE | TaskFlag_ReportingIsSupported | TaskFlag_ReportingIsEnabled),
    searchForRestrictionSites(true), sourceObj(aobj), destObj(aobj), dnaObj(so), enzymeData(cutSites)
{
    GCOUNTER(cvar,tvar,"DigestSequenceIntoFragments");
    
    assert(sourceObj != NULL);
    assert(destObj != NULL);
    assert(dnaObj != NULL);
}

void DigestSequenceTask::prepare() {
    seqRange = dnaObj->getSequenceRange();
    isCircular = dnaObj->isCircular();
    
    if (searchForRestrictionSites) {
        assert(sourceObj == destObj);
        FindEnzymesTaskConfig cfg;
        cfg.circular = isCircular;
        cfg.groupName = ANNOTATION_GROUP_ENZYME;
        Task* t = new FindEnzymesToAnnotationsTask(sourceObj, dnaObj->getDNASequence(), enzymeData, cfg);
        addSubTask(t);
    }  
        
}

AnnotationData* DigestSequenceTask::createFragment( int pos1, const DNAFragmentTerm& leftTerm, 
                                                   int pos2, const DNAFragmentTerm& rightTerm )
{
    AnnotationData* ad = new AnnotationData();
    assert(pos1 != pos2 );
    if (pos1  < pos2) {
        ad->location->regions.append(U2Region(pos1, pos2 - pos1 ));
    } else {
        ad->location->regions.append(U2Region(pos1, seqRange.endPos() - pos1 ));
        ad->location->regions.append(U2Region(seqRange.startPos, pos2 - seqRange.startPos ));
    }
    
    ad->qualifiers.append(U2Qualifier(QUALIFIER_LEFT_TERM, leftTerm.enzymeId));
    ad->qualifiers.append(U2Qualifier(QUALIFIER_RIGHT_TERM, rightTerm.enzymeId));
    
    ad->qualifiers.append(U2Qualifier(QUALIFIER_LEFT_OVERHANG, leftTerm.overhang));
    ad->qualifiers.append(U2Qualifier(QUALIFIER_RIGHT_OVERHANG, rightTerm.overhang));
    
    QString leftOverhangStrand = leftTerm.isDirect ? OVERHANG_STRAND_DIRECT : OVERHANG_STRAND_COMPL;
    ad->qualifiers.append(U2Qualifier(QUALIFIER_LEFT_STRAND, leftOverhangStrand));
    QString rightOverhangStrand = rightTerm.isDirect ? OVERHANG_STRAND_DIRECT : OVERHANG_STRAND_COMPL;
    ad->qualifiers.append(U2Qualifier(QUALIFIER_RIGHT_STRAND, rightOverhangStrand));
    
    QString leftOverhangType = leftTerm.enzymeId.isEmpty() ? OVERHANG_TYPE_BLUNT : OVERHANG_TYPE_STICKY;
    ad->qualifiers.append(U2Qualifier(QUALIFIER_LEFT_TYPE, leftOverhangType) );
    QString rightOverhangType = rightTerm.enzymeId.isEmpty() ? OVERHANG_TYPE_BLUNT : OVERHANG_TYPE_STICKY;
    ad->qualifiers.append(U2Qualifier(QUALIFIER_RIGHT_TYPE, rightOverhangType) );

    ad->qualifiers.append(U2Qualifier(QUALIFIER_SOURCE, dnaObj->getGObjectName()));
    
    return ad;
}


Task::ReportResult DigestSequenceTask::report()
{
    if (hasError() || isCanceled()) {
        return ReportResult_Finished;
    }

    saveResults();
    
    return ReportResult_Finished;
}

void DigestSequenceTask::findCutSites()
{
    foreach (const SEnzymeData& enzyme, enzymeData) {
        
        if (enzyme->cutDirect == ENZYME_CUT_UNKNOWN || enzyme->cutComplement == ENZYME_CUT_UNKNOWN) {
            setError(tr("Can't use restriction site %1 for digestion,  cleavage site is unknown ").arg(enzyme->id));
            return;
        }

        QList<Annotation*> anns;
        foreach (Annotation* a, sourceObj->getAnnotations()) {
            if (a->getAnnotationName() == enzyme->id) {
                anns.append(a);
            }
        }
                
        if (anns.isEmpty()) {
            stateInfo.setError(  QString("Restriction site %1 is not found").arg(enzyme->id) );
            continue;
        }
        
        foreach (Annotation * a, anns) {
            const QVector<U2Region>& location = a->getRegions();
            int cutPos = location.first().startPos;
            cutSiteMap.insertMulti(cutPos, enzyme);
        }

    }
}

void DigestSequenceTask::run()
{
    if (hasError() || isCanceled()) {
        return;
    }

    findCutSites();

    if (cutSiteMap.isEmpty()) {
        return;
    }

    QMap<int,SEnzymeData>::const_iterator prev = cutSiteMap.constBegin(), current = cutSiteMap.constBegin();
    int count = 2;
    const QByteArray& sourceSeq = dnaObj->getSequence();

    while ( (++current) != cutSiteMap.constEnd() )  {
        int pos1 = prev.key();
        int pos2 = current.key();
        const SEnzymeData& enzyme1 = prev.value();
        const SEnzymeData& enzyme2 = current.value();
        int len1= enzyme1->seq.length();
        int len2 = enzyme2->seq.length();

        {

            U2Region region1(pos1, len1);
            U2Region region2(pos2, len2);

            if (region1.intersects(region2)) {
                setError(tr("Unable to digest into fragments: intersecting restriction sites %1 (%2..%3) and %4 (%5..%6)")
                    .arg(enzyme1->id).arg(region1.startPos).arg(region1.endPos())
                    .arg(enzyme2->id).arg(region2.startPos).arg(region2.endPos()));
                return;
            }

        }
        
        DNAFragmentTerm leftTerm;
        int leftCutCompl = len1 - enzyme1->cutComplement;
        int leftCutPos = pos1 + qMax(enzyme1->cutDirect, leftCutCompl);
        int leftOverhangStart = pos1 + qMin(enzyme1->cutDirect, leftCutCompl);
        leftTerm.overhang = sourceSeq.mid(leftOverhangStart, leftCutPos - leftOverhangStart);
        leftTerm.enzymeId = enzyme1->id.toAscii();
        leftTerm.isDirect = enzyme1->cutDirect < leftCutCompl; 

        DNAFragmentTerm rightTerm;
        int rightCutCompl = len2 - enzyme2->cutComplement;
        int rightCutPos = pos2 + qMin(enzyme2->cutDirect, rightCutCompl );
        int rightOverhangStart = pos2 + qMax(enzyme2->cutDirect, rightCutCompl );
        rightTerm.overhang = sourceSeq.mid(rightCutPos, rightOverhangStart - rightCutPos);
        rightTerm.enzymeId = enzyme2->id.toAscii();
        rightTerm.isDirect = enzyme2->cutDirect > rightCutCompl;

        AnnotationData* ad = createFragment(leftCutPos, leftTerm,  rightCutPos, rightTerm);
        ad->name = QString("Fragment %1").arg(count);
        results.append(SharedAnnotationData(ad));
        ++count;
        ++prev;

    } 

    QMap<int,SEnzymeData>::const_iterator first = cutSiteMap.constBegin();

    const SEnzymeData& firstCutter = first.value();
    int fcLen = firstCutter->seq.length();
    int firstCutPos = first.key() + qMin(firstCutter->cutDirect, fcLen - firstCutter->cutComplement);
    int rightOverhangStart = first.key() + qMax(firstCutter->cutDirect, fcLen - firstCutter->cutComplement);
    bool rightOverhangIsDirect = firstCutter->cutDirect > fcLen - firstCutter->cutComplement;
    QByteArray firstRightOverhang = sourceSeq.mid(firstCutPos, rightOverhangStart - firstCutPos);
    
    const SEnzymeData& lastCutter = prev.value();
    int lcLen = lastCutter->seq.length();
    int lastCutPos = prev.key() + qMax(lastCutter->cutDirect, lcLen - lastCutter->cutComplement);
    int leftOverhangStart = prev.key() + qMin(lastCutter->cutDirect, fcLen - lastCutter->cutComplement);
    bool leftOverhangIsDirect = lastCutter->cutDirect < lcLen - lastCutter->cutComplement;
    
    if (lastCutPos >= sourceSeq.length()) {
        // last restriction site is situated between sequence start and end
        assert(isCircular);
        int leftCutPos = lastCutPos - sourceSeq.length();
        QByteArray leftOverhang = sourceSeq.mid(leftOverhangStart) + sourceSeq.mid(0, leftCutPos);
        QByteArray rightOverhang = first == prev ? leftOverhang : firstRightOverhang;
        AnnotationData* ad1 = createFragment(leftCutPos, DNAFragmentTerm(lastCutter->id, leftOverhang, leftOverhangIsDirect), 
            firstCutPos, DNAFragmentTerm(firstCutter->id, rightOverhang, rightOverhangIsDirect) );
        ad1->name = QString("Fragment 1");
        results.append(SharedAnnotationData(ad1));
    } else {
        QByteArray lastLeftOverhang = sourceSeq.mid(leftOverhangStart, lastCutPos - leftOverhangStart);
        if (isCircular) {
            AnnotationData* ad = createFragment(lastCutPos, DNAFragmentTerm(lastCutter->id, lastLeftOverhang, leftOverhangIsDirect), 
                firstCutPos, DNAFragmentTerm(firstCutter->id, firstRightOverhang,rightOverhangIsDirect) );
            ad->name = QString("Fragment 1");
            results.append(SharedAnnotationData(ad));
        } else {
            AnnotationData* ad1 = createFragment(seqRange.startPos, DNAFragmentTerm(), 
                firstCutPos, DNAFragmentTerm(firstCutter->id, firstRightOverhang, rightOverhangIsDirect) );
            AnnotationData* ad2 = createFragment(lastCutPos, DNAFragmentTerm(lastCutter->id, lastLeftOverhang, leftOverhangIsDirect), 
                seqRange.endPos(), DNAFragmentTerm() );
            ad1->name = QString("Fragment 1");
            ad2->name = QString("Fragment %1").arg(count);
            results.append(SharedAnnotationData(ad1));
            results.append(SharedAnnotationData(ad2));
        }
    }

}


void DigestSequenceTask::saveResults()
{
    foreach (const SharedAnnotationData& data, results) {
        destObj->addAnnotation(new Annotation(data), ANNOTATION_GROUP_FRAGMENTS);
    }
}

QString DigestSequenceTask::generateReport() const
{
    QString res;
    QString topology = dnaObj->isCircular() ? tr("circular") : tr("linear");
    res+= tr("<h3><br>Digest into fragments %1 (%2)</h3>").arg(dnaObj->getDocument()->getName()).arg(topology);
    res+=tr("<br>Generated %1 fragments.").arg(results.count());
    int counter = 1;
    foreach (const SharedAnnotationData& sdata, results) {
        int startPos = sdata->location->regions.first().startPos;
        int endPos = sdata->location->regions.first().endPos();
        res+=tr("<br><br>&nbsp;&nbsp;&nbsp;&nbsp;%1:&nbsp;&nbsp;&nbsp;&nbsp;From %3 (%2) To %5 (%4) - %6 bp ").arg(counter)
                  .arg(startPos).arg(sdata->findFirstQualifierValue(QUALIFIER_LEFT_TERM))
                  .arg(endPos).arg(sdata->findFirstQualifierValue(QUALIFIER_RIGHT_TERM))
                   .arg(endPos - startPos + 1);   
        ++counter;
    }

    return res;

}


//////////////////////////////////////////////////////////////////////////

LigateFragmentsTask::LigateFragmentsTask( const QList<DNAFragment>& fragments, const LigateFragmentsTaskConfig& config )
: Task("LigateFragmentsTask", TaskFlags_NR_FOSCOE), fragmentList(fragments), cfg(config), 
  resultDoc(NULL), resultAlphabet(NULL)
{
    GCOUNTER(cvar,tvar,"LigateFragments");
}

void LigateFragmentsTask::processOverhangs( const DNAFragment& prevFragment, const DNAFragment& curFragment, QByteArray& overhangAddition )
{
    const DNAFragmentTerm& prevTerm = prevFragment.getRightTerminus();
    const DNAFragmentTerm& curTerm = curFragment.getLeftTerminus();

    if (prevTerm.type != curTerm.type) {
        stateInfo.setError( tr("Fragments %1 and  %2 are inconsistent. Blunt and sticky ends incompatibility")
            .arg(prevFragment.getName()).arg(curFragment.getName()) );
        return;
    }
    
    QByteArray prevOverhang = prevFragment.getRightTerminus().overhang;
    QByteArray curOverhang = curFragment.getLeftTerminus().overhang;

    if (prevTerm.type == OVERHANG_TYPE_STICKY) {
        if (!overhangsAreConsistent(prevFragment.getRightTerminus(), curFragment.getLeftTerminus())) {
            stateInfo.setError( tr("Right overhang from %1 and left overhang from %2 are inconsistent.")
                .arg(prevFragment.getName()).arg(curFragment.getName()) );
            return;
        } else {
            overhangAddition += curOverhang;
        }
    } else if (prevTerm.type == OVERHANG_TYPE_BLUNT) {
        overhangAddition += prevOverhang + curOverhang;
    } else {
        assert(0);
    }

}


bool LigateFragmentsTask::overhangsAreConsistent( const DNAFragmentTerm& curTerm, const DNAFragmentTerm& prevTerm )
{
    QByteArray curOverhang = curTerm.overhang;
    QByteArray prevOverhang = prevTerm.overhang;
    
    bool curStrand = curTerm.isDirect;
    bool prevStrand = prevTerm.isDirect;
    if (curStrand == prevStrand) {
        return false;
    }

    return curOverhang == prevOverhang;
}

void LigateFragmentsTask::prepare()
{
    QByteArray resultSeq;
    QVector<U2Region> fragmentRegions;

    DNAFragment prevFragment;
    assert(prevFragment.isEmpty());

    foreach (const DNAFragment& dnaFragment, fragmentList) {

        QVector<U2Region> location = dnaFragment.getFragmentRegions();
        assert(location.size() > 0);
        
        // check alphabet consistency
        DNAAlphabet* fragmentAlphabet = dnaFragment.getAlphabet();
        if (resultAlphabet == NULL ) {
            resultAlphabet = fragmentAlphabet;
        } else if (resultAlphabet != fragmentAlphabet) {
            if (fragmentAlphabet == NULL) {
                stateInfo.setError( tr("Unknown DNA alphabet in fragment %1 of %2")
                    .arg(dnaFragment.getName()).arg(dnaFragment.getSequenceName()) );
                return;
            }
            resultAlphabet = DNAAlphabet::deriveCommonAlphabet(resultAlphabet,fragmentAlphabet);
        }
        
        // check if overhangs are compatible
        QByteArray overhangAddition;
        if (cfg.checkOverhangs ) {
             if (!prevFragment.isEmpty()) {
                processOverhangs(prevFragment, dnaFragment, overhangAddition );
                if (stateInfo.hasError()) {
                    return;
                }
             }
             prevFragment = dnaFragment;
        }
        
        // handle fragment annotations
        int resultLen = resultSeq.length() + overhangAddition.length();
        foreach (AnnotationTableObject* aObj, dnaFragment.getRelatedAnnotations()) {
            QList<Annotation*> toSave = cloneAnnotationsInFragmentRegion(dnaFragment, aObj, resultLen);
            annotations.append(toSave);
        }
        
        if (cfg.annotateFragments) {
            Annotation* a = createFragmentAnnotation(dnaFragment, resultLen);
            annotations.append(a);
        }

        resultSeq.append(overhangAddition);
        resultSeq.append(dnaFragment.getSequence());
    }

    if (cfg.makeCircular && cfg.checkOverhangs) {
        const DNAFragment& first = fragmentList.first();
        const DNAFragment& last = fragmentList.last();
        QByteArray overhangAddition;
        processOverhangs(last, first, overhangAddition);
        if (stateInfo.hasError()) {
            return;
        }
        resultSeq.append(overhangAddition);
    }

    // create comment
    Annotation* sourceAnnot = createSourceAnnotation(resultSeq.length());
    annotations.append(sourceAnnot);

    createDocument(resultSeq,annotations);
    
    if (!cfg.addDocToProject) {
        return;
    }
    
    QList<Task*> tasks;
    tasks.append(new AddDocumentTask(resultDoc));

    if (cfg.openView) {
        tasks.append(new OpenViewTask(resultDoc));
    }
    if (cfg.saveDoc) {
        tasks.append(new SaveDocumentTask(resultDoc));
    }

    Task* multiTask = new MultiTask(tr("Add constructed molecule"), tasks );
    addSubTask(multiTask);
    

}


Annotation* LigateFragmentsTask::createSourceAnnotation( int regLen )
{
    Version v = Version::ugeneVersion();
    SharedAnnotationData sd( new AnnotationData);
    sd->name = "source";
    sd->location->regions << U2Region(0, regLen);
    sd->qualifiers.append( U2Qualifier("comment", QString("Molecule is created with Unipro UGENE v%1.%2").arg(v.major).arg(v.minor)) );
    return  new Annotation(sd);

}

Annotation* LigateFragmentsTask::createFragmentAnnotation( const DNAFragment& fragment, int startPos )
{
    SharedAnnotationData sd( new AnnotationData);
    sd->name = QString("%1 %2").arg(fragment.getSequenceName()).arg(fragment.getName());
    sd->location->regions << U2Region(startPos, fragment.getLength());
    sd->qualifiers.append(U2Qualifier("source_doc", fragment.getSequenceDocName()));

    return  new Annotation(sd);
}

QList<Annotation*> LigateFragmentsTask::cloneAnnotationsInRegion( const U2Region& fragmentRegion, AnnotationTableObject* source, int globalOffset )
{
    QList<Annotation*> results;
    
    // TODO: allow to cut annotations
    
    foreach(Annotation* a, source->getAnnotations()) {
        bool ok = true;
        const QVector<U2Region>& location = a->getRegions();
        foreach(const U2Region& region, location) {
            if (!fragmentRegion.contains(region) || fragmentRegion == region) {
                ok = false;
                break;
            }
        }
        if (ok) {
            int newPos = globalOffset + location.first().startPos - fragmentRegion.startPos;
            Annotation* cloned = new Annotation(a->data());
            QVector<U2Region> newLocation;
            foreach (const U2Region& region, a->getRegions()) {
                U2Region newRegion(region);
                newRegion.startPos = newPos;
                newLocation.append(newRegion);
            }
            cloned->replaceRegions(newLocation);
            results.append(cloned);
        }

    }

    return results;

}

static bool fragmentContainsRegion(const DNAFragment& fragment, const U2Region region) {
    
    QVector<U2Region> fragmentRegions = fragment.getFragmentRegions();

    bool result = false;
    foreach (const U2Region& fR, fragmentRegions) {
        if (fR.contains(region)) {
            result = true;
            break;
        }

    }
    
    return result;

} 


static int getRelativeStartPos(const DNAFragment& fragment, const U2Region region)
{
    
    QVector<U2Region> fragmentRegions = fragment.getFragmentRegions();

    int offset = 0;
    foreach (const U2Region& fR, fragmentRegions) {
        if (fR.contains(region)) {
            return offset + region.startPos - fR.startPos;
        }
        offset += fR.length;
    }
    
    // the fragment doesn't contain the region
    return -1;
}



QList<Annotation*> LigateFragmentsTask::cloneAnnotationsInFragmentRegion( const DNAFragment& fragment, AnnotationTableObject* source, int globalOffset )
{
    QList<Annotation*> results;

    // TODO: allow to remove annotations

    foreach(Annotation* a, source->getAnnotations()) {
        QVector<U2Region> location = a->getRegions();
        
        if (a->getAnnotationName().startsWith("Fragment")) {
            continue;
        }

        bool ok = true;
        foreach (const U2Region& r, location) {
            // sneaky annotations shall not pass!
            if (!fragmentContainsRegion(fragment, r)) {
                ok = false;
                break;
            }
        }

        if (ok) {
            Annotation* cloned = new Annotation(a->data());
            QVector<U2Region> newLocation;
            foreach (const U2Region& region, location) {
                int startPos = getRelativeStartPos(fragment, region);
                if (fragment.isInverted()) {
                    startPos = fragment.getLength() - startPos - region.length;
                    U2Strand strand = cloned->getStrand();
                    if (strand.isDirect()) {
                        cloned->setStrand(U2Strand::Complementary);
                    } else {
                        cloned->setStrand(U2Strand::Direct);
                    } 
                }
                assert(startPos != -1);
                int newPos = globalOffset + startPos;
                U2Region newRegion(region);
                newRegion.startPos = newPos;
                newLocation.append(newRegion);
            }
            
            cloned->replaceRegions(newLocation);
            results.append(cloned);
        }

    }

    return results;
}


void LigateFragmentsTask::createDocument( const QByteArray& seq, const QList<Annotation*> annotations )
{
    
    DocumentFormat* df = AppContext::getDocumentFormatRegistry()->getFormatById(BaseDocumentFormats::PLAIN_GENBANK);
    IOAdapterFactory * iof = AppContext::getIOAdapterRegistry()->getIOAdapterFactoryById(BaseIOAdapters::LOCAL_FILE);
    QList<GObject*> objects;

    QString seqName = cfg.seqName.isEmpty() ? cfg.docUrl.baseFileName() : cfg.seqName;
    DNASequence dna(seqName, seq, resultAlphabet);
    dna.circular = cfg.makeCircular;
    
    // set Genbank header
    DNALocusInfo loi;
    loi.name = seqName;
    loi.topology = cfg.makeCircular ? "circular" : "linear";
    loi.molecule = "DNA";
    loi.division = "SYN";
    QDate date = QDate::currentDate();
    loi.date = QString("%1-%2-%3").arg(date.toString("dd"))
        .arg( FormatUtils::getShortMonthName(date.month()) )
        .arg( date.toString("yyyy"));
                    
    dna.info.insert(DNAInfo::LOCUS, qVariantFromValue<DNALocusInfo>(loi));

    DNASequenceObject* dnaObj = new DNASequenceObject(seqName, dna);
    objects.append(dnaObj);
   
    AnnotationTableObject* aObj = new AnnotationTableObject(QString("%1 annotations").arg(seqName));
    aObj->addAnnotations(annotations);
    objects.append(aObj);
    
    resultDoc = new Document(df, iof, cfg.docUrl, objects);
    resultDoc->setModified(true);
    aObj->addObjectRelation(dnaObj,GObjectRelationRole::SEQUENCE);
    

    
}

void LigateFragmentsTask::cleanup()
{
    if (stateInfo.hasError()) {
        qDeleteAll(annotations);    
    }
}







} // U2

